/*
 * @(#)InventoryException.java 1.0 2018-08-30
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.inventory.api;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.inventory.api.dto.request.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

/**
 * 仓库Api接口
 *
 * @author 佐助
 * @since 2.0.0
 */
@Api(tags = {"库存中心：仓库管理服务"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/warehouse")
public interface IWarehouseApi {

    /**
     * 新建仓库
     *
     * @param warehouseCreateReqDto 仓库新建Dto
     * @return Long 返回统一响应对象，业务数据为属性名ID
     */
    @PostMapping(value = "", produces = "application/json")
    @ApiOperation(value = "新建仓库", notes = "新建仓库 \t\n warehouseCreateReqDto:仓库新建Dto")
    RestResponse<Long> addWarehouse(@Valid @RequestBody WarehouseCreateReqDto warehouseCreateReqDto);

    /**
     * 修改仓库信息
     *
     * @param id                    仓库ID
     * @param warehouseUpdateReqDto 仓库信息修改Dto
     * @return 返回统一响应对象
     */
    @PutMapping(value = "/{id}", produces = "application/json")
    @ApiOperation(value = "修改仓库信息", notes = "修改仓库信息 \t\n id:仓库ID\twarehouseUpdateReqDto:仓库信息修改Dto")
    RestResponse<Void> modifyWarehouse(@NotNull(message = "仓库id不能为空") @PathVariable("id") Long id,
                                       @Valid @RequestBody WarehouseUpdateReqDto warehouseUpdateReqDto);

    /**
     * 根据code删除仓库信息
     *
     * @param code   仓库编码
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code删除仓库信息", notes = "根据code删除仓库信息 \t\n code:仓库编码code" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removeWarehouseByCode(@NotNull(message = "仓库code不能为空") @PathVariable("code") String code,
                                             @RequestParam("filter") String filter);

    /**
     * 根据id删除仓库信息
     *
     * @param id
     * @return
     */
    @DeleteMapping(value = "/id/{id}", produces = "application/json")
    @ApiOperation(value = "根据id删除仓库", notes = "根据id删除仓库")
    RestResponse<Void> removeWarehouseById(@NotNull(message = "仓库id不能为空") @PathVariable("id") Long id);


    /**
     * 根据服务区Id删除仓库信息
     *
     * @param serviceAreaId 服务区Id
     * @param filter        {"instanceId":0,"tenantId":1}
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/service-area-id/{serviceAreaId}", produces = "application/json")
    @ApiOperation(value = "根据服务区Id删除仓库信息", notes = "根据服务区Id删除仓库信息 \t\n serviceAreaId:服务区Id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removeWarehouseByServiceAreaId(@NotNull(message = "服务区ID不能为空") @PathVariable("serviceAreaId") Long serviceAreaId,
                                                      @RequestParam("filter") String filter);

    /**
     * 根据id列表删除仓库信息
     *
     * @param ids    仓库id列表
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 返回标识
     */
    @DeleteMapping(value = "/{ids}", produces = "application/json")
    @ApiOperation(value = "批量删除仓库信息", notes = "批量删除仓库信息 \t\n ids:仓库id列表" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removeWarehouseByIds(@NotNull(message = "仓库ID列表不能为空") @PathVariable("ids") String ids,
                                            @RequestParam("filter") String filter);

    /**
     * 仓库和服务区关联
     *
     * @param rAreaWarehouseReqDto 仓库和服务区关联请求Dto
     * @return Long 属性名ID
     */
    @PostMapping(value = "/relate-area-warehouse", produces = "application/json")
    @ApiOperation(value = "仓库和服务区关联", notes = "仓库和服务区关联 \t\n rAreaWarehouseDto:仓库和服务区关联Dto")
    RestResponse<Long> relateToServiceArea(@Valid @RequestBody RAreaWarehouseReqDto rAreaWarehouseReqDto);

    /**
     * 删除仓库和服务区关联关系
     *
     * @param ids    仓库和服务区关联id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 返回标识
     */
    @DeleteMapping(value = "/relate-area-warehouse/{ids}", produces = "application/json")
    @ApiOperation(value = "删除仓库和服务区关联关系", notes = "删除仓库和服务区关联关系 \t\n ids:仓库和服务区关联id列表" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removeRelation(@NotNull(message = "仓库ID列表不能为空") @PathVariable("ids") String ids,
                                      @RequestParam("filter") String filter);

    /**
     * 新建仓位
     *
     * @param positionCreateReqDto 新建仓位Dto
     * @return Long 返回统一响应对象，业务数据为属性名ID
     */
    @PostMapping(value = "/position", produces = "application/json")
    @ApiOperation(value = "新建仓库位", notes = "新建仓位 \t\n positionCreateReqDto:新建仓位Dto")
    RestResponse<Long> addPosition(@Valid @RequestBody PositionCreateReqDto positionCreateReqDto);

    /**
     * 修改仓位信息
     *
     * @param id                   仓库ID
     * @param positionUpdateReqDto 修改仓位Dto
     * @return 返回统一响应对象
     */
    @PutMapping(value = "/position/{id}", produces = "application/json")
    @ApiOperation(value = "修改仓位信息", notes = "修改仓位信息 \t\n id:仓库ID\tpositionUpdateReqDto:修改仓位Dto")
    RestResponse<Void> modifyPosition(@NotNull(message = "仓位ID不能为空") @PathVariable("id") Long id,
                                      @Valid @RequestBody PositionUpdateReqDto positionUpdateReqDto);

    /**
     * 根据id删除仓位信息
     *
     * @param id     仓位id
     * @param filter 过滤条件
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/position/{id}", produces = "application/json")
    @ApiOperation(value = "根据id删除仓位信息", notes = "根据id删除仓位信息 \t\n id:仓位id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removePositionById(@NotNull(message = "仓位ID不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 根据code删除仓位信息
     *
     * @param code   仓位编码
     * @param filter 过滤条件
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/position/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code删除仓位信息", notes = "根据code删除仓位信息 \t\n code:仓位编码code" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removePositionByCode(@NotNull(message = "仓位code不能为空") @PathVariable("code") String code, @RequestParam("filter") String filter);

    /**
     * 新增服务区
     *
     * @param serviceAreaCreateDto 服务区创建dto
     * @return 操作结果
     */
    @PostMapping(value = "service-area", produces = "application/json")
    @ApiOperation(value = "新增服务区", notes = "新增服务区")
    RestResponse<Void> addServiceArea(@Valid @RequestBody ServiceAreaCreateDto serviceAreaCreateDto);

    /**
     * 根据服务区id修改服务区信息
     *
     * @param id                   服务区id
     * @param serviceAreaUpdateDto 服务区修改dto
     * @return 返回统一响应对象
     */
    @PutMapping(value = "/service-area/{id}")
    @ApiOperation(value = "根据服务区id修改服务区信息", notes = "根据服务区id修改服务区信息")
    RestResponse<Void> modifyServiceArea(@NotNull(message = "服务区id不能为空") @PathVariable("id") Long id,
                                         @Valid @RequestBody ServiceAreaUpdateDto serviceAreaUpdateDto);

    /**
     * 根据id删除服务区
     *
     * @param id     服务区主键
     * @param filter 过滤条件
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/service-area/{id}")
    @ApiOperation(value = "删除服务区", notes = "根据服务区主键删除服务区 id:服务区id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<Void> removeServiceAreaById(@NotNull(message = "服务区id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 切换仓库的库存的共享状态
     *
     * @param id
     * @param status
     * @return
     */
    @ApiOperation(value = "切换仓库的库存共享状态")
    @PutMapping(value = "/{id}/share")
    RestResponse<Void> switchInventoryShareStatus(@PathVariable("id") Long id, @RequestParam("status") Integer status);

    /**
     * 增减信用分
     *
     * @param warehouseCode 仓库编码
     * @param changeValue   增减的分值
     * @return
     */
    @ApiOperation(value = "增减仓库信用分")
    @PutMapping(value = "/addCreditValue")
    Integer addCreditValue(@RequestParam("warehouseCode") String warehouseCode, @RequestParam("changeValue") Integer changeValue);

    /**
     * 锁定仓库
     *
     * @return
     */
    @ApiOperation(value = "锁定仓库")
    @PostMapping(value = "/lock")
    RestResponse<Void> lockWarehouse(@RequestBody  WarehouseLockReqDto warehouseLockReqDto);

    /**
     * 解锁仓库
     *
     * @return
     */
    @ApiOperation(value = "解锁仓库")
    @PostMapping(value = "/unlock")
    RestResponse<Void> unlockWarehouse(@RequestBody WarehouseUnLockReqDto warehouseUnLockReqDto);
}