/*
 * ICargoQueryApi.class 1.0 2018-08-31
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.inventory.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.inventory.api.dto.response.CargoRespDto;
import com.dtyunxi.yundt.cube.center.inventory.api.fallback.AsynShareTaskFallBack;
import com.dtyunxi.yundt.cube.center.inventory.api.fallback.CargoFallBack;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 货品查询
 *
 * @author 纳兰
 * @since 2.0.0
 */
@Api(tags = {"库存中心：货品查询服务"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/cargo",fallback = CargoFallBack.class)
public interface ICargoQueryApi {
    /**
     * 根据货品id查询货品详情
     *
     * @param id     货品id
     * @param filter 过滤条件，未启用
     * @return CargoRespDto 货品id对应的货品详情
     */
    @GetMapping("/{id}")
    @ApiOperation(value = "根据货品id查询货品详情", notes = "filter，未启用")
    RestResponse<CargoRespDto> queryById(@NotNull(message = "货品id不能为空") @PathVariable("id") Long id, @RequestParam(name = "filter", required = false) String filter);

    /**
     * 根据货品code查询货品详情
     *
     * @param code   货品code
     * @param filter 过滤条件，未启用
     * @return CargoRespDto 货品code对应的货品详情
     */
    @GetMapping("/code/{code}")
    @ApiOperation(value = "根据货品code查询货品详情", notes = "filter，未启用")
    RestResponse<CargoRespDto> queryByCode(@NotNull(message = "货品code不能为空") @PathVariable("code") String code, @RequestParam(name = "filter", required = false) String filter);

    /**
     * 批量查询货品信息
     *
     * @param cargoCodes 货品编号集合
     * @return
     */
    @PostMapping("/listByCodes")
    @ApiOperation(value = "批量查询货品信息", notes = "批量查询货品信息")
    RestResponse<List<CargoRespDto>> listByCodes(@RequestBody List<String> cargoCodes);

    /**
     * 获取货品分页列表
     *
     * @param filter   filter为CargoQueryReqDto对应的JSON字符串
     * @param pageNum  指定页码
     * @param pageSize 指定每页返回的对象个数
     * @return CargoRespDto分页信息 返回统一响应对象，业务数据data为货品对象分页列表
     */
    @GetMapping(value = "/page")
    @ApiOperation(value = "获取货品分页列表", notes = "filter为CargoQueryReqDto对应的JSON字符串 \t\n " +
            "filter:货品查询条件封装对象JSON {\n" +
            "    \"groupId\": 0,\n" +
            "    \"unit\": \"string\",\n" +
            "    \"checkConfigId\": 0,\n" +
            "    \"packageSplitId\": 0,\n" +
            "    \"instanceId\": 0,\n" +
            "    \"tenantId\": 0\n" +
            "} \t\n pageNum:指定页码 \t\n pageSize:指定每页返回的对象个数")
    RestResponse<PageInfo<CargoRespDto>> queryByPage(@RequestParam("filter") String filter,
                                                     @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                     @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 获取货品列表
     *
     * @param filter CargoQueryReqDto对应的JSON字符串
     * @return CargoRespDto集合 返回统一响应对象，业务数据dta为货品对象列表
     */
    @GetMapping(value = "/list", produces = "application/json")
    @ApiOperation(value = "获取货品列表", notes = "filter为CargoQueryReqDto对应的JSON字符串 \t\n filter:货品查询条件封装对象JSON" +
            "{\n" +
            "    \"groupId\": 0,\n" +
            "    \"unit\": \"string\",\n" +
            "    \"checkConfigId\": 0,\n" +
            "    \"packageSplitId\": 0,\n" +
            "    \"instanceId\": 0,\n" +
            "    \"tenantId\": 0\n" +
            "}")
    RestResponse<List<CargoRespDto>> queryByList(@RequestParam("filter") String filter);
}
