/*
 * @(#)InventoryException.java 1.0 2018-08-30
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.inventory.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.inventory.api.dto.request.WarehouseCoordinateQueryReqDto;
import com.dtyunxi.yundt.cube.center.inventory.api.dto.request.WarehouseQueryReqDto;
import com.dtyunxi.yundt.cube.center.inventory.api.dto.response.*;
import com.dtyunxi.yundt.cube.center.inventory.api.fallback.WarehouseFallBack;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.Max;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 仓库查询Api接口
 *
 * @author 佐助
 * @since 2.0.0
 */
@Api(tags = {"库存中心：仓库查询服务"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/warehouse",fallback = WarehouseFallBack.class )
public interface IWarehouseQueryApi {

    /**
     * 根据Id查询仓库详情
     *
     * @param id     仓库id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return WarehouseRespDto 返回统一响应对象，业务数据data为仓库Dto
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "根据Id查询仓库详情", notes = "根据Id查询仓库详情 \t\n id:仓库id " +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "仓库id", dataType = "Long", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<WarehouseRespDto> queryWarehouseById(@NotNull(message = "仓库Id不能为空") @PathVariable("id") Long id,
                                                      @RequestParam(name = "filter", required = false) String filter);

    @GetMapping(value = "/base/{id}")
    @ApiOperation(value = "根据Id查询仓库基础信息，不包含配送信息", notes = "根据Id查询仓库详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "仓库id", dataType = "Long", paramType = "path", required = true)})
    RestResponse<WarehouseRespDto> queryWarehouseBaseById(@NotNull(message = "仓库Id不能为空") @PathVariable("id") Long id);

    /**
     * 根据code查询仓库详情
     *
     * @param code   仓库编码code
     * @param filter {"instanceId":0,"tenantId":1}
     * @return WarehouseRespDto 返回统一响应对象，业务数据data为仓库Dto
     */
    @GetMapping(value = "/code/{code}")
    @ApiOperation(value = "根据code查询仓库详情", notes = "根据code查询仓库详情 \t\n code:仓库code" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "仓库编码code", dataType = "String", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<WarehouseRespDto> queryWarehouseByCode(@NotBlank(message = "仓库code不能为空") @PathVariable("code") String code,
                                                        @RequestParam(name = "filter", required = false) String filter);

    /**
     * 根据查询条件查询仓库分页列表
     *
     * @param filter   查询条件
     * @param pageNum  指定页码
     * @param pageSize 页大小
     * @return WarehouseRespDto分页信息 返回统一响应对象，业务数据data为仓库对象分页列表
     */
    @GetMapping(value = "/page", produces = "application/json")
    @ApiOperation(value = "根据查询条件查询仓库分页列表", notes = "根据查询条件查询仓库分页列表 " +
            "\t\n filter:WarehouseQueryReqDto \t\n " +
            "{\n" +
            "  \"id\":0,\n" +
            "  \"code\": \"string\",\n" +
            "  \"name\": \"string\",\n" +
            "  \"status\": \"string\",\n" +
            "  \"type\": \"string\"\n" +
            "  \"addr\": \"string\",\n" +
            "  \"areaCode\": \"string\",\n" +
            "  \"cityCode\": \"string\",\n" +
            "  \"provCode\": \"string\",\n" +
            "  \"groupId\": 0,\n" +
            "  \"instanceId\": 0,\n" +
            "  \"tenantId\": 0,\n" +
            "}")
    RestResponse<PageInfo<WarehouseRespDto>> queryWarehouseByPage(@RequestParam(name = "filter", required = false) String filter,
                                                                  @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                  @Max(value = 100, message = "最大值不能超过100")
                                                                  @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据分组id查询仓库列表
     *
     * @param groupId 分组id
     * @param filter  {"instanceId":0,"tenantId":1}
     * @return WarehouseRespDto集合 返回统一响应对象，业务数据dta为仓库对象列表
     */
    @GetMapping(value = "/group-id/{groupId}/list", produces = "application/json")
    @ApiOperation(value = "根据分组id查询仓库列表", notes = "根据分组id查询仓库列表" +
            " \t\n groupId:分组id \t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "groupId", value = "分组id", dataType = "Long", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<List<WarehouseRespDto>> queryWarehouseByGroupId(@NotNull(message = "分组id不能为空") @PathVariable("groupId") Long groupId,
                                                                 @RequestParam(name = "filter") String filter);

    /**
     * 根据仓位id查询仓位信息
     *
     * @param id     仓库id
     * @param filter 过滤条件
     * @return PositionQueryRespDto 仓位基本信息
     */
    @GetMapping("/position/{id}")
    @ApiOperation(value = "根据仓位id查询仓位信息", notes = "根据仓位id查询仓位信息 id:仓位id" +
            " \t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "仓库id", dataType = "Long", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<PositionQueryRespDto> queryPositionById(@NotNull(message = "仓位id不能为空") @PathVariable("id") Long id, @RequestParam(name = "filter") String filter);

    /**
     * 根据仓位编码查询仓位信息
     *
     * @param code   仓位code
     * @param filter 过滤条件
     * @return PositionQueryRespDto 仓位基本信息
     */
    @GetMapping("/position/code/{code}")
    @ApiOperation(value = "根据仓位编码查询仓位信息", notes = "根据仓位编码查询仓位信息 code:仓位编码" +
            " \t\n groupId:分组id \t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "仓库编码code", dataType = "String", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<PositionQueryRespDto> queryPositionByCode(@NotNull(message = "仓位编码不能为空") @PathVariable("code") String code, @RequestParam(name = "filter") String filter);

    /**
     * 根据查询条件查询仓位列表
     *
     * @param filter 仓库查询条件JSON对象
     * @return PositionQueryRespDto集合 仓库列表
     */
    @GetMapping(value = "/position/list")
    @ApiOperation(value = "根据查询条件查询仓位列表", notes = "根据查询条件查询仓位列表" +
            "{\n" +
            "    \"code\": 0,\n" +
            "    \"name\": \"string\",\n" +
            "    \"type\": 0,\n" +
            "    \"instanceId\": 0,\n" +
            "    \"tenantId\": 0\n" +
            "}")
    RestResponse<List<PositionQueryRespDto>> queryPositionByList(@RequestParam("filter") String filter);

    /**
     * 根据查询条件查询符合条件的仓位分页列表
     *
     * @param filter   仓库查询条件JSON对象
     * @param pageNum  指定页码
     * @param pageSize 页大小
     * @return PositionQueryRespDto分页信息 拆包配置分页列表
     */
    @GetMapping(value = "/position/page")
    @ApiOperation(value = "根据查询条件查询符合条件的仓位分页列表",
            notes = "根据查询条件查询符合条件的仓位分页列表 \t\n filter:仓位查询条件 " +
                    "{\n" +
                    "    \"code\": 0,\n" +
                    "    \"name\": \"string\",\n" +
                    "    \"warehouseId\": \"0\",\n" +
                    "    \"type\": 0,\n" +
                    "    \"instanceId\": 0,\n" +
                    "    \"tenantId\": 0\n" +
                    "}\t\n pageNum:指定页码 \t\n pageSize:页大小")
    RestResponse<PageInfo<PositionQueryRespDto>> queryPositionByPage(@RequestParam("filter") String filter,
                                                                     @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                     @Max(value = 100, message = "最大值不能超过100")
                                                                     @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据服务区ID查询
     *
     * @param id     主键
     * @param filter 过滤条件
     * @return ServiceAreaRespDto
     */
    @GetMapping(value = "/service-area/{id}")
    @ApiOperation(value = "根据服务区ID查询", notes = "根据服务区ID查询 id:服务区id" +
            " \t\n groupId:分组id \t\n filter {\"instanceId\":0,\"tenantId\":1}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "服务区id", dataType = "Long", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "{\"instanceId\":0,\"tenantId\":1}", dataType = "String", paramType = "query")})
    RestResponse<ServiceAreaRespDto> queryServiceAreaById(@NotNull(message = "服务区id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 获取服务区分页列表
     *
     * @param filter   服务区查询条件封装对象JSON
     * @param pageNum  指定页码
     * @param pageSize 指定每页返回的对象个数
     * @return ServiceAreaRespDto分页信息 返回统一响应对象，业务数据data为货品对象分页列表
     */
    @GetMapping(value = "/service-area/page")
    @ApiOperation(value = "获取服务区分页列表", notes = "获取服务区分页列表 \t\n filter:服务区查询条件封装对象JSON " +
            "{\n" +
            "    \"code\": 0,\n" +
            "    \"name\": \"string\",\n" +
            "    \"groupId\": 0,\n" +
            "    \"instanceId\": 0,\n" +
            "    \"tenantId\": 0\n" +
            "}" +
            "\t\n pageNum:指定页码 \t\n pageSize:指定每页返回的对象个数")
    RestResponse<PageInfo<ServiceAreaRespDto>> queryServiceAreaByPage(@RequestParam("filter") String filter,
                                                                      @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                      @Max(value = 100, message = "最大值不能超过100")
                                                                      @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据查询条件查询仓库列表
     * @param queryReqDto 查询条件
     * @param pageSize       分页参数-页大小 不得小于1
     * @param pageNum           分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表
     */
    @GetMapping(value = "/warehousePage")
    @ApiOperation(value = "查询仓库列表", notes = "查询仓库列表")
    RestResponse<PageInfo<WarehouseRespDto>> queryWarehousePage(@SpringQueryMap WarehouseQueryReqDto queryReqDto,
                                                                @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据当前经纬度按距离返回仓库列表
     *
     * @param coordinateQueryReqDto
     * @return
     */
    @PostMapping(value = "/queryByLogAndLat")
    @ApiOperation(value = "根据当前经纬度按距离返回仓库列表", notes = "根据当前经纬度按距离返回仓库列表")
    RestResponse<PageInfo<WarehouseRespDto>> queryByLogAndLat(@RequestBody WarehouseCoordinateQueryReqDto coordinateQueryReqDto);

    /*
     * 查询仓库配送方列表
     *
     * @param warehousesId
     * @return
     */
    @GetMapping(value = "/{warehouseId}/delivery")
    @ApiOperation(value = "查询仓库配送方列表", notes = "查询仓库配送方列表")
    RestResponse<List<WarehouseDeliveryQueryRespDto>> queryWarehouseDeliveryList(@PathVariable("warehouseId") Long warehousesId);


    /**
     * 查询仓库类型
     * @param parentCode
     * @return
     */
    @GetMapping(value = "/type")
    @ApiOperation(value = "查询仓库类型", notes = "查询仓库类型")
    RestResponse<List<WarehouseTypeDto>> queryWarehouseType(@RequestParam(value = "parentCode", required = false) Integer parentCode);


    /**
     * 根据仓库code集合查询仓库信息（新接口）
     *
     * @param codes code集合
     * @return 仓库信息集合
     */
    @PostMapping(value = "/queryListByCodes")
    @ApiOperation(value = "根据仓库code集合查询仓库信息（新接口）", notes = "根据仓库code集合查询仓库信息（新接口）")
    RestResponse<List<WarehouseRespDto>> queryListByCodes(@RequestBody List<String> codes);
}