package com.dtyunxi.yundt.cube.center.inventory.api.utils;

/**
 * @(#)ReflectionUtil.java 1.0 2018年4月9日
 *
 * Copyright (c) 2016, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

public class ReflectionUtil {
	private static final String SETTER_PREFIX = "set";

	private static final String GETTER_PREFIX = "get";


	private static Map<String,List<Field>> filedMap = new HashMap<>();
	private static Map<String, Map<String, Method>> methodMap = new HashMap<>();
	private static Object lock = new Object();


	
	public static <T> T invokeGetterAndConvert(Object obj, String propertyName, Class<T> clazz) {
		String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(propertyName);
		T objVal = (T) invokeMethod(obj, getterMethodName, new Class[] {}, new Object[] {});
		return objVal;
	}

	/**
	 * 调用Setter方法, 仅匹配方法名。
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 * @time 2018年4月9日下午2:28:59
	 */
	public static void invokeSetter(Object obj, String propertyName, Object value) {
		String setterMethodName = SETTER_PREFIX + StringUtils.capitalize(propertyName);
		invokeMethodByName(obj, setterMethodName, new Object[] { value });
	}

	/**
	 * 判断是否存在get方法
	 */
	public static boolean isExistGetter(Object obj, String propertyName) {
		String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(propertyName);
		return isExistMethod(obj, getterMethodName);
	}
	
	/**
	 * 判断是否存在set方法
	 */
	public static boolean isExistSetter(Object obj, String propertyName) {
		String setterMethodName = SETTER_PREFIX + StringUtils.capitalize(propertyName);
		return isExistMethod(obj, setterMethodName);
	}
	
	/**
	 * 是否存在该方法
	 */
	public static boolean isExistMethod(Object obj, String methodName) {
		Map<String, Method> map = methodMap.get(obj.getClass().getName());
		if(map != null && map.get(methodName) != null) {
			return true;
		}
		map = new HashMap<String, Method>();
		methodMap.put(obj.getClass().getName(), map);
		
		for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
			Method[] methods = searchType.getDeclaredMethods();
			for (Method method : methods) {
				map.put(method.getName(), method);
				if (method.getName().equals(methodName)) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * 直接调用对象方法, 无视private/protected修饰符.
	 * 用于一次性调用的情况，否则应使用getAccessibleMethod()函数获得Method后反复调用.
	 * 同时匹配方法名+参数类型，
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 * @time 2018年4月9日下午2:29:45
	 */
	public static Object invokeMethod(final Object obj, final String methodName, final Class<?>[] parameterTypes,
			final Object[] args) {
		Method method = getAccessibleMethod(obj, methodName, parameterTypes);
		if (method == null) {
			throw new IllegalArgumentException("在 [" + obj + "]"+"类里面找不到方法 [" + methodName + "]");
		}

		try {
			return method.invoke(obj, args);
		} catch (Exception e) {
			throw convertReflectionExceptionToUnchecked(e);
		}
	}

	/**
	 * 直接调用对象方法, 无视private/protected修饰符，
	 * 用于一次性调用的情况，否则应使用getAccessibleMethodByName()函数获得Method后反复调用.
	 * 只匹配函数名，如果有多个同名函数调用第一个。
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 */
	public static Object invokeMethodByName(final Object obj, final String methodName, final Object[] args) {
		Method method = getAccessibleMethodByName(obj, methodName);
		if (method == null) {
			throw new IllegalArgumentException("在 [" + obj + "]"+"类里面找不到方法 [" + methodName + "]");
		}

		try {
			return method.invoke(obj, args);
		} catch (Exception e) {
			throw convertReflectionExceptionToUnchecked(e);
		}
	}



	/**
	 * 获取对象和超类的属性字段
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 */
	public static List<Field> getDeclaredFields(final Class<?> clazz){
		List<Field> list = new ArrayList<>();
		for (Class<?> superClass = clazz; superClass != Object.class; superClass = superClass.getSuperclass()) {
			Field[] fields = superClass.getDeclaredFields();
			list.addAll(Arrays.asList(fields));
		}
		return list;
	}

	/**
	 * 获取对象和超类的属性字段
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 */
	public static List<Field> getDeclaredFieldsUseCache(final Class<?> clazz){
		String key = getFiledMapKey(clazz);
		List<Field> fields = filedMap.get(key);
		if(fields == null){
			synchronized (lock) {
				 fields = filedMap.get(key);
				 if(fields == null){
					 initFileds(clazz);
					 fields = filedMap.get(key);
				 }
			}
		}
		return fields;
	}

	private static void initFileds (final Class<?> clazz){
		List<Field> fields = ReflectionUtil.getDeclaredFields(clazz);
		filedMap.put(getFiledMapKey(clazz), fields);
	}

	private static String getFiledMapKey(final Class<?> clazz){
		return clazz.getName();
	}

	/**
	 * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
	 * 如向上转型到Object仍无法找到, 返回null.
	 * 匹配函数名+参数类型。
	 * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 */
	public static Method getAccessibleMethod(final Object obj, final String methodName,
			final Class<?>... parameterTypes) {
		Validate.notNull(obj, "用于反射的对象不能为NULL");
		Validate.notBlank(methodName, "反射字段不能为空");

		for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
			try {
				Method method = searchType.getDeclaredMethod(methodName, parameterTypes);
				makeAccessible(method);
				return method;
			} catch (NoSuchMethodException e) {
				// Method不在当前类定义,继续向上转型
			}
		}
		return null;
	}

	/**
	 * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
	 * 如向上转型到Object仍无法找到, 返回null.
	 * 只匹配函数名。
	 * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 * @time 2018年4月9日下午2:44:16
	 * @param obj
	 * @param methodName
	 * @return
	 */
	public static Method getAccessibleMethodByName(final Object obj, final String methodName) {
		Validate.notNull(obj, "用于反射的对象不能为NULL");
		Validate.notBlank(methodName, "反射字段不能为空");

		for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
			Method[] methods = searchType.getDeclaredMethods();
			for (Method method : methods) {
				if (method.getName().equals(methodName)) {
					makeAccessible(method);
					return method;
				}
			}
		}
		return null;
	}

	/**
	 * 改变private/protected的方法为public，尽量不调用实际改动的语句，避免JDK的SecurityManager抱怨。
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 * @time 2018年4月9日下午2:50:00
	 * @param method
	 */
	public static void makeAccessible(Method method) {
		if ((!Modifier.isPublic(method.getModifiers()) || !Modifier.isPublic(method.getDeclaringClass().getModifiers()))
				&& !method.isAccessible()) {
			method.setAccessible(true);
		}
	}




	/**
	 * 将反射时的checked exception转换为unchecked exception.
	 * @author lin.jianzhong<guanqin.ljz@dtyunxi.com>
	 */
	public static RuntimeException convertReflectionExceptionToUnchecked(Exception e) {
		if (e instanceof IllegalAccessException || e instanceof IllegalArgumentException
				|| e instanceof NoSuchMethodException) {
			return new IllegalArgumentException(e);
		} else if (e instanceof InvocationTargetException) {
			return new RuntimeException(((InvocationTargetException) e).getTargetException());
		} else if (e instanceof RuntimeException) {
			return (RuntimeException) e;
		}
		return new RuntimeException("Unexpected Checked Exception.", e);
	}


}

