/*
 * IShippingApi.class 2.0 2018-09-25
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.shipping.api;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.request.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * 物流管理Api
 *
 * @author 佐助
 * @since 2.0.0
 */
@Api(tags = {"库存中心：物流管理服务"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/shipping")
public interface IShippingApi {


    /**
     * 新建物流公司
     *
     * @param shippingCompanyCreateReqDto 物流公司创建Dto
     * @return Long 返回统一响应对象，业务数据data为主键id
     */
    @PostMapping(value = "/company", produces = "application/json")
    @ApiOperation(value = "新建物流公司", notes = "新建物流公司 \t\n shippingCompanyCreateReqDto:物流公司创建Dto  返回主键id")
    RestResponse<Long> addShippingCompany(@Valid @RequestBody ShippingCompanyCreateReqDto shippingCompanyCreateReqDto);

    /**
     * 修改物流公司信息
     *
     * @param shippingCompanyUpdateReqDto 物流公司更新Dto
     * @param id                          物流公司ID
     * @return 操作结果
     */
    @PutMapping(value = "/company/{id}", produces = "application/json")
    @ApiOperation(value = "修改物流公司信息", notes = "修改物流公司信息 " +
            "\t\n shippingCompanyQueryReqDto:物流公司请求Dto \t\n id:物流公司id")
    RestResponse<Void> modifyShippingCompany(@Valid @RequestBody ShippingCompanyUpdateReqDto shippingCompanyUpdateReqDto,
                                             @NotNull(message = "物流公司id不能为空") @PathVariable("id") Long id);

    /**
     * 根据id删除物流公司
     *
     * @param id     物流公司id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 操作结果
     */
    @DeleteMapping(value = "/company/{id}", produces = "application/json")
    @ApiOperation(value = "根据id删除物流公司", notes = "根据id删除物流公司 \t\n id:物流公司id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有用到")
    RestResponse<Void> removeCompanyById(@NotNull(message = "物流公司id不能为空") @PathVariable("id") Long id,
                                         @RequestParam("filter") String filter);

    /**
     * 根据code删除物流公司
     *
     * @param code   物流公司编码
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 操作结果
     */
    @DeleteMapping(value = "/company/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code删除物流公司", notes = "根据code删除物流公司 \t\n code:物流公司code " +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1} #扩展用处的filter目前没有用到")
    RestResponse<Void> removeCompanyByCode(@NotBlank(message = "物流公司code不能为空") @PathVariable("code") String code,
                                           @RequestParam("filter") String filter);

    /**
     * 新增运费模板与规则
     *
     * @param shippingTemplateCreateReqDto 运费模板与规则新建Dto
     * @return 操作结果
     */
    @PostMapping(value = "/template", produces = "application/json")
    @ApiOperation(value = "新增运费模板与规则", notes = "新增运费模板与规则 " +
            "\t\n shippingTemplateQueryReqDto:运费模板与规则请求参数Dto")
    RestResponse<Long> addShippingTemplateAndRule(@Valid @RequestBody ShippingTemplateCreateReqDto shippingTemplateCreateReqDto);

    /**
     * 修改运费模板与规则
     *
     * @param shippingTemplateUpdateReqDto 运费模板与规则修改Dto
     * @return 操作结果
     */
    @PutMapping(value = "/template/{id}", produces = "application/json")
    @ApiOperation(value = "修改运费模板与规则", notes = "修改运费模板与规则 " +
            "\t\n shippingTemplateQueryReqDto:运费模板与规则请求参数Dto")
    RestResponse<Void> modifyTemplateAndRule(@NotNull(message = "运费模板id不能为空") @PathVariable("id") Long id,
                                             @Valid @RequestBody ShippingTemplateUpdateReqDto shippingTemplateUpdateReqDto);

    /**
     * 根据id删除运费模板与规则
     *
     * @param id     运费模板id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 操作结果
     */
    @DeleteMapping(value = "/template/{id}", produces = "application/json")
    @ApiOperation(value = "根据id删除运费模板与规则", notes = "根据id删除运费模板与规则 " +
            "\t\n id:运费模板id \t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter入参目前没有做处理")
    RestResponse<Void> removeTemplateById(@NotNull(message = "运费模板id不能为空") @PathVariable("id") Long id,
                                          @RequestParam("filter") String filter);

    /**
     * 根据code删除运费模板与规则
     *
     * @param code   编码
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 操作结果
     */
    @DeleteMapping(value = "/template/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code删除运费模板与规则", notes = "根据code删除运费模板与规则 \t\n code:编码" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter入参目前没有做处理")
    RestResponse<Void> removeTemplateByCode(@NotBlank(message = "运费模板code不能为空") @PathVariable("code") String code,
                                            @RequestParam("filter") String filter);

    /**
     * 运费模板与地区关联
     *
     * @param rTtemplateAreaReqDto 运费模板与地区关联Dto
     * @return 操作结果
     */
    @PostMapping(value = "/template/relate/area", produces = "application/json")
    @ApiOperation(value = "运费模板与地区关联", notes = "运费模板与地区关联")
    RestResponse<Void> addRelationToArea(@Valid @RequestBody RTtemplateAreaReqDto rTtemplateAreaReqDto);

    /**
     * 根据运费模板id删除关联关系
     *
     * @param id     运费模板id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return 操作结果
     */
    @DeleteMapping(value = "/template/{id}/relate/area", produces = "application/json")
    @ApiOperation(value = "根据运费模板id删除关联关系", notes = "根据运费模板id删除关联关系 \t\n" +
            "filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter入参目前没有做处理")
    RestResponse<Void> removeRelationByShippingTplId(@NotNull(message = "运费模板id不能为空") @PathVariable("id") Long id,
                                                     @RequestParam("filter") String filter);

    /**
     * 新增物流渠道
     *
     * @param shippingPartnerCreateReqDto 物流渠道创建封装对象
     * @return 操作结果
     */
    @PostMapping("/partner")
    @ApiOperation(value = "物流渠道新增", notes = "物流渠道新增")
    RestResponse<Void> addShippingPartner(@Valid @RequestBody ShippingPartnerCreateReqDto shippingPartnerCreateReqDto);

    /**
     * 根据物流渠道id更新物流渠道
     *
     * @param id                          物流渠道id
     * @param shippingPartnerUpdateReqDto 物流渠道更新封装对象
     * @return 操作结果
     */
    @PutMapping("/partner/{id}")
    @ApiOperation(value = "根据物流渠道id更新物流渠道", notes = "根据物流渠道id更新物流渠道")
    RestResponse<Void> modifyShippingPartnerById(@NotNull(message = "物流渠道id") @PathVariable("id") Long id,
                                                 @Valid @RequestBody ShippingPartnerUpdateReqDto shippingPartnerUpdateReqDto);

    /**
     * 根据id删除物流渠道
     *
     * @param id 物流渠道id
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/partner/{id}")
    @ApiOperation(value = "根据id删除物流渠道", notes = "根据id删除物流渠道 id:物流渠道id" +
            "filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有处理")
    RestResponse<Void> removePartnerById(@NotNull(message = "物流渠道id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 新增物流渠道配置
     *
     * @param shippingPartnerConfigCreateReqDto 物流渠道配置创建封装对象
     * @return 操作结果
     */
    @PostMapping("/partner/config")
    @ApiOperation(value = "新增物流渠道配置", notes = "新增物流渠道配置")
    RestResponse<Void> addShippingPartnerConfig(@Valid @RequestBody ShippingPartnerConfigCreateReqDto shippingPartnerConfigCreateReqDto);

    /**
     * 根据物流渠道配置id更新物流渠道
     *
     * @param id                                物流渠道配置id
     * @param shippingPartnerConfigUpdateReqDto 物流渠道配置更新封装对象
     * @return 操作结果
     */
    @PutMapping("/partner/config/{id}")
    @ApiOperation(value = "根据物流渠道配置id更新物流渠道配置", notes = "根据物流渠道配置id更新物流渠道配置")
    RestResponse<Void> modifyShippingPartnerConfigById(@NotNull(message = "物流渠道配置id") @PathVariable("id") Long id,
                                                       @Valid @RequestBody ShippingPartnerConfigUpdateReqDto shippingPartnerConfigUpdateReqDto);

    /**
     * 根据id删除物流渠道配置
     *
     * @param id 物流渠道配置id
     * @return 返回统一响应对象
     */
    @DeleteMapping(value = "/partner/config/{id}")
    @ApiOperation(value = "根据id删除物流渠道配置", notes = "根据id删除物流渠道配置 id:物流渠道配置id" +
            "filter {\"instanceId\":0,\"tenantId\":1} #扩展用处的filter目前没有处理")
    RestResponse<Void> removeConfigById(@NotNull(message = "物流渠道配置id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);
}