/*
 * IPackageQueryApi.class 1.0 2018-09-06
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.shipping.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.PackageRespDto;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.PackageShippingLogRespDto;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.PackageSplitConfigRespDto;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.constraints.Max;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 包裹物流查询服务
 *
 * @author 纳兰
 * @since 2.0.0
 */
@Api(tags = {"库存中心：包裹物流管理服务"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/package")
public interface IPackageQueryApi {
    /**
     * 根据发货单ID查询发货单拆包情况
     *
     * @param deliveryOrderId 发货单ID
     * @return PackageRespDto 货品id对应的货品详情
     */
    @GetMapping("/delivery-order-id/{deliveryOrderId}")
    @ApiOperation(value = "根据发货单ID查询发货单拆包情况", notes = "根据发货单ID查询发货单拆包情况 \t\n deliveryOrderId:发货单ID")
    RestResponse<PackageRespDto> queryPackageById(@NotNull(message = "发货单ID不能为空") @PathVariable("deliveryOrderId") Long deliveryOrderId);

    /**
     * 根据id查询拆包配置信息
     *
     * @param id     拆包配置id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return PackageSplitConfigRespDto 返回统一响应对象，业务数据data为拆包配置请求Dto
     */
    @GetMapping(value = "/split-config/{id}", produces = "application/json")
    @ApiOperation(value = "根据id查询拆包配置信息", notes = "根据id查询拆包配置信息 \t\n id:拆包配置id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<PackageSplitConfigRespDto> queryConfigById(@NotNull(message = "拆包配置id不能为空") @PathVariable("id") Long id,
                                                            @RequestParam("filter") String filter);

    /**
     * 根据包裹查询物流信息
     *
     * @param id 包裹id
     * @return String 统一响应对象，业务数据data为物流信息
     */
    @GetMapping(value = "/{id}/log", produces = "application/json")
    @ApiOperation(value = "根据包裹查询物流信息", notes = "根据包裹查询物流信息 \t\n id:包裹id")
    RestResponse<PackageShippingLogRespDto> queryPackLogById(@PathVariable("id") Long id);

    /**
     * 根据code查询拆包配置信息
     *
     * @param code   编码
     * @param filter {"instanceId":0,"tenantId":1}
     * @return PackageSplitConfigRespDto 返回统一响应对象，业务数据data为拆包配置请求Dto
     */
    @GetMapping(value = "/split-config/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code查询拆包配置信息", notes = "根据code查询拆包配置信息 \t\n code:拆包配置编码code" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<PackageSplitConfigRespDto> queryConfigByCode(@NotBlank(message = "拆包配置code不能为空") @PathVariable("code") String code,
                                                              @RequestParam("filter") String filter);

    /**
     * 根据查询条件查询符合条件的拆包配置分页列表
     *
     * @param filter   查询条件
     * @param pageNum  指定页码
     * @param pageSize 页大小
     * @return PackageSplitConfigRespDto分页信息 拆包配置分页列表
     */
    @GetMapping(value = "/split-config/page", produces = "application/json")
    @ApiOperation(value = "根据查询条件查询拆包配置分页列表",
            notes = "根据查询条件查询拆包配置分页列表 \t\n filter:查询条件 \t\n pageNum:指定页码 \t\n pageSize:页大小")
    RestResponse<PageInfo<PackageSplitConfigRespDto>> queryConfigByPage(@RequestParam("filter") String filter,
                                                                        @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                        @Max(value = 100, message = "最大值不能超过100")
                                                                        @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据分组id查询拆包配置列表
     *
     * @param groupId 分组id
     * @param filter  {"instanceId":0,"tenantId":1}
     * @return PackageSplitConfigRespDto集合 返回统一响应对象，业务数据dta为拆包配置列表
     */
    @GetMapping(value = "/split-config/group-id/{groupId}/list", produces = "application/json")
    @ApiOperation(value = "根据分组id查询拆包配置列表", notes = "根据分组id查询拆包配置列表 \t\n groupId:分组id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}")
    RestResponse<List<PackageSplitConfigRespDto>> queryConfigByGroupId(@NotNull(message = "分组id不能为空") @PathVariable("groupId") Long groupId,
                                                                       @RequestParam("filter") String filter);


}
