/**
 * @(#)IShippingApi.java 1.0 2018年9月6日
 * <p>
 * Copyright (c) 2016, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.shipping.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.ShippingCompanyRespDto;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.ShippingPartnerConfigRespDto;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.ShippingPartnerRespDto;
import com.dtyunxi.yundt.cube.center.shipping.api.dto.response.ShippingTemplateRespDto;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.constraints.Max;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 物流查询API接口
 *
 * @author 水镜
 * @since 2.0.0
 */
@Api(tags = {"库存中心：物流查询接口"})
@FeignClient(
        name = "${dtyunxi.yundt.cube_base-center-inventory_api.name:base-center-inventory}",
        url = "${dtyunxi.yundt.cube_base-center-inventory_api:}",
        path = "/v2/shipping")
public interface IShippingQueryApi {

    /**
     * 根据运单号查询物流信息
     *
     * @param shippingNo 运单号
     * @return String 统一响应对象，业务数据data为物流信息
     */
    @GetMapping(value = "/shipping-no/{shippingNo}/log", produces = "application/json")
    @ApiOperation(value = "根据运单号查询物流信息 返回数据为物流信息", notes = "根据运单号查询物流信息 返回数据为物流信息")
    RestResponse<String> queryLogByShippingNo(@NotNull(message = "运单号不能为空") @PathVariable("shippingNo") String shippingNo);


    /**
     * 根据id查询物流公司信息
     *
     * @param id     物流公司id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return ShippingCompanyRespDto 返回统一响应对象，业务对象data为物流公司响应Dto
     */
    @GetMapping(value = "/company/{id}", produces = "application/json")
    @ApiOperation(value = "根据id查询物流公司信息", notes = "根据id查询物流公司信息 \t\n id:物流公司id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有用到")
    RestResponse<ShippingCompanyRespDto> queryCompanyById(@NotNull(message = "物流公司id不能为空") @PathVariable("id") Long id,
                                                          @RequestParam("filter") String filter);

    /**
     * 根据code查询物流公司信息
     *
     * @param code   物流公司code
     * @param filter {"instanceId":0,"tenantId":1}
     * @return ShippingCompanyRespDto 返回统一响应对象，业务对象data为物流公司响应Dto
     */
    @GetMapping(value = "/company/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据code查询物流公司信息", notes = "根据code查询物流公司信息 \t\n code:物流公司code" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}   #扩展用处的filter目前没有用到")
    RestResponse<ShippingCompanyRespDto> queryCompanyByCode(@NotBlank(message = "物流公司code不能为空") @PathVariable("code") String code,
                                                            @RequestParam("filter") String filter);

    /**
     * 根据条件查询物流公司分页列表
     *
     * @param filter   查询条件
     * @param pageNum  指定页
     * @param pageSize 页大小
     * @return ShippingCompanyRespDto分页信息 返回统一响应对象，业务对象data为物流公司响应Dto列表
     */
    @GetMapping(value = "/company/page", produces = "application/json")
    @ApiOperation(value = "根据条件查询物流公司分页列表", notes = "根据条件查询物流公司分页列表  code 物流公司编码  name 物流公司名称" +
            "\t\n filter:ShippingCompanyQueryReqDto \t\n" +
            "{\n" +
            "  \"id\":0,\n" +
            "  \"code\": \"string\",\n" +
            "  \"name\": \"string\",\n" +
            "  \"instanceId\": 0,\n" +
            "  \"tenantId\": 0\n" +
            "}")
    RestResponse<PageInfo<ShippingCompanyRespDto>> queryCompanyByPage(@RequestParam(name = "filter", required = false) String filter,
                                                                      @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                      @Max(value = 100, message = "最大值不能超过100")
                                                                      @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据物流渠道id查询物流渠道详情
     *
     * @param id 物流渠道id
     * @return ShippingPartnerRespDto 物流渠道id对应的物流渠道详情
     */
    @GetMapping("/partner/{id}")
    @ApiOperation(value = "根据物流渠道id查询物流渠道详情", notes = "根据物流渠道id查询物流渠道详情 id:物流渠道id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1} #扩展用处的filter目前没有用到")
    RestResponse<ShippingPartnerRespDto> queryPartnerById(@NotNull(message = "物流渠道id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 根据物流渠道配置id查询物流渠道配置详情
     *
     * @param id 物流渠道配置id
     * @return ShippingPartnerConfigRespDto 物流渠道配置id对应的物流渠道配置详情
     */
    @GetMapping("/partner/config/{id}")
    @ApiOperation(value = "根据物流渠道配置id查询物流渠道配置详情", notes = "根据物流渠道配置id查询物流渠道配置详情 id:物流渠道配置id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有用到")
    RestResponse<ShippingPartnerConfigRespDto> queryConfigById(@NotNull(message = "物流渠道配置id不能为空") @PathVariable("id") Long id, @RequestParam("filter") String filter);


    /**
     * 根据id查询运费模板与规则
     *
     * @param id     运费模板id
     * @param filter {"instanceId":0,"tenantId":1}
     * @return ShippingTemplateRespDto 运费模板响应Dto
     */
    @GetMapping(value = "/template/{id}", produces = "application/json")
    @ApiOperation(value = "根据id查询运费模板与规则", notes = "根据id查询运费模板与规则 \t\n id:运费模板id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有用到")
    RestResponse<ShippingTemplateRespDto> queryTemplateById(@NotNull(message = "运费模板id不能为空") @PathVariable("id") Long id,
                                                            @RequestParam("filter") String filter);

    /**
     * 根据运费模板编码code查询运费模板与规则
     *
     * @param code   编码
     * @param filter {"instanceId":0,"tenantId":1}
     * @return ShippingTemplateRespDto 运费模板响应Dto
     */
    @GetMapping(value = "/template/code/{code}", produces = "application/json")
    @ApiOperation(value = "根据运费模板编码code查询运费模板与规则", notes = "根据code查询运费模板与规则 \t\n code:运费模板编码 " +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1}  #扩展用处的filter目前没有用到")
    RestResponse<ShippingTemplateRespDto> queryTemplateByCode(@NotBlank(message = "运费模板code不能为空") @PathVariable("code") String code,
                                                              @RequestParam("filter") String filter);

    /**
     * 根据分组id查询运费模板与规则
     *
     * @param groupId 分组id
     * @param filter  {"instanceId":0,"tenantId":1}
     * @return ShippingTemplateRespDto集合 运费模板响应Dto列表
     */
    @GetMapping(value = "/template/group-id/{groupId}", produces = "application/json")
    @ApiOperation(value = "根据分组id查询运费模板与规则", notes = "根据分组id查询运费模板与规则 \t\n groupId:分组id " +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1} #扩展用处的filter目前没有用到")
    RestResponse<List<ShippingTemplateRespDto>> queryTemplateByGroupId(@NotNull(message = "分组id不能为空") @PathVariable("groupId") Long groupId,
                                                                       @RequestParam("filter") String filter);

    /**
     * 根据服务区id查询运费模板与规则
     *
     * @param serviceAreaId 服务区id
     * @param filter        {"instanceId":0,"tenantId":1}
     * @return ShippingTemplateRespDto集合 运费模板响应Dto列表
     */
    @GetMapping(value = "/template/service-area-id/{serviceAreaId}", produces = "application/json")
    @ApiOperation(value = "根据服务区id查询运费模板与规则", notes = "根据服务区id查询运费模板与规则 \t\n serviceAreaId:服务区id" +
            "\t\n filter {\"instanceId\":0,\"tenantId\":1} #扩展用处的filter目前没有用到")
    RestResponse<List<ShippingTemplateRespDto>> queryTemplateByServiceAreaId(@NotNull(message = "服务区id不能为空") @PathVariable("serviceAreaId") Long serviceAreaId,
                                                                             @RequestParam("filter") String filter);

    /**
     * 根据条件查询 运费模板与规则 分页列表
     *
     * @param filter   查询条件
     * @param pageNum  指定页
     * @param pageSize 页大小
     * @return ShippingTemplateRespDto分页信息 费模板与规则分页列表
     */
    @GetMapping(value = "/template/page", produces = "application/json")
    @ApiOperation(value = "根据条件查询 运费模板与规则分页列表", notes = "根据条件查询 运费模板与规则分页列表" +
            " 【code ：运费模板编码；name ：运费模板名称 ；computeMode： BY_PIECE按件 BY_VOLUME按体积 BY_WEIGHT按重量 BY_ORDER按订单；type：CUSTOMER 面向顾客 OPERATER】" +
            "\t\nfilter:\t\n" +
            "{\n" +
            "  \"id\":0,\n" +
            "  \"code\": \"string\",\n" +
            "  \"name\": \"string\",\n" +
            "  \"type\": \"string\",\n" +
            "  \"computeMode\": \"string\",\n" +
            "  \"groupId\": 0,\n" +
            "  \"instanceId\": 0,\n" +
            "  \"tenantId\": 0\n" +
            "}")
    RestResponse<PageInfo<ShippingTemplateRespDto>> queryTemplateByPage(@RequestParam("filter") String filter,
                                                                        @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNum,
                                                                        @Max(value = 100, message = "最大值不能超过100")
                                                                        @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize);


}