/**
 * @(#)IUserApi.java 1.0 May 17, 2018
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved. YUNXI PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.request.AddUserAndGiveRolesReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.request.UserModifyReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.vo.ImportUserResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Map;

/**
 * 用户服务API接口
 *
 * @author 黑马
 * @since 1.0.0
 */
@Api(tags = {"用户中心：用户账号服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-IUserApi", path = "/v1/user", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IUserApi {

    /**
     * 批量导入用户。2019年6月份版本(2.4.6.0)移除该API
     * <p>
     * 业务需要改用  #addBatchUsers() 方法。
     *
     * @param instanceId 实例Id
     * @param list       批量导入用户List
     * @return List
     */
    @Deprecated
    @PostMapping("/{instanceId}/outer")
    @ApiOperation(value = "批量导入用户", notes = "批量导入用户，6月份版本(2.4.6.0)移除")
    @Capability(capabilityCode = "user.user.import-users")
    RestResponse<List<Map<String, String>>> importUsers(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<ImportUsersDto> list);


    /**
     * iMarket专用定制接口，其他项目不能使用。批量导入用户,返回成功添加的用户和失败原因
     * 2019年6月份版本(2.4.6.0)移除该API
     *
     * @param instanceId 实例Id
     * @param list       批量导入用户List
     * @return List
     */
    @Deprecated
    @PostMapping("/{instanceId}/batch")
    @ApiOperation(value = "iMarket专用定制接口，其他项目不能使用。批量导入用户，返回成功添加的用户和失败原因",
            notes = "6月份版本(2.4.6.0)移除。iMarket专用定制接口，其他项目不能使用。\n1、批量导入用户，返回成功添加的用户和失败原因\n2、2019-03-12，增加batch字段返回")
    @Capability(capabilityCode = "user.user.add-batch-users")
    RestResponse<ImportUserResult> addBatchUsers(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<ImportUsersDto> list);


    /**
     * 批量导入用户,返回成功添加的用户和失败原因
     * 2019年6月份版本(2.4.6.0)移除该API
     *
     * @param instanceId 实例IdbatchImportUser
     * @param list       批量导入用户List
     * @return List
     */
    @Deprecated
    @PostMapping("/{instanceId}/batch-import")
    @ApiOperation(value = "批量导入用户，返回成功添加的用户和失败原因", notes = "批量导入用户，返回成功添加的用户和失败原因，6月份版本(2.4.6.0)移除")
    @Capability(capabilityCode = "user.user.batch-import-user")
    RestResponse<ImportUserResult> batchImportUser(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<ImportUsersDto> list);

    /**
     * 用户关系建立
     *
     * @param instanceId   应用Id
     * @param firstUserId  第一个用户的ID
     * @param secondUserId 第二个用户的ID
     * @param relationType 用户关系类型
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/relation/{relationType}/between/{firstUserId}/{secondUserId}"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "绑定用户关系", notes = "instanceId:应用ID \t\n relationType： 用户关系类型 (1.上下级关系、2.同级关系、3.主账号与子账号、4.邀请关系) \t\n firstUserId:第一个用户的ID \t\n secondUserId:第二个用户的ID")
    @Capability(capabilityCode = "user.user.bind-user-relation")
    RestResponse<Void> bindUserRelation(@PathVariable("instanceId") Long instanceId,
                                        @PathVariable("firstUserId") Long firstUserId,
                                        @PathVariable("secondUserId") Long secondUserId,
                                        @PathVariable("relationType") Integer relationType);

    /**
     * 用户关系解绑
     *
     * @param instanceId   应用Id
     * @param firstUserId  第一个用户的ID
     * @param secondUserId 第二个用户的ID
     * @param relationType 用户关系类型
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/relation/{relationType}/between/{firstUserId}/{secondUserId}/unbind"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "解绑用户关系", notes = "instanceId:应用ID \t\n relationType： 用户关系类型 (1.上下级关系、2.同级关系、3.主账号与子账号、4.邀请关系) \t\n firstUserId:第一个用户的ID \t\n secondUserId:第二个用户的ID")
    @Capability(capabilityCode = "user.user.unbind-user-relation")
    RestResponse<Void> unbindUserRelation(@PathVariable("instanceId") Long instanceId,
                                          @PathVariable("firstUserId") Long firstUserId,
                                          @PathVariable("secondUserId") Long secondUserId,
                                          @PathVariable("relationType") Integer relationType);

    /**
     * 创建一个用户，分配ID；如果有邀请码，接口将建立用户间的推荐关系
     *
     * @param instanceId 应用Id
     * @param userDto    用户
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建用户", notes = "instanceId：应用ID \t\n userDto：用户Dto")
    @Capability(capabilityCode = "user.user.add-user")
    RestResponse<Long> addUser(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody UserDto userDto);

    /**
     * 导入用户，分配ID；如果有邀请码，接口将建立用户间的推荐关系，并且存储密码md5的salt,不修改密码
     * 导入历史用户时使用，此时密码是md5(明文+salt值)的，原样存储userDto.password即可，无需再加密处理
     *
     * @param instanceId 应用Id
     * @param userDto    用户
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/salt"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "导入用户", notes = "导入用户，instanceId：应用ID \t\n userDto：用户Dto")
    @Capability(capabilityCode = "user.user.add-user-with-salt")
    RestResponse<Long> addUserWithSalt(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody UserDto userDto);

    /**
     * 创建一个用户，并进行第三方绑定
     *
     * @param instanceId           应用Id
     * @param thirdPartyUserReqDto 用户
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/third-party-user"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建第三方绑定用户", notes = "创建一个用户，并进行第三方绑定。instanceId：应用ID \t\n thirdPartyUserReqDto：第三方用户Dto")
    @Capability(capabilityCode = "user.user.add-third-party-user")
    RestResponse<Long> addThirdPartyUser(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody ThirdPartyUserReqDto thirdPartyUserReqDto);

    /**
     * 创建租户管理员
     *
     * @param tenantId                  租户ID
     * @param tenantManagerCreateReqDto 新增租户管理员dto
     * @return
     * @see #addTenantAdmin
     */
    @Deprecated
    @RequestMapping(value = {"/{tenantId}/manger"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建租户管理员", notes = "tenantId：租户ID \t\n userDto：用户dto")
    @Capability(capabilityCode = "user.user.add-tenant-manager")
    RestResponse<Long> addTenantManager(@PathVariable("tenantId") Long tenantId, @Valid @RequestBody TenantManagerCreateReqDto tenantManagerCreateReqDto);


    /**
     * 创建租户管理员
     *
     * @param tenantManagerCreateReqDto 新增租户管理员dto
     * @return 新增账号的id
     */
    @RequestMapping(value = {"/tenant-admin"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建租户管理员", notes = "创建账号并关联租户管理员")
    @Capability(capabilityCode = "user.user.add-tenant-admin")
    RestResponse<Long> addTenantAdmin(@Valid @RequestBody TenantManagerCreateReqDto tenantManagerCreateReqDto);


    /**
     * 更新用户，不允许修改account和password
     *
     * @param id      ID
     * @param userDto 用户
     * @return void
     */
    @RequestMapping(value = {"/{id}"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "更新用户，不允许修改account和password、角色、个人、组织、地址、联系人等信息", notes = "更新用户，不允许修改account和password。\t\n id：用户ID \t\n userDto：用户Dto")
    @Capability(capabilityCode = "user.user.update")
    RestResponse<Void> update(@PathVariable("id") Long id, @Valid @RequestBody UserDto userDto);

    /**
     * 修改租户管理员
     *
     * @param id                        租户管理员ID
     * @param tenantManagerUpdateReqDto 修改租户管理员dto
     * @return
     */
    @RequestMapping(value = {"/tenant/manager/{id}"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "更新租户管理员", notes = "更新租户管理员。\t\n id：用户ID \t\n tenantManagerUpdateReqDto：修改租户管理员dto")
    @Capability(capabilityCode = "user.user.update-tenant-manager")
    RestResponse<Void> updateTenantManager(@PathVariable("id") Long id, @Valid @RequestBody TenantManagerUpdateReqDto tenantManagerUpdateReqDto);

    /**
     * 更新用户，允许修改密码，不允许修改account
     *
     * @param id      用户ID
     * @param userDto 用户请求Dto
     * @return void
     */
    @PutMapping(value = {"/all/{id}"}, produces = "application/json")
    @ApiOperation(value = "更新用户，允许修改密码，不允许修改account、角色、个人、组织、地址、联系人等信息", notes = "更新用户，允许修改密码，不允许修改account。\t\n id：用户ID \t\n userDto：用户Dto")
    @Capability(capabilityCode = "user.user.update-password-permitted")
    RestResponse<Void> updatePasswordPermitted(@PathVariable("id") Long id, @Valid @RequestBody UserDto userDto);

    /**
     * 更新用户，允许修改密码。UserDto.password原样存储到us_user
     *
     * @param id      用户ID
     * @param userDto 用户请求Dto
     * @return void
     */
    @PutMapping(value = {"/all/{id}/salt"}, produces = "application/json")
    @ApiOperation(value = "更新用户，允许修改密码，不修改角色、个人、组织、地址、联系人等信息", notes = "更新用户，允许修改密码 \t\n id：用户ID \t\n userDto：用户Dto")
    @Capability(capabilityCode = "user.user.update-password-permitted-with-salt")
    RestResponse<Void> updatePasswordPermittedWithSalt(@PathVariable("id") Long id, @Valid @RequestBody UserDto userDto);

    /**
     * 启用租户管理员
     *
     * @param userId 租户管理员
     * @return
     */
    @RequestMapping(value = {"/tenant/manager/{userId}/enable"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "启用租户管理员", notes = "启用租户管理员")
    @Capability(capabilityCode = "user.user.enable-tenant-manager")
    RestResponse<Void> enableTenantManager(@PathVariable("userId") Long userId);

    /**
     * 禁用租户管理员
     *
     * @param userId 租户管理员
     * @return
     */
    @RequestMapping(value = {"/tenant/manager/{userId}/disable"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "禁用租户管理员", notes = "禁用租户管理员")
    @Capability(capabilityCode = "user.user.disable-tenant-manager")
    RestResponse<Void> disableTenantManager(@PathVariable("userId") Long userId);

    /**
     * 删除用户
     * 1。逻辑删除us_user表记录
     * 2.逻辑删除us_login_config登录配置表记录
     * 3.物理删除用户角色关联关系
     * @param userId 用户Id
     * @return void
     */
    @RequestMapping(value = {"/{userId}"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "删除用户", notes = "删除用户：1.逻辑删除用户表us_user记录 2.逻辑删除us_login_config登录配置表记录" +
            "3.物理删除用户角色关联关系")
    @Capability(capabilityCode = "user.user.delete")
    RestResponse<Void> delete(@PathVariable("userId") Long userId);


    /**
     * 根据批次号批量删除用户
     * 1、逻辑删除us_user表记录
     * 2、逻辑删除us_employee员工信息表记录
     * 3、逻辑删除us_login_config登录配置表记录
     *
     * @param batch 批次号
     * @return void
     */
    @RequestMapping(value = {"/batch/{batch}"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "根据批次号批量删除用户", notes = "根据批次号批量删除用户：1、逻辑删除us_user表记录 2、逻辑删除us_employee员工信息表记录 3、逻辑删除us_login_config登录配置表记录")
    @Capability(capabilityCode = "user.user.batch-delete")
    RestResponse<Void> batchDelete(@PathVariable("batch") String batch);

    /**
     * 用户登录方式(启用/禁用)，同时更新us_user和us_login_config。当禁用了用户所有登录方式，该用户也将被禁用
     *
     * @param userId 用户Id
     * @param type   登录类型
     * @param status 状态
     * @return void
     */
    @RequestMapping(value = {"/{userId}/loginType/{type}/enabled/{status}"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "用户登录方式(启用/禁用)，同时更新us_user和us_login_config，当禁用了用户所有登录方式，该用户也将被禁用", notes = "userId:用户ID status: 1启用、2禁用 type: 登录类型(name,phone,email,account)")
    @Capability(capabilityCode = "user.user.enable-login")
    RestResponse<Void> enableLogin(@PathVariable("userId") Long userId,
                                   @PathVariable("type") String type,
                                   @PathVariable("status") Integer status);

    /**
     * 注册一个开发者用户，并分配开发者角色；如果没有开发者角色，先创建一个
     *
     * @param userDto 用户
     * @return Long
     */
    @RequestMapping(value = {"/developer"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "注册开发者", notes = "注册开发者用户，并分配开发者角色；如果没有开发者角色，先创建一个")
    @Capability(capabilityCode = "user.user.register-developer")
    RestResponse<Long> registerDeveloper(@Valid @RequestBody UserDto userDto);

    /**
     * 注册一个租户
     *
     * @param tenantDto 租户
     * @return Long
     */
    @RequestMapping(value = {"/tenant"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "注册租户", notes = "注册租户")
    @Capability(capabilityCode = "user.user.register-tenant")
    RestResponse<Long> registerTenant(@Valid @RequestBody TenantDto tenantDto);

    /**
     * 修改租户信息，tenantId、instanceId不能被修改
     *
     * @param tenantDto 租户
     * @return Long
     */
    @RequestMapping(value = {"/tenant"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "修改租户", notes = "修改租户信息，tenantId、instanceId不能被修改")
    @Capability(capabilityCode = "user.user.update-tenant")
    RestResponse<Void> updateTenant(@Valid @RequestBody TenantDto tenantDto);

    /**
     * 启用租户
     *
     * @param tenantId
     * @return
     */
    @RequestMapping(value = {"/tenant/{tenantId}/enable"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "启用租户", notes = "启用租户")
    @Capability(capabilityCode = "user.user.enable-tenant")
    RestResponse<Void> enableTenant(@PathVariable("tenantId") Long tenantId);

    /**
     * 禁用租户
     *
     * @param tenantId
     * @return
     */
    @RequestMapping(value = {"/tenant/{tenantId}/disable"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "禁用租户", notes = "禁用租户")
    @Capability(capabilityCode = "user.user.disable-tenant")
    RestResponse<Void> disableTenant(@PathVariable("tenantId") Long tenantId);

    /**
     * 用户唯一性判断
     *
     * @param loginType 登录类型
     * @param value     loginType的值
     * @return Boolean
     */
    @Deprecated
    @RequestMapping(value = {"/isUnique/{loginType}/{value}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "检查用户唯一性", notes = "loginType:登录类型(userName,phone,email) value:loginType的值")
    @Capability(capabilityCode = "user.user.check-user-unique")
    RestResponse<Boolean> checkUserUnique(@PathVariable("loginType") String loginType, @PathVariable("value") String value);


    /**
     * 用户唯一性判断
     *
     * @param loginType 登录类型
     * @param value     loginType的值
     * @return Boolean
     */
    @RequestMapping(value = {"/{instanceId}/isUnique/{loginType}/{value}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "检查用户唯一性", notes = "loginType:登录类型(userName,phone,email) value:loginType的值")
    @Capability(capabilityCode = "user.user.check-user-unique")
    RestResponse<Boolean> checkUserUnique(@PathVariable("instanceId") Long instanceId, @PathVariable("loginType") String loginType, @PathVariable("value") String value);

    /**
     * 设置租户的注册隔离配置
     *
     * @param tenantId
     * @param isolationType
     * @return
     */
    @RequestMapping(value = {"/tenant/{tenantId}/isolation-type/{isolationType}"}, produces = "application/json", method = RequestMethod.POST)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tenantId", value = "租户ID", dataType = "integer", paramType = "path"),
            @ApiImplicitParam(name = "isolationType", value = "注册用户的隔离类型 1.租户级别隔离  2.应用级别隔离", dataType = "integer", paramType = "path")})
    @ApiOperation(value = "设置租户的注册隔离配置", notes = "设置租户的注册隔离配置，若租户配置已存在则更新配置")
    @Capability(capabilityCode = "user.user.set-tenant-register-config")
    RestResponse<Void> setTenantRegisterConfig(@PathVariable("tenantId") Long tenantId, @PathVariable("isolationType") Integer isolationType);

    /**
     * 根据登录字段、密码、应用ID，用户账户合法性验证
     *
     * @param instanceId 应用ID
     * @param loginStr   登录字段，可能是手机号码、用户名、邮箱、account
     * @param password   登录密码
     * @return true 合法用户，false - 非法用户
     */
    @Deprecated
    @RequestMapping(value = {"/{instanceId}/isValid"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "检查用户合法性", notes = "instanceId：实例ID \t\n loginStr：登录字段，可能是手机号码、用户名、邮箱、account \t\n password:密码")
    @Capability(capabilityCode = "user.user.is-valid")
    RestResponse<Boolean> isValid(@PathVariable(name = "instanceId", required = false) Long instanceId,
                                  @RequestParam(name = "tenantId", required = false) Long tenantId,
                                  @RequestParam("loginStr") String loginStr,
                                  @RequestParam("password") String password);

    /**
     * 根据登录字段、密码、应用ID，用户账户合法性验证
     *
     * @param userValidDto 用户账户合法性验证请求dto
     * @return true 合法用户，false - 非法用户
     */
    @RequestMapping(value = {"/validate"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "检查用户合法性", notes = "instanceId：实例ID \t\n loginStr：登录字段，可能是手机号码、用户名、邮箱、account \t\n password:密码")
    @Capability(capabilityCode = "user.user.is-valid")
    RestResponse<Boolean> isValid(@RequestBody UserValidDto userValidDto);


    /**
     * 用户第三方绑定
     *
     * @param userId      用户ID
     * @param bindingsDto 第三方绑定信息
     * @return 绑定的ID
     */
    @RequestMapping(value = {"/{userId}/binding"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "用户第三方绑定", notes = "用户第三方绑定")
    @Capability(capabilityCode = "user.user.add-binding")
    RestResponse<Long> addBinding(@PathVariable("userId") Long userId, @RequestBody BindingsDto bindingsDto);

    /**
     * 第三方登录用户绑定后，解除绑定操作，即删除us_bindings记录
     *
     * @param userId      用户表主键Id
     * @param bindingsDto 解绑操作请求DTO
     */
    @PutMapping(value = {"/{userId}/unbind"}, produces = "application/json")
    @ApiOperation(value = "用户第三方登录的解绑定操作", notes = "用户第三方登录的解绑定操作，即逻辑删除us_bindings记录")
    @Capability(capabilityCode = "user.user.remove-binding")
    RestResponse<Void> removeBinding(@PathVariable("userId") Long userId, @RequestBody BindingsDto bindingsDto);

    /**
     * 用户第三方登录后，解除第三方平台用户openid、unionid和会员的绑定关系
     *
     * @param filter JSON字符串，格式：
     *               "filter":{
     *               "phone":"131688075xx", // phone 和userId 两者其中一个必填
     *               "userId":"12025566465255",
     *               "unionId":"", // 微信unionId  选填
     *               "openId":"", //  第三方openId 选填
     *               "thirdpartyType":"1" // 1 微信，2 QQ 必填
     *               }
     * @return 返回删除的条数
     */
    @DeleteMapping(value = {"/unbinding"}, produces = "application/json")
    @ApiOperation(value = "第三方登录解绑定", notes = "filter=BindingsDto \n filter格式：\n\"filter\":{\n" +
            "       \t\t\"phone\":\"131688075xx\", // phone 和userId 两者其中一个必填\n" +
            "       \t\t\"userId\":\"12025566465255\",\n" +
            "       \t\t\"unionId\":\"\", // 微信unionId  选填 \n" +
            "       \t\t\"openId\":\"\", //  第三方openId 选填\n" +
            "       \t\t\"thirdpartyType\":\"1\" // 1 微信，2 QQ 必填\n" +
            "   }\n")
    @Capability(capabilityCode = "user.user.unbinding")
    RestResponse<Integer> unbinding(@RequestParam("filter") String filter);

    /**
     * 将用户加入用户组
     *
     * @param userId  用户ID
     * @param groupId 用户组ID
     * @return 操作结果
     */
    @RequestMapping(value = {"/{userId}/user-group/{groupId}"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "将用户加入用户组", notes = "将用户加入用户组，userId：用户ID \t\n groupId：用户组ID")
    @Capability(capabilityCode = "user.user.add-user-to-group")
    RestResponse<Void> addUserToGroup(@PathVariable("userId") Long userId, @PathVariable("groupId") Long groupId);

    /**
     * 解除用户角色
     *
     * @param userId  用户ID
     * @param roleIds 角色ID数组
     * @return 操作结果
     */
    @RequestMapping(value = {"/{userId}/roles/{roleIds}"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "解除用户角色", notes = "解除用户角色，userId：用户ID \t\n roleIds：角色ID数组")
    @Capability(capabilityCode = "user.user.remove-user-roles")
    RestResponse<Void> removeUserRoles(@PathVariable("userId") Long userId, @PathVariable("roleIds") Long[] roleIds);

    /**
     * 解除用户绑定的手机号，同时移除该手机登录方式
     *
     * @param userId 用户ID
     * @return 操作结果
     */
    @RequestMapping(value = {"/{userId}/phone"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "解除用户绑定的手机号，同时移除该手机登录方式", notes = "解除用户绑定的手机号，同时移除该手机登录方式")
    @Capability(capabilityCode = "user.user.remove-user-phone")
    RestResponse<Void> removeUserPhone(@PathVariable("userId") Long userId);

    @RequestMapping(value = "", method = RequestMethod.DELETE)
    @ApiOperation(value = "批量删除用户", notes = "批量删除用户")
    @Capability(capabilityCode = "user.user.batch-delete")
    RestResponse<Void> batchDelete(@RequestParam("ids") List<Long> ids);

    @RequestMapping(value = "", method = RequestMethod.POST)
    @ApiOperation(value = "新增用户并关联角色", notes = "新增用户并关联角色")
    @Capability(capabilityCode = "user.user.create-user-and-give-roles")
    RestResponse<Long> createUserAndGiveRoles(@RequestBody AddUserAndGiveRolesReqDto addUserAndGiveRolesReqDto);

    @RequestMapping(value = "", method = RequestMethod.PUT)
    @ApiOperation(value = "编辑用户并关联角色", notes = "编辑用户并关联角色")
    @Capability(capabilityCode = "user.user.modify-user-and-give-roles")
    RestResponse<Void> modifyUserAndGiveRoles(@RequestBody UserModifyReqDto userModifyReqDto);

    /**
     * 创建一个用户，分配ID；如果有邀请码，接口将建立用户间的推荐关系
     *
     * @param userDto 用户
     * @return void
     */
    @RequestMapping(value = {"/bind/role"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建用户并且绑定角色", notes = "userDto：用户Dto")
    @Capability(capabilityCode = "user.user.add-user-and-bind-role")
    RestResponse<Long> addUserAndBindRole(@Valid @RequestBody UserCreateReqDto userDto);

    /**
     * 用户授权多个应用(全量分配，会先删除用户已授权的应用信息)
     *
     * @param userInstanceRelationReqDto 请求Dto
     * @return 操作结果
     */
    @PostMapping({"/instance/relate/all"})
    @ApiOperation(value = "用户授权多个应用(全量分配，会先删除用户已授权的应用信息))")
    @Capability(capabilityCode = "user.user.add-user-instance-relation-all")
    RestResponse<Void> addUserInstanceRelationAll(@Valid @RequestBody UserInstanceRelationReqDto userInstanceRelationReqDto);

    /**
     * 用户授权多个应用(增量分配)
     *
     * @param userInstanceRelationReqDto 请求Dto
     * @return 操作结果
     */
    @PostMapping({"/instance/relate"})
    @ApiOperation(value = "用户授权多个应用(增量分配)")
    @Capability(capabilityCode = "user.user.add-user-instance-relation")
    RestResponse<Void> addUserInstanceRelation(@Valid @RequestBody UserInstanceRelationReqDto userInstanceRelationReqDto);

    /**
     * 解除用户授权应用的关联关系
     *
     * @param userInstanceRelationReqDto 请求Dto
     * @return 操作结果
     */
    @PutMapping({"/instance/relate"})
    @ApiOperation(value = "解除用户授权应用的关联关系)")
    @Capability(capabilityCode = "user.user.delete-user-instance-relation")
    RestResponse<Void> deleteUserInstanceRelation(@Valid @RequestBody UserInstanceRelationReqDto userInstanceRelationReqDto);

}
