package com.dtyunxi.yundt.module.customer.biz.impl.user;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;
import com.alibaba.fastjson.JSON;
import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.entity.NacosPropertiesOrgEntity;
import com.dtyunxi.yundt.cube.center.user.api.dto.entity.ImportBaseResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.tcbj.EmployeeImportReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserOrgizationRespDto;
import com.dtyunxi.yundt.cube.center.user.api.enums.YesOrNoEnum;
import com.dtyunxi.yundt.cube.center.user.api.util.AssertUtil;
import com.dtyunxi.yundt.cube.center.user.api.util.BeanPropertyNullUtil;
import com.dtyunxi.yundt.module.context.api.IContext;
import com.dtyunxi.yundt.module.customer.api.dto.request.user.EmployeeExpandReqDto;
import com.dtyunxi.yundt.module.customer.api.user.ICustomerEmployeeExpandService;
import com.dtyunxi.yundt.module.customer.api.user.IEmployeeExpand2Service;
import com.dtyunxi.yundt.module.customer.api.vo.CustomerEmployeeExcelVo;
import com.dtyunxi.yundt.module.customer.biz.util.CommonUtil;
import com.dtyunxi.yundt.module.customer.biz.util.EasyPoiExportUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @author liangfa
 * @Description 客户人员管理接口实现
 * @date 2022-01-14 14:06
 * @since 1.0.0
 */
@Service
public class CustomerEmployeeExpandServiceImpl implements ICustomerEmployeeExpandService {

    private static final Logger logger = LoggerFactory.getLogger(CustomerEmployeeExpandServiceImpl.class);

    @Resource
    private IEmployeeExpand2Service employeeExpand2Service;
    @Resource
    private NacosPropertiesOrgEntity nacosPropertiesOrgEntity;
    @Resource
    private HttpServletRequest request;
    @Resource
    private IContext context;

    /**
     * 获取请求头的组织ID
     *
     * @return 组织ID
     */
    public String headerOrgId() {
        String organizationId = request.getHeader("yes-req-cus-b2b-organizationId");
        logger.info("获取请求头组织ID：{}", organizationId);
        return organizationId;
    }
    /**
     * 根据实例ID查询TenantId
     *
     * @return
     */
    public Long tenantId() {
        Long tenantId = context.tenantId();
        logger.info("tenantId >>> {}", tenantId);
        return tenantId;
    }



    /**
     * b2b人员导入
     * @return
     */
    @Override
    public RestResponse<ImportBaseResponse> importExcelEmployee(EmployeeImportReqDto importReqDto) {
        String fileUrl = importReqDto.getFileUrl();
        AssertUtil.assertNotBlank(fileUrl, "导入的文件路径为空");
        Long instanceId = importReqDto.getInstanceId();
        String orgIdStr = this.headerOrgId();
        //orgIdStr = "1288861123298304011";
        logger.info("请求头组织ID：{}", orgIdStr);
        // 再次获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(orgIdStr)){
            orgIdStr = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", orgIdStr);
        }
        AssertUtil.assertNotBlank(orgIdStr, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        Long orgId = Long.valueOf(orgIdStr);
        /* 设置默认结果 */
        ImportBaseResponse response = new ImportBaseResponse();
        /* 处理文件，读取数据 */
        ImportParams importParams = new ImportParams();
        importParams.setHeadRows(1);// 表格标题行数,默认0
        importParams.setTitleRows(0);//表头行数,默认1
        importParams.setKeyIndex(null);
        List<CustomerEmployeeExcelVo> importDataList = new ArrayList<>();

        try {
            ExcelImportResult<CustomerEmployeeExcelVo> result = ExcelImportUtil.importExcelMore(EasyPoiExportUtil.getInputStream(fileUrl), CustomerEmployeeExcelVo.class, importParams);
            if (ObjectUtils.isNotEmpty(result)) {
                importDataList = result.getList();
                //筛选对象属性全部为null数据
                importDataList = BeanPropertyNullUtil.getAllFieldNullList(importDataList);
            }
            if (CollectionUtils.isEmpty(importDataList)) {
                throw new BizException("无法读取 Excel 中的数据，请检查数据是否符合模板格式要求");
            }
        } catch (Exception e) {
            throw new BizException("无法读取 Excel 文件，请检查文件类型");
        }
        if (CollectionUtils.isEmpty(importDataList)){
            response.setTotalNum(0);
            response.setSuccessNum(0);
            response.setErrorNum(0);
            return new RestResponse<>(response);
        }
        logger.info("导入数据：{}", JSON.toJSONString(importDataList));
        response.setTotalNum(importDataList.size());


        //获取账户信息
        List<String> accountList = importDataList.stream().filter(data -> StringUtils.isNotEmpty(data.getAccount())).map(CustomerEmployeeExcelVo::getAccount).distinct().collect(Collectors.toList());
        logger.info("获取账户信息获取账号信息：{}", JSON.toJSONString(accountList));
        Map<String, List<UserOrgizationRespDto>> userOrgMap = employeeExpand2Service.getAccoutMap(accountList);
        //获取客户信息
        List<String> customerCodeList = importDataList.stream().filter(data -> StringUtils.isNotEmpty(data.getCustomerCode())).map(CustomerEmployeeExcelVo::getCustomerCode).distinct().collect(Collectors.toList());
        Map<String, CustomerRespDto> orgCustomerMap = employeeExpand2Service.getOrgCustomerMap(customerCodeList);
        /* 校验数据 */
        importDataList = this.verifyData(importDataList,orgCustomerMap,userOrgMap,orgId);
        List<CustomerEmployeeExcelVo> successList = importDataList.stream().filter(dto -> StringUtils.isEmpty(dto.getErrorMsg())).collect(Collectors.toList());

        Map<String,Long> employeeInfoMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(successList)){
            List<Long> userIdList = successList.stream().map(CustomerEmployeeExcelVo::getUserId).collect(Collectors.toList());
            employeeInfoMap = employeeExpand2Service.getEmployeeMap(userIdList,Collections.singletonList(successList.get(0).getOrgId()));
        }

        /* 处理数据 */
        for (CustomerEmployeeExcelVo data : successList){
            //构建人员对象
            EmployeeExpandReqDto employeeExtReqDto = new EmployeeExpandReqDto();
            employeeExtReqDto.setOrganizationId(data.getOrganizationId());
            employeeExtReqDto.setOrgId(data.getOrgId());
            employeeExtReqDto.setName(data.getUserName());//姓名
            employeeExtReqDto.setPosition(data.getPosition());//职位
            employeeExtReqDto.setSex(data.getSex());//性别
            employeeExtReqDto.setStation(data.getDepartment());//部门
            employeeExtReqDto.setPhoneNum(data.getPhoneNumber());//手机号码
            employeeExtReqDto.setUserId(data.getUserId());//账号
            employeeExtReqDto.setOrganizationStatus(YesOrNoEnum.NO.getStatus());
            employeeExtReqDto.setType("内部".equals(data.getEmployeeType()) ? 0 : 1);
            //查询该账号是否已被绑定，若已绑定，则修改，没有则新增
            String key = data.getOrganizationId() + "_" + data.getUserId() + "_" + data.getPhoneNumber();
            try {
                if(!employeeInfoMap.isEmpty() && employeeInfoMap.containsKey(key)){
                    Long id = employeeInfoMap.get(key);
                    //修改
                    employeeExtReqDto.setId(id);
                    logger.info("修改客户人员入参:{}",JSON.toJSONString(employeeExtReqDto));
                    employeeExpand2Service.modifyEmployee(instanceId,employeeExtReqDto);
                }else {
                    employeeExtReqDto.setStartTime(new Date());
                    employeeExtReqDto.setStatus(1);
                    employeeExtReqDto.setType(1);
                    //新增
                    logger.info("新增客户人员入参:{}",JSON.toJSONString(employeeExtReqDto));
                    employeeExpand2Service.addEmployee(instanceId,employeeExtReqDto);
                }
            }catch (Exception e){
                logger.info("客户人员执行失败:{}",e);
                setErrorsMsg(importDataList,data.getAccount(),"执行失败！"+e.getMessage());
            }
        }

        List<CustomerEmployeeExcelVo> errorList = importDataList.stream().filter(vo -> StringUtils.isNotEmpty(vo.getErrorMsg())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(errorList)){
            response.setErrorNum(errorList.size());
            String excelName ="客户人员导入失败信息-"+ com.dtyunxi.cube.utils.DateUtil.getDateFormat(new Date(), com.dtyunxi.cube.utils.DateUtil.YMDSTRING_DATA);
            String exportUrl;
            try {
                exportUrl = EasyPoiExportUtil.getExportUrl(errorList,CustomerEmployeeExcelVo.class, null, excelName, "xls");
                response.setErrorFileUrl(exportUrl);
            } catch (Exception e) {
                logger.info("客户人员导入失败信息文件生成失败");
                logger.info(e.getMessage(),e);
            }
        }

        response.setSuccessNum(response.getTotalNum()-response.getErrorNum());
        return new RestResponse<>(response);
    }

    /**
     *
     * @param list 原数据
     * @param account 账号
     * @param errorMsg 错误信息
     * @return
     */
    public List<CustomerEmployeeExcelVo> setErrorsMsg(List<CustomerEmployeeExcelVo> list,String account,String errorMsg){
        list.forEach(vo->{
            if(StringUtils.isBlank(vo.getErrorMsg()) && vo.getAccount().equals(account)){
                vo.setErrorMsg(errorMsg);
            }
        });
        return list;
    }

    /**
     * 校验数据
     * @param list 原数据
     * @return
     */
    public List<CustomerEmployeeExcelVo> verifyData(List<CustomerEmployeeExcelVo> list,  Map<String, CustomerRespDto> orgCustomerMap,
                                            Map<String, List<UserOrgizationRespDto>> userOrgMap,Long orgId){

        for (CustomerEmployeeExcelVo vo : list) {
            if (StringUtils.isBlank(vo.getCustomerCode())){
                vo.setErrorMsg("客户编号不能为空");
                continue;
            }
            if(orgCustomerMap.isEmpty()|| !orgCustomerMap.containsKey(vo.getCustomerCode())){
                vo.setErrorMsg("客户编号数据错误");
                continue;
            }
            CustomerRespDto customerRespDto = orgCustomerMap.get(vo.getCustomerCode());
            if(Objects.isNull(customerRespDto.getMerchantId()) || customerRespDto.getMerchantId().compareTo(orgId)!=0){
                vo.setErrorMsg("客户编号所属组织数据错误");
                continue;
            }
            vo.setCustomerId(customerRespDto.getId());
            if (StringUtils.isBlank(vo.getAccount())){
                vo.setErrorMsg("登录账号不能为空");
                continue;
            }

            if(userOrgMap.isEmpty() || !userOrgMap.containsKey(vo.getAccount())){
                vo.setErrorMsg("登录账号错误");
                continue;
            }
            List<UserOrgizationRespDto> userOrgizationVos = userOrgMap.get(vo.getAccount());
            List<UserOrgizationRespDto> userOrgList = userOrgizationVos.stream().filter(e -> orgId.compareTo(e.getOrgId()) == 0).collect(Collectors.toList());
            if(CollectionUtils.isEmpty(userOrgList)){
                vo.setErrorMsg("登录账号所属组织错误");
                continue;
            }
            vo.setUserId(userOrgList.get(0).getUserId());
            vo.setOrgId(orgId);
            vo.setOrganizationId(customerRespDto.getOrgInfoId());
            if (StringUtils.isBlank(vo.getUserName())){
                vo.setErrorMsg("姓名不能为空");
                continue;
            }
            if (StringUtils.isBlank(vo.getPosition())){
                vo.setErrorMsg("职位不能为空");
                continue;
            }
            if (StringUtils.isBlank(vo.getEmployeeType())){
                vo.setErrorMsg("人员类型不能为空");
                continue;
            }
            if (!("内部".equals(vo.getEmployeeType()) || "外部".equals(vo.getEmployeeType()))){
                vo.setErrorMsg("人员类型只能为'内部'或'外部'");
                continue;
            }
            if(StringUtils.isNotEmpty(vo.getPhoneNumber())){
                 if(!CommonUtil.checkPhoneNum(vo.getPhoneNumber())){
                    vo.setErrorMsg("手机号格式不正确;");
                    continue;
                }
            }



        }

        return list;
    }




}
