/**
 * @(#)IOauth2Login.java 1.0 2019年7月15日
 *
 * Copyright (c) 2016, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.module.customer.rest.user;

import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.huieryun.cache.api.ICacheService;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.identity.api.dto.request.AuthLoginReqDto;
import com.dtyunxi.yundt.cube.center.identity.api.dto.request.AuthLoginTokenReqDto;
import com.dtyunxi.yundt.cube.center.identity.api.dto.request.UserBindLoginThirdDto;
import com.dtyunxi.yundt.cube.center.identity.api.dto.response.ThirdLoginRespDto;
import com.dtyunxi.yundt.cube.center.identity.api.dto.response.TokenRespDto;
import com.dtyunxi.yundt.module.customer.api.user.IOauth2Login;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;

/**
 * 第三方登录服务rest（目前仅支持微信）
 * 
 * @author wu.sheng
 * @time 2019年7月15日 下午9:09:44
 */
@Api(tags = { "用户组件：第三方登录服务（目前仅支持微信）" })
@RestController
@RequestMapping("/v1/user/oauth2")
public class Oauth2LoginRest {

	private static final Logger logger = LoggerFactory.getLogger(Oauth2LoginRest.class);

	@Resource
	private IOauth2Login oauth2Login;

	@Resource
	private ICacheService cacheService;

	@ApiOperation(value = "第一步：获取登录授权地址", notes = "第三方类型（1：微信，2：QQ，3：Sina微博）")
	@GetMapping("/uri")
	public RestResponse<String> getAuthUri(AuthLoginReqDto authLoginReqDto) {
		return new RestResponse<>(oauth2Login.getAuthUri(authLoginReqDto));
	}

	@GetMapping("/code")
	@ApiOperation(value = "接受微信网页授权code,重定向到redirectUri", notes = "因为前端redirectUri中带有#号，微信重定向会丢失，所以后端服务重定向")
	public void redirectUri(String code, String uriKey, HttpServletResponse httpServletResponse) {
		logger.info("微信网页授权code={}, uri={}", code, uriKey);
		String redirectUri = cacheService.getCache(uriKey, String.class);
		if (StringUtils.isBlank(redirectUri)) {
			throw new BizException("缺少redirectUri，请刷新重试!");
		}
		int index = redirectUri.indexOf("?");
		String codeStr = null;
		if (index > 0) {
			codeStr = "&code=" + code;
		} else {
			codeStr = "?code=" + code;
		}
		redirectUri = redirectUri + codeStr;

		try {
			httpServletResponse.sendRedirect(redirectUri);
		} catch (IOException e) {
			logger.error("忽略重定向异常");
		}
	}

	@ApiOperation(value = "第二步：通过第三方授权码code申请认证中心token令牌", notes = "未绑定手机号码，需要第三步绑定手机号码")
	@PostMapping("/token")
	public RestResponse<ThirdLoginRespDto> getAuthToken(@Valid @RequestBody AuthLoginTokenReqDto authLoginTokenReqDto) {
		return new RestResponse<>(oauth2Login.getAuthToken(authLoginTokenReqDto));
	}

	
	@ApiOperation(value = "第三步：openId绑定手机号,并生成认证中心token令牌")
	@PostMapping("/bind")
	public RestResponse<TokenRespDto> bind(@RequestBody UserBindLoginThirdDto userBindLoginThirdDto) {
		return new RestResponse<>(oauth2Login.bind(userBindLoginThirdDto));
	}
}
