package com.dtyunxi.yundt.module.customer.rest;

import com.alibaba.fastjson.JSON;
import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.customer.api.customer.exception.CustomerBusinessRuntimeException;
import com.dtyunxi.yundt.cube.center.customer.api.customer.exception.CustomerExceptionCode;
import com.dtyunxi.yundt.cube.center.data.api.IAreaGroupApi;
import com.dtyunxi.yundt.cube.center.data.api.dto.request.AreaGroupReqDto;
import com.dtyunxi.yundt.cube.center.data.api.dto.response.AreaGroupDetailRespDto;
import com.dtyunxi.yundt.cube.center.data.api.dto.response.AreaGroupRespDto;
import com.dtyunxi.yundt.cube.center.data.api.dto.response.AreaGroupTreeRespDto;
import com.dtyunxi.yundt.cube.center.data.api.query.IAreaGroupQueryApi;
import com.dtyunxi.yundt.module.context.api.IContext;
import com.dtyunxi.yundt.module.customer.api.ICustomerService;
import com.dtyunxi.yundt.module.customer.api.dto.request.RegionReqDto;
import com.dtyunxi.yundt.module.customer.api.exception.CustomerModuleExceptionCode;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * 区域分组信息
 * @author xuhuaqiang
 * @version 1.0.0
 * @description
 * @date 2020/9/29
 */
@Api(tags = {"客商组件：区域分组信息服务"})
//@RestController
//@RequestMapping("/v1/basedata/area-group")
public class AreaGroupRest {

    @Resource
    private IAreaGroupApi areaGroupApi;

    @Resource
    private IAreaGroupQueryApi areaGroupQueryApi;

    @Resource
    private ICustomerService customerService;

    @Resource
    private IContext context;

    @PostMapping(value = "", produces = "application/json")
    @ApiOperation(
            value = "新增区域分组",
            notes = "新增区域分组"
    )
    RestResponse<Long> saveAreaGroup(@Validated @RequestBody AreaGroupReqDto areaGroupReqDto){
        areaGroupReqDto.setCode(UUID.randomUUID().toString().replaceAll("-", ""));
        areaGroupReqDto.setInstanceId(context.instanceId());
        areaGroupReqDto.setTenantId(context.tenantId());
        this.checkAreaGroupParam(null,areaGroupReqDto);
        Long data = areaGroupApi.saveAreaGroup(areaGroupReqDto).getData();
        return new RestResponse<>(data);
    }

    @DeleteMapping(value = "/{code}", produces = "application/json")
    @ApiOperation(
            value = "删除区域分组",
            notes = "根据区域分组ID删除数据"
    )
    @ApiImplicitParams({@ApiImplicitParam(
            name = "code",
            value = "区域分组CODE",
            dataType = "String",
            paramType = "path",
            required = true
    )})
    RestResponse<Void> removeByCode(@PathVariable("code") String code){
        //校验该区域分组下是否存在子区域分组
        List<AreaGroupRespDto> respDtoList = areaGroupQueryApi.queryByParentCode(code).getData();
        if (CollectionUtils.isNotEmpty(respDtoList)){
            throw new BizException(CustomerModuleExceptionCode.REGION_EXISTS_CHILD.getCode(),
                    CustomerModuleExceptionCode.REGION_EXISTS_CHILD.getMsg());
        }
        //校验该区域分组下是否存在客户
        Boolean isExist = customerService.isExistByRegionCode(code);
        if (isExist){
            throw new BizException(CustomerModuleExceptionCode.REGION_EXISTS_CUSTOMER.getCode(),
                    CustomerModuleExceptionCode.REGION_EXISTS_CUSTOMER.getMsg());
        }
        AreaGroupDetailRespDto respDto = areaGroupQueryApi.queryByCode(code).getData();
        areaGroupApi.removeById(respDto.getId());
        return RestResponse.VOID;
    }

    @PutMapping(value = "/{id}", produces = "application/json")
    @ApiOperation(
            value = "修改区域分组",
            notes = "修改区域分组"
    )
    @ApiImplicitParams({@ApiImplicitParam(
            name = "id",
            value = "区域分组ID",
            dataType = "Long",
            paramType = "path",
            required = true
    )})
    RestResponse<Void> modifyAreaGroup(@PathVariable("id") Long id, @Validated @RequestBody AreaGroupReqDto areaGroupReqDto){
        areaGroupReqDto.setInstanceId(context.instanceId());
        areaGroupReqDto.setTenantId(context.tenantId());
        this.checkAreaGroupParam(id,areaGroupReqDto);
        areaGroupApi.modifyAreaGroup(id,areaGroupReqDto);
        return RestResponse.VOID;
    }

    @GetMapping(value = "/page", produces = "application/json")
    @ApiOperation(
            value = "分页查询区域分组",
            notes = "分页查询区域分组"
    )
    RestResponse<PageInfo<AreaGroupRespDto>> queryByPage(@RequestParam("filter") String filter, @RequestParam(name = "pageNum",required = false,defaultValue = "1") Integer pageNum, @RequestParam(name = "pageSize",required = false,defaultValue = "10") Integer pageSize){
        PageInfo<AreaGroupRespDto> pageInfo = areaGroupQueryApi.queryByPage(filter, pageNum, pageSize).getData();
        return new RestResponse<>(pageInfo);
    }

    @GetMapping(value = "/code/{code}", produces = "application/json")
    @ApiOperation(
            value = "根据区域分组编码查询区域分组",
            notes = "根据区域分组编码查询区域分组"
    )
    @ApiImplicitParams({@ApiImplicitParam(
            name = "code",
            value = "区域分组编码",
            dataType = "String",
            paramType = "path",
            required = true
    )})
    RestResponse<AreaGroupDetailRespDto> queryByCode(@PathVariable("code") String code){
        AreaGroupDetailRespDto respDto = areaGroupQueryApi.queryByCode(code).getData();
        return new RestResponse<>(respDto);
    }

    @GetMapping(value = "/parent-code/{parentCode}", produces = "application/json")
    @ApiOperation(
            value = "根据区域分组父编码查询区域分组",
            notes = "根据区域分组父编码查询区域分组"
    )
    @ApiImplicitParams({@ApiImplicitParam(
            name = "parentCode",
            value = "区域分组父编码",
            dataType = "String",
            paramType = "path",
            required = true
    )})
    RestResponse<List<AreaGroupRespDto>> queryByParentCode(@PathVariable("parentCode") String parentCode){
        List<AreaGroupRespDto> respDtoList = areaGroupQueryApi.queryByParentCode(parentCode).getData();
        return new RestResponse<>(respDtoList);
    }

//    @GetMapping(value = "/area-code/{areaCodes}", produces = "application/json")
//    @ApiOperation(
//            value = "根据区域分组编码查询区域分组",
//            notes = "根据区域分组编码查询区域分组"
//    )
//    @ApiImplicitParams({@ApiImplicitParam(
//            name = "areaCodes",
//            value = "区域编码数组'',''",
//            dataType = "String",
//            paramType = "path",
//            required = true
//    )})
//    RestResponse<List<AreaGroupRespDto>> queryByAreaCodes(@PathVariable("areaCodes") String[] areaCodes){
//        List<AreaGroupRespDto> respDtoList = areaGroupQueryApi.queryByAreaCodes(areaCodes).getData();
//        return new RestResponse<>(respDtoList);
//    }

//    @GetMapping(value = "/by-area-codes", produces = "application/json")
//    @ApiOperation(
//            value = "v3根据区域分组编码查询区域分组",
//            notes = "根据区域分组编码查询区域分组"
//    )
//    @ApiImplicitParams({@ApiImplicitParam(
//            name = "areaCodes",
//            value = "区域编码数组'',''",
//            dataType = "String",
//            paramType = "query",
//            required = true
//    )})
//    RestResponse<List<AreaGroupRespDto>> queryByAreaCodesV3(@RequestParam("areaCodes") String[] areaCodes){
//        List<AreaGroupRespDto> respDtoList = areaGroupQueryApi.queryByAreaCodesV3(areaCodes).getData();
//        return new RestResponse<>(respDtoList);
//    }

    @GetMapping(value = "/tree", produces = "application/json")
    @ApiOperation(
            value = "查询区域分组树",
            notes = "查询区域分组树"
    )
    RestResponse<List<AreaGroupTreeRespDto>> queryForTree(){
        List<AreaGroupTreeRespDto> respDtoList = areaGroupQueryApi.queryForTree().getData();
        if (CollectionUtils.isNotEmpty(respDtoList)){
            Iterator<AreaGroupTreeRespDto> it = respDtoList.iterator();
            while (it.hasNext()) {
                AreaGroupTreeRespDto respDto = it.next();
                if ("0".equals(respDto.getCode())){
                    it.remove();
                }
            }
        }
        return new RestResponse<>(respDtoList);
    }

    @GetMapping(value = "/list/level/{level}", produces = "application/json")
    @ApiOperation(
            value = "按层级查询区域分组",
            notes = "按层级查询区域分组"
    )
    @ApiImplicitParams({@ApiImplicitParam(
            name = "level",
            value = "查询层级",
            dataType = "Long",
            paramType = "path",
            required = true
    )})
    RestResponse<List<AreaGroupRespDto>> queryByLevel(@PathVariable("level") Integer level){
        List<AreaGroupRespDto> respDtoList = areaGroupQueryApi.queryByLevel(level).getData();
        return new RestResponse<>(respDtoList);
    }

    private void checkAreaGroupParam(Long id,AreaGroupReqDto areaGroupReqDto) {
        RegionReqDto regionReqDto = new RegionReqDto();
        regionReqDto.setTenantId(areaGroupReqDto.getTenantId());
//        regionReqDto.setInstanceId(areaGroupReqDto.getInstanceId());
        regionReqDto.setName(areaGroupReqDto.getName());
        PageInfo<AreaGroupRespDto> pageInfo = areaGroupQueryApi.queryByPage(JSON.toJSONString(regionReqDto), 1, 10).getData();
        if (Objects.nonNull(pageInfo)){
            if (id == null && CollectionUtils.isNotEmpty(pageInfo.getList())) {
                throw new BizException(CustomerModuleExceptionCode.REGION_NAME_EXISTS.getCode(),
                        CustomerModuleExceptionCode.REGION_NAME_EXISTS.getMsg() );
            } else if (id != null && CollectionUtils.isNotEmpty(pageInfo.getList()) && !pageInfo.getList().get(0).getId().equals(id)) {
                throw new BizException(CustomerModuleExceptionCode.REGION_NAME_EXISTS.getCode(),
                        CustomerModuleExceptionCode.REGION_NAME_EXISTS.getMsg() );
            }
        }
    }
}
