package com.dtyunxi.yundt.module.customer.rest;

import com.dtyunxi.huieryun.core.util.JSON;
import com.dtyunxi.icommerce.utils.RestResponseHelper;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.tcbj.api.dto.response.ItBrandDto;
import com.dtyunxi.tcbj.api.query.IStoreSearchQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.constants.StoreDataOriginEnum;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.QueryStoreCreditAndNameReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.StoreQueryReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.response.QueryStoreCreditAndNameRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.response.StoreSellerGovernExcelRespDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.BizImportRespDto;
import com.dtyunxi.yundt.module.customer.biz.service.IStoreService;
import com.github.pagehelper.PageInfo;
import com.yx.tcbj.center.customer.api.IStoreApi;
import com.yx.tcbj.center.customer.api.dto.request.store.StoreReqDto;
import com.yx.tcbj.center.customer.api.dto.response.store.StoreRespDto;
import com.yx.tcbj.center.customer.api.query.IStoreQueryApi;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 将customer中心的接口提升至module层
 *
 * @author 李少武
 * @since 2021-12-09
 */
@Api(tags = {"客商组件：药店相关服务"})
@RestController
@RequestMapping("/v1/store")
public class StoreRest{

    @Resource
    private IStoreQueryApi storeQueryApi;
    @Resource
    private IStoreApi storeApi;
    @Resource
    private IStoreSearchQueryApi storeSearchQueryApi;

    @Resource
    private IStoreService storeService;

    /**
     * 大B端药店查询批量设置导入
     * @param file 文件
     * @return 结果
     */
    @PostMapping(value = "/excel", produces = "application/json")
    @ApiOperation(value = "药店数据Excel导入", notes = "药店数据Excel导入")
    public RestResponse<StoreSellerGovernExcelRespDto> excel(@RequestParam("file") MultipartFile file) {
        List<ItBrandDto> brandDtoList = RestResponseHelper.extractData(storeSearchQueryApi.queryBrands());
        Map<String,String> brandMap = brandDtoList.stream().collect(Collectors.toMap(e->e.getName(), t->t.getCode(),(e1, e2)->e1));
        return storeApi.excel(file, JSON.toJsonString(brandMap));
    }

    /**
     * 批量插入店铺数据
     *
     * @param reqDtos 入参
     */
    @PostMapping("/addStoreBatch")
    @ApiOperation(value = "批量插入店铺数据", notes = "批量插入店铺数据")
    RestResponse<Void> addStoreBatch(@RequestBody List<StoreReqDto> reqDtos){
        return storeApi.addStoreBatch(reqDtos);
    }

    /**
     * 批量插入店铺数据
     *
     * @param reqDtos 入参
     */
    @PostMapping("/addStore")
    @ApiOperation(value = "插入店铺数据", notes = "插入店铺数据")
    RestResponse<Void> addStoreBatch(@RequestBody StoreReqDto reqDtos){
        return storeApi.addStoreBatch(Arrays.asList(reqDtos));
    }

    /**
     * 批量修改企业信息
     *
     * @param reqDtos 入参
     */
    @PostMapping("/updateStoreBatch")
    @ApiOperation(value = "批量修改企业信息", notes = "批量修改企业信息")
    RestResponse<Void> updateStoreBatch(@RequestBody List<StoreReqDto> reqDtos){
        return storeApi.updateStoreBatch(reqDtos);
    }

    /**
     * 修改企业信息
     *
     * @param reqDto 企业信息入参
     */
    @PostMapping("/updateStore")
    @ApiOperation(value = "修改企业信息", notes = "修改企业信息")
    RestResponse<Void> updateStore(@RequestBody StoreReqDto reqDto){
        return storeApi.updateStore(reqDto);
    }

    /**
     * 药店分页查询（品牌端）
     * @param reqDto 请求参数对象
     * @return 分页数据
     */
    @PostMapping("/queryPage")
    @ApiOperation(value = "药店分页查询（品牌端）", notes = "药店分页查询（品牌端）")
    RestResponse<PageInfo<StoreRespDto>> queryPage(@RequestBody StoreQueryReqDto reqDto){
        return storeQueryApi.queryPage(reqDto);
    }

    @GetMapping("/queryStoreRespDtoByCreditNum")
    @ApiOperation(value = "根据信用代码查询药店详情", notes = "根据信用代码查询药店详情")
    RestResponse<StoreRespDto> queryStoreRespDtoByCreditNum(@RequestParam("creditNum")String creditNum){
        return storeQueryApi.queryStoreRespDtoExtByCreditNum(creditNum);
    }

    @PostMapping("/saveOrUpdateStore")
    @ApiOperation(value = "新增修改药店数据", notes = "新增修改药店数据")
    RestResponse<Void> saveOrUpdateStore(@RequestBody StoreReqDto reqDto){
        //insert情况 写入字段 药店表数据来源: 手动录入（包含导入）
        if(reqDto.getId() == null){
            reqDto.setDataOrigin(StoreDataOriginEnum.MANUALLY_ENTER.getCode());
        }
        return storeService.saveOrUpdateStore(reqDto);
    }

    @PostMapping("/saveOrUpdateStoreByExcel")
    @ApiOperation(value = "药店数据导入变更", notes = "药店数据导入变更")
    RestResponse<Void> saveOrUpdateStoreByExcel(@RequestParam("file") MultipartFile file){
        storeService.saveOrUpdateStoreByExcel(file);
        return RestResponse.SUCCESS;
    }

    /**
     * 注销药店模糊匹配查询
     */
    @PostMapping("/queryCreditAndNameInDisable")
    @ApiOperation(value = "注销药店模糊匹配查询", notes = "注销药店模糊匹配查询 查询字段storeName: 包括storeName与credit的模糊匹配; " +
            "返回字段storeName: 包括storeName与credit")
    RestResponse<List<QueryStoreCreditAndNameRespDto>> queryCreditAndNameInDisable(@RequestBody QueryStoreCreditAndNameReqDto reqDto){
        return storeQueryApi.queryCreditAndNameInDisable(reqDto);
    }


    /**
     * 注销药店模糊匹配查询
     */
    @PostMapping("/queryCreditAndNameNoDisable")
    @ApiOperation(value = "在业药店模糊匹配查询", notes = "在业药店模糊匹配查询 查询字段storeName: 包括storeName与credit的模糊匹配; " +
            "返回字段storeName: 包括storeName与credit")
    RestResponse<List<QueryStoreCreditAndNameRespDto>> queryCreditAndNameNoDisable(@RequestBody QueryStoreCreditAndNameReqDto reqDto){
        return storeQueryApi.queryCreditAndNameNoDisable(reqDto);
    }

}
