package com.dtyunxi.tcbj.api.biz.promotion;

import com.dtyunxi.tcbj.api.biz.promotion.dto.CustomerInfo;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionActivityQueryDto;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionExportRow;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.util.*;

@Service
public class PromotionExportService {

    @Resource
    private JdbcTemplate jdbcTemplate;

    /**
     * 根据orgId导出促销活动数据
     */
    public List<PromotionExportRow> exportByOrgId(PromotionActivityQueryDto req) {
        String orgId = req.getOrgId();
        // 1. 查询该orgId对应的所有活动ID，拼成逗号分隔字符串
        String activityIds = jdbcTemplate.queryForObject(
                "SELECT GROUP_CONCAT(id) FROM mk_activity WHERE organization_id = ?",
                new Object[]{orgId},
                String.class
        );

        if (activityIds == null || activityIds.isEmpty()) {
            // 活动ID为空，返回空列表
            return Collections.emptyList();
        }

        // 2. 调用存储过程，获取促销活动基础商品及换购商品数据
        List<PromotionExportRow> exportRows = callGenerateActivityExport(activityIds);

        // 3. 针对每个活动ID调用存储过程，获取对应的客户类型和客户明细，缓存结果
        Map<String, CustomerInfo> customerInfoMap = new HashMap<>();
        for (String activityId : activityIds.split(",")) {
            CustomerInfo info = callGetCustomerInfo(activityId);
            if (info != null) {
                customerInfoMap.put(activityId, info);
            }
        }

        // 4. 将客户类型和客户明细信息补充到每条促销商品数据中
        for (PromotionExportRow row : exportRows) {
            CustomerInfo info = customerInfoMap.get(row.getActivityCode());
            if (info != null) {
                row.setCustomerType(info.getCustomerType());
                row.setCustomerDetail(info.getCustomerDetail());
            } else {
                row.setCustomerType("");
                row.setCustomerDetail("");
            }
        }

        // 5. 返回最终拼接完整的促销活动导出数据
        return exportRows;
    }

    private List<PromotionExportRow> callGenerateActivityExport(String activityIds) {
        return jdbcTemplate.execute(
                (Connection con) -> {
                    CallableStatement cs = con.prepareCall("{CALL yx_prod_report.generate_activity_export(?)}");
                    cs.setString(1, activityIds);
                    return cs;
                },
                (CallableStatement cs) -> {
                    List<PromotionExportRow> list = new ArrayList<>();
                    boolean hasResults = cs.execute();
                    while (hasResults) {
                        try (ResultSet rs = cs.getResultSet()) {
                            while (rs.next()) {
                                PromotionExportRow row = new PromotionExportRow();
                                row.setIndex(rs.getInt("序号"));
                                row.setDivision(rs.getString("事业部"));
                                row.setActivityCode(rs.getString("活动编号"));
                                row.setActivityName(rs.getString("活动名称"));
                                row.setProductType(rs.getString("产品类型"));
                                row.setProductCode(rs.getString("产品编码"));
                                row.setProductName(rs.getString("产品名称"));
                                row.setPrice(rs.getString("建议零售价"));
                                row.setExchangePrice(rs.getString("换购价"));
                                row.setNotAloneBuy(rs.getString("不支持单独购买"));
                                row.setExchangeMethod(rs.getString("换购方式"));
                                row.setQuotaType(rs.getString("额度类型"));
                                row.setRatio(rs.getString("比例"));
                                row.setCustomerType("");
                                row.setCustomerDetail("");
                                row.setActivityStartTime(rs.getString("活动开始时间"));
                                row.setActivityEndTime(rs.getString("活动结束时间"));
                                row.setActivityStatus(rs.getString("活动状态"));
                                list.add(row);
                            }
                        }
                        hasResults = cs.getMoreResults();
                    }
                    return list;
                });
    }

    private CustomerInfo callGetCustomerInfo(String activityId) {
        return jdbcTemplate.execute(
                (Connection con) -> {
                    CallableStatement cs = con.prepareCall("{CALL yx_prod_report.proc_get_activity_customer_info_full(?)}");
                    cs.setString(1, activityId);
                    return cs;
                },
                (CallableStatement cs) -> {
                    CustomerInfo info = null;
                    boolean hasResults = cs.execute();
                    while (hasResults) {
                        try (ResultSet rs = cs.getResultSet()) {
                            if (rs.next()) {
                                info = new CustomerInfo();
                                info.setActivityId(activityId);
                                info.setCustomerType(rs.getString("customer_type"));
                                info.setCustomerDetail(rs.getString("customer_detail"));
                            }
                        }
                        hasResults = cs.getMoreResults();
                    }
                    return info;
                });
    }
}
