package com.dtyunxi.tcbj.api.biz.promotion;

import com.dtyunxi.tcbj.api.biz.promotion.dto.CustomerInfo;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionActivityQueryDto;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionExportRow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.sql.*;
import java.util.*;

@Service
public class PromotionExportService {

    private static final Logger log = LoggerFactory.getLogger(PromotionExportService.class);

    @Resource
    private JdbcTemplate jdbcTemplate;

    /**
     * 根据orgId导出促销活动数据
     */
    public List<PromotionExportRow> exportByOrgId(PromotionActivityQueryDto req) {
        String orgId = req.getOrgId();
        log.info("开始导出促销活动数据，orgId={}", orgId);

        // 1. 查询该orgId对应的所有活动ID，拼成逗号分隔字符串
        String activityIds = jdbcTemplate.queryForObject(
                "SELECT GROUP_CONCAT(id) FROM mk_activity WHERE organization_id = ?",
                new Object[]{orgId},
                String.class
        );

        log.info("查询到的活动ID列表: {}", activityIds);

        if (activityIds == null || activityIds.isEmpty()) {
            // 活动ID为空，返回空列表
            log.info("未找到匹配的活动ID，返回空列表");
            return Collections.emptyList();
        }

        // 2. 调用存储过程，获取促销活动基础商品及换购商品数据
        List<PromotionExportRow> exportRows = callGenerateActivityExport(activityIds);

        // 3. 针对每个活动ID调用存储过程，获取对应的客户类型和客户明细，缓存结果
        Map<String, CustomerInfo> customerInfoMap = new HashMap<>();
        for (String activityId : activityIds.split(",")) {
            CustomerInfo info = callGetCustomerInfo(activityId);
            if (info != null) {
                customerInfoMap.put(activityId, info);
            }
        }

        // 4. 将客户类型和客户明细信息补充到每条促销商品数据中
        for (PromotionExportRow row : exportRows) {
            CustomerInfo info = customerInfoMap.get(row.getActivityCode());
            if (info != null) {
                row.setCustomerType(info.getCustomerType());
                row.setCustomerDetail(info.getCustomerDetail());
            } else {
                row.setCustomerType("");
                row.setCustomerDetail("");
            }
        }

        // 5. 返回最终拼接完整的促销活动导出数据
        log.info("成功导出{}条促销活动数据", exportRows.size());
        return exportRows;
    }

    private List<PromotionExportRow> callGenerateActivityExport(String activityIds) {
        log.info("准备调用存储过程generate_activity_export，activityIds长度: {}", activityIds.length());

        // 临时测试：使用固定参数
        // activityIds = "1391100987105077507"; // 取消注释此行用于测试单个活动ID

        return jdbcTemplate.execute(
                (Connection con) -> {
                    log.info("获取数据库连接，连接对象: {}", con);
                    log.info("连接是否关闭: {}", con.isClosed());
                    log.info("连接是否有效: {}", con.isValid(5));

                    CallableStatement cs = con.prepareCall("{CALL yx_prod_report.generate_activity_export(?)}");
                    cs.setString(1, activityIds);
                    log.info("成功创建CallableStatement，准备执行存储过程");
                    return cs;
                },
                (CallableStatement cs) -> {
                    List<PromotionExportRow> list = new ArrayList<>();
                    try {
                        log.info("开始执行存储过程...");
                        boolean hasResults = cs.execute();
                        log.info("存储过程执行完成，是否有结果集: {}", hasResults);

                        int resultSetCount = 0;
                        while (hasResults) {
                            resultSetCount++;
                            log.info("处理第{}个结果集", resultSetCount);

                            try (ResultSet rs = cs.getResultSet()) {
                                if (rs == null) {
                                    log.warn("第{}个结果集为null", resultSetCount);
                                } else if (!rs.isBeforeFirst()) {
                                    log.info("第{}个结果集为空", resultSetCount);
                                } else {
                                    log.info("开始解析第{}个结果集", resultSetCount);
                                    int rowCount = 0;
                                    while (rs.next()) {
                                        rowCount++;
                                        PromotionExportRow row = new PromotionExportRow();

                                        // 使用安全的方式读取可能为null的字段
                                        Integer index = rs.getObject("序号", Integer.class);
                                        row.setIndex(index != null ? index : 0);

                                        row.setDivision(rs.getString("事业部"));
                                        row.setActivityCode(rs.getString("活动编号"));
                                        row.setActivityName(rs.getString("活动名称"));
                                        row.setProductType(rs.getString("产品类型"));
                                        row.setProductCode(rs.getString("产品编码"));
                                        row.setProductName(rs.getString("产品名称"));
                                        row.setPrice(rs.getString("建议零售价"));
                                        row.setExchangePrice(rs.getString("换购价"));
                                        row.setNotAloneBuy(rs.getString("不支持单独购买"));
                                        row.setExchangeMethod(rs.getString("换购方式"));
                                        row.setQuotaType(rs.getString("额度类型"));
                                        row.setRatio(rs.getString("比例"));
                                        row.setCustomerType("");
                                        row.setCustomerDetail("");
                                        row.setActivityStartTime(rs.getString("活动开始时间"));
                                        row.setActivityEndTime(rs.getString("活动结束时间"));
                                        row.setActivityStatus(rs.getString("活动状态"));

                                        list.add(row);
                                    }
                                    log.info("第{}个结果集解析完成，共{}行", resultSetCount, rowCount);
                                }
                            }
                            hasResults = cs.getMoreResults();
                            log.info("是否存在下一个结果集: {}", hasResults);
                        }

                        log.info("所有结果集处理完成，共处理{}个结果集", resultSetCount);
                        return list;
                    } catch (SQLException e) {
                        log.error("执行存储过程时发生SQL异常", e);
                        log.error("SQL状态码: {}", e.getSQLState());
                        log.error("错误码: {}", e.getErrorCode());
                        log.error("详细消息: {}", e.getMessage());
                        throw e;
                    } catch (Exception e) {
                        log.error("执行存储过程时发生未知异常", e);
                        throw new RuntimeException("执行存储过程时发生未知异常", e);
                    }
                });
    }

    private CustomerInfo callGetCustomerInfo(String activityId) {
        log.info("准备调用存储过程proc_get_activity_customer_info_full，activityId: {}", activityId);

        return jdbcTemplate.execute(
                (Connection con) -> {
                    CallableStatement cs = con.prepareCall("{CALL yx_prod_report.proc_get_activity_customer_info_full(?)}");
                    cs.setString(1, activityId);
                    return cs;
                },
                (CallableStatement cs) -> {
                    CustomerInfo info = null;
                    try {
                        boolean hasResults = cs.execute();
                        int resultSetCount = 0;

                        while (hasResults) {
                            resultSetCount++;
                            try (ResultSet rs = cs.getResultSet()) {
                                if (rs != null && rs.next()) {
                                    info = new CustomerInfo();
                                    info.setActivityId(activityId);
                                    info.setCustomerType(rs.getString("customer_type"));
                                    info.setCustomerDetail(rs.getString("customer_detail"));
                                    log.info("成功获取活动{}的客户信息", activityId);
                                }
                            }
                            hasResults = cs.getMoreResults();
                        }

                        if (info == null) {
                            log.warn("未找到活动{}的客户信息", activityId);
                        }

                        return info;
                    } catch (SQLException e) {
                        log.error("调用proc_get_activity_customer_info_full时发生SQL异常，activityId: {}", activityId, e);
                        throw e;
                    }
                });
    }
}