package com.dtyunxi.tcbj.api.biz.promotion;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.dtyunxi.tcbj.api.biz.promotion.dto.CustomerInfo;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionActivityQueryDto;
import com.dtyunxi.tcbj.api.biz.promotion.dto.PromotionExportRow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class PromotionExportService {

    private static final Logger log = LoggerFactory.getLogger(PromotionExportService.class);

    // 常量定义（与需求中模板ID对应）
    private static final String EXCHANGE_CONDITION_TEMPLATE_ID = "1196944771412880296";
    private static final String CUSTOMER_CONDITION_TEMPLATE_ID = "1196944771412880293";
    private static final String PRICE_TYPE_ID = "1"; // 建议零售价类型

    @Resource
    private JdbcTemplate jdbcTemplate;

    /**
     * 纯JDBC实现：根据orgId导出促销活动数据（严格匹配需求）
     */
    public List<PromotionExportRow> exportByOrgId(PromotionActivityQueryDto req) {
        String orgId = req.getOrgId();
        log.info("开始导出促销活动数据（纯JDBC），orgId={}", orgId);

        // 1. 查询该orgId对应的所有活动基础信息
        List<ActivityBase> activityBases = queryActivityBases(orgId);
        log.info("查询到{}个活动，开始处理商品数据", activityBases.size());

        if (activityBases.isEmpty()) {
            log.info("未找到匹配的活动，返回空列表");
            return Collections.emptyList();
        }

        // 2. 处理所有活动的商品数据（活动商品+换购商品）
        List<PromotionExportRow> allProductRows = new ArrayList<>();
        for (ActivityBase activity : activityBases) {
            log.debug("开始处理活动：{} - {}", activity.getActivityCode(), activity.getActivityName());

            // 2.1 处理活动商品
            List<PromotionExportRow> activityProductRows = handleActivityProducts(activity);
            log.debug("活动{}的活动商品处理完成，共{}条", activity.getActivityCode(), activityProductRows.size());

            // 2.2 处理换购商品
            List<PromotionExportRow> exchangeProductRows = handleExchangeProducts(activity);
            log.debug("活动{}的换购商品处理完成，共{}条", activity.getActivityCode(), exchangeProductRows.size());

            // 合并当前活动的商品数据
            allProductRows.addAll(activityProductRows);
            allProductRows.addAll(exchangeProductRows);
        }

        // 3. 生成序号（按活动编号分组递增）
        generateSerialNumber(allProductRows);
        log.debug("序号生成完成，共{}条数据", allProductRows.size());

        // 4. 处理客户信息并补充到商品数据中
        Map<String, CustomerInfo> customerInfoMap = queryAllCustomerInfo(activityBases);
        log.debug("客户信息查询完成，共{}个活动的客户信息", customerInfoMap.size());

        supplementCustomerInfo(allProductRows, customerInfoMap);
        log.debug("客户信息补充完成");

        // 5. 补充活动时间和状态
        supplementActivityTimeAndStatus(allProductRows, activityBases);
        log.debug("活动时间和状态补充完成");

        // 6. 排序：按序号、产品类型（活动商品在前）、产品编码
        sortExportRows(allProductRows);
        log.debug("数据排序完成");

        log.info("纯JDBC导出完成，共{}条数据", allProductRows.size());
        return allProductRows;
    }

    /**
     * 步骤1：查询活动基础信息
     */
    private List<ActivityBase> queryActivityBases(String orgId) {
        log.debug("开始查询组织{}的活动基础信息", orgId);
        String sql = "SELECT " +
                "id, activity_code, activity_name, organization_name, " +
                "select_type, begin_time, end_time, activity_status " +
                "FROM yx_prod_report.mk_activity " +
                "WHERE organization_id = ? AND dr = 0";

        List<ActivityBase> result = jdbcTemplate.query(sql, new Object[]{orgId}, (rs, rowNum) -> {
            ActivityBase base = new ActivityBase();
            base.setId(rs.getString("id"));
            base.setActivityCode(rs.getString("activity_code"));
            base.setActivityName(rs.getString("activity_name"));
            base.setOrganizationName(rs.getString("organization_name"));
            base.setSelectType(rs.getInt("select_type")); // 0-不限商品，1-指定商品
            base.setBeginTime(rs.getString("begin_time"));
            base.setEndTime(rs.getString("end_time"));
            base.setActivityStatus(rs.getString("activity_status"));
            log.trace("查询到活动：{} - {}", base.getActivityCode(), base.getActivityName());
            return base;
        });

        log.debug("查询到{}个活动基础信息", result.size());
        return result;
    }

    /**
     * 步骤2.1：处理活动商品（严格匹配需求）
     */
    private List<PromotionExportRow> handleActivityProducts(ActivityBase activity) {
        log.debug("开始处理活动{}的活动商品", activity.getActivityCode());
        List<PromotionExportRow> rows = new ArrayList<>();

        // 获取换购条件（用于提取换购方式、额度类型等）
        ExchangeCondition exchangeCondition = queryExchangeCondition(activity.getId());
        log.trace("活动{}的换购条件：{}", activity.getActivityCode(),
                exchangeCondition != null ? exchangeCondition.getExchangeMethod() : "无");

        if (activity.getSelectType() == 1) {
            // 1-指定商品：从关联表查询
            String sql = "SELECT " +
                    "sku.code AS product_code, " +
                    "item.name AS product_name, " +
                    "price.item_price AS suggest_price " +
                    "FROM yx_prod_report.mk_activity_item t2 " +
                    "LEFT JOIN yx_prod_report.it_item_sku sku ON sku.id = t2.sku_id " +
                    "LEFT JOIN yx_prod_report.it_item item ON item.id = sku.item_id " +
                    "LEFT JOIN yx_prod_report.pr_base_price_item price ON price.sku_id = sku.id AND price.type_id = ? " +
                    "WHERE t2.activity_id = ? AND t2.dr = 0";

            List<Map<String, Object>> productMaps = jdbcTemplate.queryForList(
                    sql, PRICE_TYPE_ID, activity.getId()
            );

            if (productMaps.isEmpty()) {
                log.warn("活动{}的select_type为1（指定商品），但未找到关联商品", activity.getActivityCode());
                // 无指定商品时，生成一条空数据
                PromotionExportRow row = buildBaseRow(activity, "活动商品");
                fillActivityProductFields(row, null, exchangeCondition);
                rows.add(row);
            } else {
                log.debug("活动{}找到{}个指定商品", activity.getActivityCode(), productMaps.size());
                // 有指定商品时，逐条生成
                for (Map<String, Object> map : productMaps) {
                    PromotionExportRow row = buildBaseRow(activity, "活动商品");
                    fillActivityProductFields(row, map, exchangeCondition);
                    rows.add(row);
                    log.trace("添加活动商品：{} - {}", row.getProductCode(), row.getProductName());
                }
            }
        } else {
            // 0-不限商品：固定显示"不限"
            log.debug("活动{}的select_type为0（不限商品）", activity.getActivityCode());
            PromotionExportRow row = buildBaseRow(activity, "活动商品");
            row.setProductCode("不限");
            row.setProductName("-");
            row.setPrice("-");
            row.setExchangePrice("-");
            row.setNotAloneBuy("-");
            row.setExchangeMethod(exchangeCondition != null ? exchangeCondition.getExchangeMethod() : "-");
            row.setQuotaType("-");
            row.setRatio("-");
            rows.add(row);
            log.trace("添加不限商品行");
        }

        return rows;
    }

    /**
     * 步骤2.2：处理换购商品（严格匹配需求）
     */
    private List<PromotionExportRow> handleExchangeProducts(ActivityBase activity) {
        log.debug("开始处理活动{}的换购商品", activity.getActivityCode());
        List<PromotionExportRow> rows = new ArrayList<>();

        // 1. 查询换购条件JSON
        String conditionSql = "SELECT condition_params FROM yx_prod_report.mk_condition " +
                "WHERE activity_id = ? AND condition_template_id = ?";
        String jsonStr = null;
        try {
            jsonStr = jdbcTemplate.queryForObject(
                    conditionSql, new Object[]{activity.getId(), EXCHANGE_CONDITION_TEMPLATE_ID},
                    String.class
            );
        } catch (Exception e) {
            log.warn("活动{}无换购条件配置，错误：{}", activity.getActivityCode(), e.getMessage());
            return rows;
        }

        if (!StringUtils.hasText(jsonStr)) {
            log.warn("活动{}无换购条件配置", activity.getActivityCode());
            return rows;
        }

        log.trace("活动{}的换购条件JSON长度：{}", activity.getActivityCode(), jsonStr.length());

        // 2. 解析JSON获取换购商品列表及公共参数（严格按需求JSON结构解析）
        try {
            JSONObject root = JSON.parseObject(jsonStr);
            JSONObject exchangeRule = root.getJSONObject("ExchangeCondition.rule");
            if (exchangeRule == null) {
                log.warn("活动{}换购条件JSON格式错误，找不到ExchangeCondition.rule节点", activity.getActivityCode());
                return rows;
            }

            // 公共参数（所有换购商品共享）
            String thresholdType = exchangeRule.getString("thresholdType");
            String exchangeMethod = parseExchangeMethod(thresholdType); // 修复：按需求0=免费，1=加价

            JSONObject excRuleContent = exchangeRule.getJSONArray("excRuleContents").getJSONObject(0);
            int discountUnit = excRuleContent.getIntValue("discountUnit"); // 换购商品用discountUnit
            String quotaType = discountUnit == 1 ? "金额" : (discountUnit == 2 ? "数量" : "-"); // 修复：1金额2数量

            int threshold = excRuleContent.getIntValue("threshold");
            int discount = excRuleContent.getIntValue("discount");
            String ratio = threshold + ":" + discount; // 修复：按threshold:discount拼接

            log.trace("活动{}的换购方式：{}，额度类型：{}，比例：{}",
                    activity.getActivityCode(), exchangeMethod, quotaType, ratio);

            // 换购商品列表
            JSONArray exchangeItems = excRuleContent.getJSONArray("exchangeItems");
            if (exchangeItems == null || exchangeItems.isEmpty()) {
                log.warn("活动{}无换购商品配置", activity.getActivityCode());
                return rows;
            }

            log.debug("活动{}找到{}个换购商品", activity.getActivityCode(), exchangeItems.size());

            // 3. 逐条处理换购商品
            for (int i = 0; i < exchangeItems.size(); i++) {
                JSONObject itemJson = exchangeItems.getJSONObject(i);
                String skuId = itemJson.getString("skuId");
                if (!StringUtils.hasText(skuId)) {
                    log.warn("活动{}的第{}个换购商品SKU ID为空，跳过", activity.getActivityCode(), i);
                    continue;
                }

                // 查询商品信息
                String productSql = "SELECT " +
                        "sku.code AS product_code, " +
                        "item.name AS product_name, " +
                        "price.item_price AS suggest_price " +
                        "FROM yx_prod_report.it_item_sku sku " +
                        "LEFT JOIN yx_prod_report.it_item item ON item.id = sku.item_id " +
                        "LEFT JOIN yx_prod_report.pr_base_price_item price ON price.sku_id = sku.id AND price.type_id = ? " +
                        "WHERE sku.id = ? AND item.dr = 0";

                List<Map<String, Object>> productMaps = jdbcTemplate.queryForList(
                        productSql, PRICE_TYPE_ID, skuId
                );

                // 构建行数据
                PromotionExportRow row = buildBaseRow(activity, "换购商品");
                if (!productMaps.isEmpty()) {
                    Map<String, Object> productMap = productMaps.get(0);
                    row.setProductCode((String) productMap.get("product_code"));
                    row.setProductName((String) productMap.get("product_name"));
                    row.setPrice(formatPrice(productMap.get("suggest_price")));
                    log.trace("换购商品{}信息已填充", row.getProductCode());
                } else {
                    log.warn("活动{}的换购商品SKU ID={}未找到商品信息", activity.getActivityCode(), skuId);
                    row.setProductCode("未知商品");
                    row.setProductName("未知商品");
                    row.setPrice("-");
                }

                // 填充换购特有字段（严格按需求）
                row.setExchangePrice(itemJson.getBigDecimal("exchangePrice").toString()); // 从exchangePrice提取
                row.setNotAloneBuy(itemJson.getIntValue("supportSingleBuy") == 1 ? "是" : "否"); // 1是0否
                row.setExchangeMethod(exchangeMethod);
                row.setQuotaType(quotaType);
                row.setRatio(ratio);

                rows.add(row);
                log.trace("添加换购商品：{} - {}", row.getProductCode(), row.getProductName());
            }
        } catch (Exception e) {
            log.error("解析活动{}的换购条件JSON失败", activity.getActivityCode(), e);
            return rows;
        }

        return rows;
    }

    /**
     * 步骤3：生成序号（按活动编号分组递增，匹配需求）
     */
    private void generateSerialNumber(List<PromotionExportRow> rows) {
        log.debug("开始生成序号，共{}条数据", rows.size());

        // 按活动编号分组并排序
        Map<String, List<PromotionExportRow>> activityGroups = rows.stream()
                .collect(Collectors.groupingBy(PromotionExportRow::getActivityCode));
        List<String> sortedActivityCodes = new ArrayList<>(activityGroups.keySet());
        sortedActivityCodes.sort(Comparator.naturalOrder());

        log.debug("活动分组完成，共{}个活动", sortedActivityCodes.size());

        // 分配序号（按活动递增）
        int serialNumber = 1;
        for (String activityCode : sortedActivityCodes) {
            List<PromotionExportRow> groupRows = activityGroups.get(activityCode);
            for (PromotionExportRow row : groupRows) {
                row.setIndex(serialNumber);
            }
            log.trace("活动{}分配序号：{}", activityCode, serialNumber);
            serialNumber++;
        }

        log.debug("序号生成完成，最大序号：{}", serialNumber - 1);
    }

    /**
     * 步骤4：查询所有活动的客户信息（严格匹配需求）
     */
    private Map<String, CustomerInfo> queryAllCustomerInfo(List<ActivityBase> activities) {
        Map<String, CustomerInfo> resultMap = new HashMap<>();
        for (ActivityBase activity : activities) {
            CustomerInfo info = queryCustomerInfo(activity.getId());
            resultMap.put(activity.getActivityCode(), info);
            log.trace("活动{}的客户信息：{} - {}", activity.getActivityCode(),
                    info.getCustomerType(), info.getCustomerDetail());
        }
        return resultMap;
    }

    /**
     * 查询单个活动的客户信息（修复表名：用cs_前缀而非bi_）
     */
    private CustomerInfo queryCustomerInfo(String activityId) {
        log.debug("查询活动{}的客户信息", activityId);

        // 1. 查询客户条件JSON
        String sql = "SELECT condition_params FROM yx_prod_report.mk_condition " +
                "WHERE activity_id = ? AND condition_template_id = ?";
        String jsonStr = null;
        try {
            jsonStr = jdbcTemplate.queryForObject(
                    sql, new Object[]{activityId, CUSTOMER_CONDITION_TEMPLATE_ID},
                    String.class
            );
        } catch (Exception e) {
            log.warn("活动{}无客户条件配置，错误：{}", activityId, e.getMessage());
            return new CustomerInfo(activityId, "不限客户", "");
        }

        if (!StringUtils.hasText(jsonStr)) {
            log.debug("活动{}无客户条件配置", activityId);
            return new CustomerInfo(activityId, "不限客户", "");
        }

        log.trace("活动{}的客户条件JSON长度：{}", activityId, jsonStr.length());

        // 2. 解析JSON（严格按需求type=1/2处理）
        try {
            JSONObject root = JSON.parseObject(jsonStr);
            int customerType = root.getIntValue("TargetCustomerCondition.type");
            JSONObject customerValue = root.getJSONObject("TargetCustomerCondition.value");

            log.trace("活动{}的客户类型：{}", activityId, customerType);

            if (customerType == 1) {
                // 1-指定类型：客户类型+区域+剔除客户（修复：按需求拼接）
                String typeStr = "指定类型";
                StringBuilder detailSb = new StringBuilder();

                // 客户类型（修复：查询cs_customer_type）
                JSONArray typeIds = customerValue.getJSONArray("customerTypeIds");
                if (typeIds != null && !typeIds.isEmpty()) {
                    String typeNames = queryCustomerTypeNames(typeIds);
                    if (StringUtils.hasText(typeNames)) {
                        detailSb.append("客户类型: ").append(typeNames).append("\n");
                        log.trace("活动{}的客户类型：{}", activityId, typeNames);
                    }
                }

                // 客户区域（修复：查询cs_customer_area）
                JSONArray areaCodes = customerValue.getJSONArray("customerAreaCodes");
                if (areaCodes != null && !areaCodes.isEmpty()) {
                    String areaNames = queryCustomerAreaNames(areaCodes);
                    if (StringUtils.hasText(areaNames)) {
                        detailSb.append("客户区域: ").append(areaNames).append("\n");
                        log.trace("活动{}的客户区域：{}", activityId, areaNames);
                    }
                }

                // 剔除客户（修复：查询cs_customer）
                JSONArray blackIds = customerValue.getJSONArray("blackCustomerIds");
                if (blackIds != null && !blackIds.isEmpty()) {
                    String blackNames = queryCustomerNames(blackIds);
                    if (StringUtils.hasText(blackNames)) {
                        detailSb.append("剔除客户: ").append(blackNames).append("\n");
                        log.trace("活动{}的剔除客户：{}", activityId, blackNames);
                    }
                }

                String detail = detailSb.toString().trim();
                log.debug("活动{}的客户明细：{}", activityId, detail);
                return new CustomerInfo(activityId, typeStr, detail);
            } else if (customerType == 2) {
                // 2-指定客户（修复：按需求取customerIds）
                String typeStr = "指定客户";
                JSONArray customerIds = customerValue.getJSONArray("customerIds");
                String customerNames = customerIds != null ? queryCustomerNames(customerIds) : "";
                log.debug("活动{}的指定客户：{}", activityId, customerNames);
                return new CustomerInfo(activityId, typeStr, customerNames);
            } else {
                // 其他-不限客户
                log.debug("活动{}的客户类型为其他，返回不限客户", activityId);
                return new CustomerInfo(activityId, "不限客户", "");
            }
        } catch (Exception e) {
            log.error("解析活动{}的客户条件JSON失败", activityId, e);
            return new CustomerInfo(activityId, "不限客户", "");
        }
    }

    /**
     * 步骤5：补充活动时间和状态（匹配需求）
     */
    private void supplementActivityTimeAndStatus(List<PromotionExportRow> rows, List<ActivityBase> activityBases) {
        log.debug("开始补充活动时间和状态，共{}条数据", rows.size());

        // 构建活动ID到活动基础信息的映射
        Map<String, ActivityBase> activityMap = activityBases.stream()
                .collect(Collectors.toMap(ActivityBase::getId, activity -> activity));

        // 补充时间和状态信息
        for (PromotionExportRow row : rows) {
            ActivityBase activity = activityMap.get(row.getActivityId());
            if (activity != null) {
                row.setActivityStartTime(activity.getBeginTime());
                row.setActivityEndTime(activity.getEndTime());
                row.setActivityStatus(parseActivityStatus(activity.getActivityStatus())); // 修复：状态转换
                log.trace("为活动{}补充时间和状态信息", row.getActivityCode());
            } else {
                log.warn("找不到活动ID={}的基础信息", row.getActivityId());
                row.setActivityStartTime("-");
                row.setActivityEndTime("-");
                row.setActivityStatus("-");
            }
        }

        log.debug("活动时间和状态补充完成");
    }

    /**
     * 步骤6：排序（活动商品在前，匹配需求）
     */
    private void sortExportRows(List<PromotionExportRow> rows) {
        log.debug("开始排序，共{}条数据", rows.size());

        // 修复：活动商品排在换购商品前面
        rows.sort(Comparator.comparingInt(PromotionExportRow::getIndex)
                .thenComparing(row -> "活动商品".equals(row.getProductType()) ? 0 : 1) // 活动商品在前
                .thenComparing(PromotionExportRow::getProductCode));

        log.debug("数据排序完成");
    }

    /**
     * 构建基础行数据
     */
    private PromotionExportRow buildBaseRow(ActivityBase activity, String productType) {
        PromotionExportRow row = new PromotionExportRow();
        row.setActivityId(activity.getId());
        row.setDivision(activity.getOrganizationName()); // 事业部=organization_name
        row.setActivityCode(activity.getActivityCode());
        row.setActivityName(activity.getActivityName());
        row.setProductType(productType);
        // 初始值设置
        row.setCustomerType("");
        row.setCustomerDetail("");
        row.setActivityStartTime("");
        row.setActivityEndTime("");
        row.setActivityStatus("");
        return row;
    }

    /**
     * 填充活动商品字段（修复：活动商品用unit）
     */
    private void fillActivityProductFields(PromotionExportRow row, Map<String, Object> productMap, ExchangeCondition exchangeCondition) {
        if (productMap != null) {
            row.setProductCode((String) productMap.get("product_code"));
            row.setProductName((String) productMap.get("product_name"));
            row.setPrice(formatPrice(productMap.get("suggest_price")));
        } else {
            row.setProductCode("-");
            row.setProductName("-");
            row.setPrice("-");
        }
        row.setExchangePrice("-");
        row.setNotAloneBuy("-");
        row.setExchangeMethod(exchangeCondition != null ? exchangeCondition.getExchangeMethod() : "-");
        row.setQuotaType(exchangeCondition != null ? exchangeCondition.getQuotaType() : "-"); // 活动商品用unit
        row.setRatio(exchangeCondition != null ? exchangeCondition.getRatio() : "-");
    }

    /**
     * 查询换购条件（用于活动商品行的换购方式等字段，修复：活动商品用unit）
     */
    private ExchangeCondition queryExchangeCondition(String activityId) {
        log.debug("查询活动{}的换购条件", activityId);

        String sql = "SELECT condition_params FROM yx_prod_report.mk_condition " +
                "WHERE activity_id = ? AND condition_template_id = ?";
        String jsonStr = null;
        try {
            jsonStr = jdbcTemplate.queryForObject(
                    sql, new Object[]{activityId, EXCHANGE_CONDITION_TEMPLATE_ID},
                    String.class
            );
        } catch (Exception e) {
            log.warn("活动{}无换购条件配置，错误：{}", activityId, e.getMessage());
            return null;
        }

        if (!StringUtils.hasText(jsonStr)) {
            log.debug("活动{}无换购条件配置", activityId);
            return null;
        }

        log.trace("活动{}的换购条件JSON长度：{}", activityId, jsonStr.length());

        try {
            JSONObject root = JSON.parseObject(jsonStr);
            JSONObject exchangeRule = root.getJSONObject("ExchangeCondition.rule");
            if (exchangeRule == null) {
                log.warn("活动{}换购条件JSON格式错误，找不到ExchangeCondition.rule节点", activityId);
                return null;
            }

            // 换购方式（thresholdType）
            String thresholdType = exchangeRule.getString("thresholdType");
            String exchangeMethod = parseExchangeMethod(thresholdType);

            // 活动商品额度类型（unit）
            JSONObject excRuleContent = exchangeRule.getJSONArray("excRuleContents").getJSONObject(0);
            int unit = excRuleContent.getIntValue("unit");
            String quotaType = unit == 1 ? "金额" : (unit == 2 ? "数量" : "-");

            // 比例（threshold:discount）
            int threshold = excRuleContent.getIntValue("threshold");
            int discount = excRuleContent.getIntValue("discount");
            String ratio = threshold + ":" + discount;

            ExchangeCondition condition = new ExchangeCondition();
            condition.setExchangeMethod(exchangeMethod);
            condition.setQuotaType(quotaType);
            condition.setRatio(ratio);

            log.trace("活动{}的换购方式：{}，额度类型：{}", activityId, exchangeMethod, quotaType);
            return condition;
        } catch (Exception e) {
            log.error("解析活动{}的换购条件JSON失败", activityId, e);
            return null;
        }
    }

    /**
     * 解析换购方式（修复：严格按需求0=免费换购，1=加价换购）
     */
    private String parseExchangeMethod(String thresholdType) {
        if ("0".equals(thresholdType)) {
            return "免费换购";
        } else if ("1".equals(thresholdType)) {
            return "加价换购";
        } else {
            return "-";
        }
    }

    /**
     * 解析活动状态（匹配存储过程逻辑）
     */
    private String parseActivityStatus(String status) {
        Map<String, String> statusMap = new HashMap<>();
        statusMap.put("NEW", "新建");
        statusMap.put("READY", "就绪");
        statusMap.put("ACTIVATE", "活动中");
        statusMap.put("PAUSE", "暂停");
        statusMap.put("FINISH", "活动结束");
        return statusMap.getOrDefault(status, "未知状态");
    }

    /**
     * 格式化价格（null显示为-）
     */
    private String formatPrice(Object priceObj) {
        if (priceObj == null) {
            return "-";
        }
        if (priceObj instanceof BigDecimal) {
            return ((BigDecimal) priceObj).toString();
        } else {
            return priceObj.toString();
        }
    }

    /**
     * 查询客户类型名称（修复：查询cs_customer_type）
     */
    private String queryCustomerTypeNames(JSONArray typeIds) {
        if (typeIds == null || typeIds.isEmpty()) {
            return "";
        }

        List<String> idList = new ArrayList<>();
        for (int i = 0; i < typeIds.size(); i++) {
            idList.add(typeIds.getString(i));
        }

        String placeholders = String.join(",", Collections.nCopies(idList.size(), "?"));
        String sql = "SELECT name FROM cs_customer_type WHERE id IN (" + placeholders + ")";

        List<String> names = jdbcTemplate.query(sql, idList.toArray(),
                (rs, rowNum) -> rs.getString("name"));

        return String.join("、", names);
    }

    /**
     * 查询客户区域名称（修复：查询cs_customer_area）
     */
    private String queryCustomerAreaNames(JSONArray areaCodes) {
        if (areaCodes == null || areaCodes.isEmpty()) {
            return "";
        }

        List<String> codeList = new ArrayList<>();
        for (int i = 0; i < areaCodes.size(); i++) {
            codeList.add(areaCodes.getString(i));
        }

        String placeholders = String.join(",", Collections.nCopies(codeList.size(), "?"));
        String sql = "SELECT name FROM cs_customer_area WHERE code IN (" + placeholders + ")";

        List<String> names = jdbcTemplate.query(sql, codeList.toArray(),
                (rs, rowNum) -> rs.getString("name"));

        return String.join("、", names);
    }

    /**
     * 查询客户名称（修复：查询cs_customer）
     */
    private String queryCustomerNames(JSONArray customerIds) {
        if (customerIds == null || customerIds.isEmpty()) {
            return "";
        }

        List<String> idList = new ArrayList<>();
        for (int i = 0; i < customerIds.size(); i++) {
            idList.add(customerIds.getString(i));
        }

        String placeholders = String.join(",", Collections.nCopies(idList.size(), "?"));
        String sql = "SELECT name FROM cs_customer WHERE id IN (" + placeholders + ")";

        List<String> names = jdbcTemplate.query(sql, idList.toArray(),
                (rs, rowNum) -> rs.getString("name"));

        return String.join("、", names);
    }

    /**
     * 补充客户信息到行数据
     */
    private void supplementCustomerInfo(List<PromotionExportRow> rows, Map<String, CustomerInfo> customerInfoMap) {
        log.debug("开始补充客户信息，共{}条数据", rows.size());

        for (PromotionExportRow row : rows) {
            CustomerInfo info = customerInfoMap.get(row.getActivityCode());
            if (info != null) {
                row.setCustomerType(info.getCustomerType());
                row.setCustomerDetail(info.getCustomerDetail());
                log.trace("为活动{}补充客户信息", row.getActivityCode());
            } else {
                log.warn("找不到活动{}的客户信息", row.getActivityCode());
                row.setCustomerType("");
                row.setCustomerDetail("");
            }
        }

        log.debug("客户信息补充完成");
    }

    /**
     * 活动基础信息内部类
     */
    private static class ActivityBase {
        private String id;
        private String activityCode;
        private String activityName;
        private String organizationName; // 事业部
        private int selectType; // 0-不限商品，1-指定商品
        private String beginTime; // 活动开始时间
        private String endTime; // 活动结束时间
        private String activityStatus; // 活动状态

        // Getters and Setters
        public String getId() { return id; }
        public void setId(String id) { this.id = id; }
        public String getActivityCode() { return activityCode; }
        public void setActivityCode(String activityCode) { this.activityCode = activityCode; }
        public String getActivityName() { return activityName; }
        public void setActivityName(String activityName) { this.activityName = activityName; }
        public String getOrganizationName() { return organizationName; }
        public void setOrganizationName(String organizationName) { this.organizationName = organizationName; }
        public int getSelectType() { return selectType; }
        public void setSelectType(int selectType) { this.selectType = selectType; }
        public String getBeginTime() { return beginTime; }
        public void setBeginTime(String beginTime) { this.beginTime = beginTime; }
        public String getEndTime() { return endTime; }
        public void setEndTime(String endTime) { this.endTime = endTime; }
        public String getActivityStatus() { return activityStatus; }
        public void setActivityStatus(String activityStatus) { this.activityStatus = activityStatus; }
    }

    /**
     * 换购条件内部类（用于活动商品行）
     */
    private static class ExchangeCondition {
        private String exchangeMethod; // 换购方式
        private String quotaType; // 额度类型（活动商品用unit）
        private String ratio; // 比例（threshold:discount）

        // Getters and Setters
        public String getExchangeMethod() { return exchangeMethod; }
        public void setExchangeMethod(String exchangeMethod) { this.exchangeMethod = exchangeMethod; }
        public String getQuotaType() { return quotaType; }
        public void setQuotaType(String quotaType) { this.quotaType = quotaType; }
        public String getRatio() { return ratio; }
        public void setRatio(String ratio) { this.ratio = ratio; }
    }
}