/**
 * @(#)IWeixinServiceImpl.java 1.0 2020/2/28
 * <p>
 * Copyright (c) 2016, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.module.customer.biz.impl.user;

import com.alibaba.fastjson.JSONObject;
import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.cube.utils.bean.ObjectHelper;
import com.dtyunxi.huieryun.cache.api.ICacheService;
import com.dtyunxi.huieryun.oss.api.IObjectStorageService;
import com.dtyunxi.huieryun.oss.vo.OssRegistryVo;
import com.dtyunxi.util.HttpUtil;
import com.dtyunxi.util.MD5Util;
import com.dtyunxi.yundt.module.customer.api.user.IWeixinService;
import com.dtyunxi.yundt.module.customer.api.user.dto.response.WxUserInfoDto;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Service
public class IWeixinServiceImpl implements IWeixinService {

    private static final Logger logger = LoggerFactory.getLogger(IWeixinServiceImpl.class);

    @Resource
    private ICacheService cacheService;

    /**
     * 获取调用接口access_token地址
     */
    private final static String ACCESS_TOKEN_URL = "https://api.weixin.qq.com/cgi-bin/token";

    /**
     * 获取ticket地址
     */
    private final static String TICKET_URL = "https://api.weixin.qq.com/cgi-bin/ticket/getticket";

    /**
     * 微信授权码code地址
     */
    private final static String CODE_URL = "https://open.weixin.qq.com/connect/oauth2/authorize";


    /**
     * 获取网页授权access_token地址
     */
    private final static String AUTH_ACCESS_TOKEN_URL = "https://api.weixin.qq.com/sns/oauth2/access_token";


    /**
     * 获取用户心信息地址
     */
    private final static String USERINFO_URL = "https://api.weixin.qq.com/sns/userinfo";

    /**
     * 有效时间秒，微信默认7200秒有效，设置为7150
     */
    private final static int EFFECT_TIME_SEC = 7150;


    /**
     * 公众号appId
     */
    @Value("${appid:}")
    private String appid;

    /**
     * 公众号appSecret
     */
    @Value("${appsecret:}")
    private String secret;


    @Value("${mini.appid:}")
    private String miniAppid;

    @Value("${mini.appsecret:}")
    private String miniAppsecret;

    /**
     * 接受微信code地址
     */
    @Value("${systemCodeUri:}")
    private String systemCodeUri;


    @Autowired(required = false)
    private RestTemplateBuilder builder;

    private RestTemplate template;

    @Autowired
    private IObjectStorageService ossService;

    @Autowired
    private OssRegistryVo ossRegistryVo;


    @PostConstruct
    public void init() {
        if (null != builder) {
            template = builder.setConnectTimeout(Duration.ofMillis(1000)).setReadTimeout(Duration.ofMillis(1000)).rootUri("https://api.weixin.qq.com/").build();
        } else {
            template = null;
        }
    }

    @Override
    public HashMap<String, String> jsSdkSign(String url) {
        HashMap<String, String> jssdk = new HashMap<String, String>();
        try {
            String nonce_str = this.createNonceStr();
            String timestamp = String.valueOf(System.currentTimeMillis());
            String jsapi_ticket = this.getJsApiTicket();
            // 注意这里参数名必须全部小写，且必须有序
            String string1 = "jsapi_ticket=" + jsapi_ticket + "&noncestr=" + nonce_str
                    + "&timestamp=" + timestamp + "&url=" + url;
            MessageDigest crypt = MessageDigest.getInstance("SHA-1");
            crypt.reset();
            crypt.update(string1.getBytes("UTF-8"));
            String signature = byteToHex(crypt.digest());

            jssdk.put("appId", appid);
            jssdk.put("timestamp", timestamp);
            jssdk.put("nonceStr", nonce_str);
            jssdk.put("signature", signature);
        } catch (UnsupportedEncodingException e) {
            throw new BizException("获取微信配置信息失败");
        } catch (BizException e) {
            throw new BizException("获取微信配置信息失败:" + e.getMessage());
        } catch (NoSuchAlgorithmException e) {
            throw new BizException("获取微信配置信息失败");
        }

        return jssdk;
    }

    private String byteToHex(final byte[] hash) {
        Formatter formatter = new Formatter();
        for (byte b : hash) {
            formatter.format("%02x", b);
        }
        String result = formatter.toString();
        formatter.close();
        return result;
    }

    private String createNonceStr() {
        return UUID.randomUUID().toString();
    }

    @Override
    public String getAccessToken(boolean isMini) {
        String tokenKey = "access_token" + (isMini ? "_mini":"");

        String accessToken = cacheService.getCache(tokenKey, String.class);
        String tAppId = appid;
        String tAppSecret = secret;
        if (isMini) {
            tAppId = miniAppid;
            tAppSecret = miniAppsecret;
        }

        if (StringUtils.isBlank(accessToken)) {
            // 获取access_token
            String param = "grant_type=client_credential&appid=" + tAppId + "&secret=" + tAppSecret;
            String jstoken = HttpUtil.sendGet(ACCESS_TOKEN_URL, param);
            accessToken = JSONObject.parseObject(jstoken).getString("access_token");
            if (null == accessToken) {
                throw new BizException(jstoken);
            }

            cacheService.setCache(tokenKey, accessToken, EFFECT_TIME_SEC);
        }

        logger.info("=============access_token={}", accessToken);

        return accessToken;
    }

    @Override
    public String getJsApiTicket() {
        String jsapiTicket = cacheService.getCache("jsapi_ticket", String.class);

        if (StringUtils.isBlank(jsapiTicket)) {
            String accessToken = this.getAccessToken(false);

            //获取jsticket
            String ticketParam = "access_token=" + accessToken + "&type=jsapi";
            String jsticket = HttpUtil.sendGet(TICKET_URL, ticketParam);
            jsapiTicket = JSONObject.parseObject(jsticket).getString("ticket");
            if (null == jsapiTicket) {
                throw new BizException(jsticket);
            }

            cacheService.setCache("jsapi_ticket", jsapiTicket, EFFECT_TIME_SEC);
        }

        logger.info("=============jsapi_ticket={}", jsapiTicket);

        return jsapiTicket;
    }


    @Override
    public String getCodeUri(String redirectUri) {
        String state = UUID.randomUUID().toString();
        cacheService.add(state, redirectUri);

        String middleRedirectUri = URLEncoder.encode(systemCodeUri);

        String param = "appid=" + appid + "&redirect_uri=" + middleRedirectUri + "&response_type=code&scope" +
                "=snsapi_base" +
                "&state=" + state + "#wechat_redirect";

        String uri = CODE_URL + "?" + param;

        return uri;
    }

    @Override
    public String getOpenId(String code) {
        String param = "appid=" + appid + "&secret=" + secret + "&code=" + code + "&grant_type=authorization_code";
        String accessTokenStr = HttpUtil.sendGet(AUTH_ACCESS_TOKEN_URL, param);

        logger.info("code={},响应结果：{}", code, accessTokenStr);

        String openId = JSONObject.parseObject(accessTokenStr).getString("openid");
        if (StringUtils.isBlank(openId)) {
            throw new BizException("invalid code");
        }

        return openId;
    }

    @Override
    public WxUserInfoDto getUserInfo(String accessToken, String openId) {
        String param = "access_token=" + accessToken + "&openid=" + openId + "&lang=zh_CN";
        String userInfo = HttpUtil.sendGet(USERINFO_URL, param);
        WxUserInfoDto userInfoDto = ObjectHelper.Json2Bean(userInfo, WxUserInfoDto.class);
        logger.info("微信用户信息={}", userInfo);
        return userInfoDto;
    }

    /**
     * 获取小程序二维码
     */
    private final static String WXACODE = "/wxa/getwxacode?access_token={accessToken}";

    @Override
    public String getWxaCode(String path, int width) {
        // 二维码最小宽度
        if (width < 280) {
            width = 280;
        }
        // 二维码最大宽度
        if (width > 1280) {
            width = 1280;
        }

        String key = MD5Util.getMd5ByString(path + width);
        String truePath = cacheService.getCache(key, String.class);
        if (org.springframework.util.StringUtils.isEmpty(truePath)) {
            String accessToken = this.getAccessToken(true);
            Map<String, String> params = new HashMap<>();
            params.put("accessToken", accessToken);

            Map<String, String> body = new HashMap<>();
            body.put("path", path);
            body.put("width", width + "");
            byte[] images = template.postForObject(WXACODE, body, byte[].class, params);
            String fileName = UUID.randomUUID().toString() + ".png";
            ossService.put(ossRegistryVo.getBucketName(), fileName, images);
            truePath = ossRegistryVo.getEndpoint() + "/" + fileName;
            cacheService.setCache(key, truePath, 7 * 24 * 3600);
        }
        return truePath;
    }

}
