package com.dtyunxi.yundt.module.customer.biz.impl.user;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;
import com.alibaba.fastjson.JSON;
import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.cube.utils.DateUtil;
import com.dtyunxi.cube.utils.bean.CubeBeanUtils;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.customer.api.dto.entity.NacosPropertiesOrgEntity;
import com.dtyunxi.yundt.cube.center.user.api.dto.entity.ImportBaseResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeRoleReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.tcbj.EmployeeImportReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.RoleExpandRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserOrgizationRespDto;
import com.dtyunxi.yundt.cube.center.user.api.enums.YesOrNoEnum;
import com.dtyunxi.yundt.cube.center.user.api.util.AssertUtil;
import com.dtyunxi.yundt.cube.center.user.api.util.BeanPropertyNullUtil;
import com.dtyunxi.yundt.module.context.api.IContext;
import com.dtyunxi.yundt.module.customer.api.dto.request.user.EmployeeExpandReqDto;
import com.dtyunxi.yundt.module.customer.api.user.IB2bEmployeeExpandService;
import com.dtyunxi.yundt.module.customer.api.user.IEmployeeExpand2Service;
import com.dtyunxi.yundt.module.customer.api.vo.B2bEmployeeExcelVo;
import com.dtyunxi.yundt.module.customer.biz.util.CommonUtil;
import com.dtyunxi.yundt.module.customer.biz.util.EasyPoiExportUtil;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


/**
 * @author liangfa
 * @Description 人员管理接口实现
 * @date 2022-01-14 14:06
 * @since 1.0.0
 */
@Service
public class B2bEmployeeExpandServiceImpl implements IB2bEmployeeExpandService {

    private static final Logger logger = LoggerFactory.getLogger(B2bEmployeeExpandServiceImpl.class);

    @Resource
    private IEmployeeExpand2Service employeeExpand2Service;
    @Resource
    private NacosPropertiesOrgEntity nacosPropertiesOrgEntity;
    @Resource
    private HttpServletRequest request;
    @Resource
    private IContext context;

    /**
     * 获取请求头的组织ID
     *
     * @return 组织ID
     */
    public String headerOrgId() {
        String organizationId = request.getHeader("yes-req-cus-b2b-organizationId");
        logger.info("获取请求头组织ID：{}", organizationId);
        return organizationId;
    }
    /**
     * 根据实例ID查询TenantId
     *
     * @return
     */
    public Long tenantId() {
        Long tenantId = context.tenantId();
        logger.info("tenantId >>> {}", tenantId);
        return tenantId;
    }


    /**
     * b2b人员导入
     * @return
     */
    @Override
    public RestResponse<ImportBaseResponse> importB2bExcelEmployee(EmployeeImportReqDto importReqDto) {
        logger.info("importB2bExcelEmployee==>{}",JSON.toJSONString(importReqDto));
        String fileUrl = importReqDto.getFileUrl();
        AssertUtil.assertNotBlank(fileUrl, "导入的文件路径为空");
        Long instanceId = importReqDto.getInstanceId();
        AssertUtil.assertNotNull(instanceId, "导入的实例ID为空");
        String orgIdStr = this.headerOrgId();
        //orgIdStr = "1288861123298304011";
        logger.info("请求头组织ID：{}", orgIdStr);
        // 再次获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(orgIdStr)){
            orgIdStr = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", orgIdStr);
        }
        AssertUtil.assertNotBlank(orgIdStr, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        Long orgId = Long.valueOf(orgIdStr);
        /* 设置默认结果 */
        ImportBaseResponse response = new ImportBaseResponse();
        /* 处理文件，读取数据 */
        ImportParams importParams = new ImportParams();
        importParams.setHeadRows(1);// 表格标题行数,默认0
        importParams.setTitleRows(0);//表头行数,默认1
        importParams.setKeyIndex(null);
        List<B2bEmployeeExcelVo> importDataList = new ArrayList<>();

        try {
            ExcelImportResult<B2bEmployeeExcelVo> result = ExcelImportUtil.importExcelMore(EasyPoiExportUtil.getInputStream(fileUrl), B2bEmployeeExcelVo.class, importParams);

            if (ObjectUtils.isNotEmpty(result)) {
                importDataList = result.getList();
                //筛选对象属性全部为null数据
                importDataList = BeanPropertyNullUtil.getAllFieldNullList(importDataList);
            }
            if (CollectionUtils.isEmpty(importDataList)) {
                throw new BizException("无法读取 Excel 中的数据，请检查数据是否符合模板格式要求");
            }
        } catch (Exception e) {
            logger.info(e.getMessage());
            e.printStackTrace();
            throw new BizException("无法读取 Excel 文件，请检查文件类型");
        }
        if (CollectionUtils.isEmpty(importDataList)){
            response.setTotalNum(0);
            response.setSuccessNum(0);
            response.setErrorNum(0);
            return new RestResponse<>(response);
        }
        logger.info("导入数据：{}", JSON.toJSONString(importDataList));
        response.setTotalNum(importDataList.size());

        List<String> roleNameList = importDataList.stream().filter(data -> StringUtils.isNotEmpty(data.getRoleName())).map(B2bEmployeeExcelVo::getRoleName).distinct().collect(Collectors.toList());

        /* 获取角色列表数据 */
        Map<String, List<RoleExpandRespDto>> roleMap =employeeExpand2Service.getRoleMap(roleNameList,orgId);

        //获取账户信息
        List<String> accountList = importDataList.stream().filter(data -> StringUtils.isNotEmpty(data.getAccount())).map(B2bEmployeeExcelVo::getAccount).distinct().collect(Collectors.toList());
        logger.info("获取账户信息获取账号信息：{}", JSON.toJSONString(accountList));
        Map<String, List<UserOrgizationRespDto>> userOrgMap = employeeExpand2Service.getAccoutMap(accountList);
        /* 校验数据 */
        importDataList = this.verifyData(importDataList,roleMap,userOrgMap,orgId);
//        logger.info("品牌端人员校验数据:{}",JSON.toJSONString(importDataList));
        List<B2bEmployeeExcelVo> successList = importDataList.stream().filter(dto -> StringUtils.isEmpty(dto.getErrorMsg())).collect(Collectors.toList());
        /* 合并数据 */
        List<B2bEmployeeExcelVo> employeeExcelVoList = this.mergeData(successList);

        Map<String,Long> employeeInfoMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(employeeExcelVoList)){
            List<Long> userIdList = employeeExcelVoList.stream().map(B2bEmployeeExcelVo::getUserId).collect(Collectors.toList());
            employeeInfoMap = employeeExpand2Service.getEmployeeMap(userIdList,Collections.singletonList(employeeExcelVoList.get(0).getOrgId()));
        }

        /* 处理数据 */
        for (B2bEmployeeExcelVo data : employeeExcelVoList){
            //构建人员对象
            EmployeeExpandReqDto employeeExtReqDto = new EmployeeExpandReqDto();
            employeeExtReqDto.setOrganizationId(data.getOrgId());
            employeeExtReqDto.setOrgId(data.getOrgId());
            employeeExtReqDto.setOrganizationStatus(YesOrNoEnum.NO.getStatus());
            employeeExtReqDto.setName(data.getUserName());//姓名
            employeeExtReqDto.setPosition(data.getPosition());//职位
            employeeExtReqDto.setSex(data.getSex());//性别
            employeeExtReqDto.setStation(data.getDepartment());//部门
            // 设置人员类型，内部为0，外部为1
            employeeExtReqDto.setType("内部".equals(data.getEmployeeType()) ? 0 : 1);
            employeeExtReqDto.setPhoneNum(data.getPhoneNumber());//手机号码
            employeeExtReqDto.setEmployeeRoleDtoList(data.getRoleList());//角色
            employeeExtReqDto.setUserId(data.getUserId());//账号
            employeeExtReqDto.setStartTime(new Date());
            employeeExtReqDto.setStatus(1);
            //查询该账号在组织下是否已被绑定，若已绑定，则修改，没有则新增
            String key = data.getOrgId() + "_" + data.getUserId();
            try {
                if(!employeeInfoMap.isEmpty() && employeeInfoMap.containsKey(key)){
                    Long id = employeeInfoMap.get(key);
                    //修改
                    employeeExtReqDto.setId(id);
                    logger.info("修改b2b人员入参:{}",JSON.toJSONString(employeeExtReqDto));
                    employeeExpand2Service.modifyEmployee(instanceId,employeeExtReqDto);
                }else {
                    //新增
                    logger.info("新增b2b人员入参:{}",JSON.toJSONString(employeeExtReqDto));
                    employeeExpand2Service.addEmployee(instanceId,employeeExtReqDto);
                }
            }catch (Exception e){
                logger.info("品牌端人员执行失败:{}",e);
                setErrorsMsg(importDataList,data.getAccount(),"执行失败！"+e.getMessage());
            }
        }
//        logger.info("品牌端人员校验数据22222:{}",JSON.toJSONString(importDataList));
        List<B2bEmployeeExcelVo> errorList = importDataList.stream().filter(vo -> StringUtils.isNotEmpty(vo.getErrorMsg())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(errorList)){
//            logger.info("品牌端人员异常数据:{}",JSON.toJSONString(errorList));
            response.setErrorNum(errorList.size());
            String excelName ="品牌端人员导入失败信息-"+ DateUtil.getDateFormat(new Date(), DateUtil.YMDSTRING_DATA);
            String exportUrl;
            try {
                exportUrl = EasyPoiExportUtil.getExportUrl(errorList,B2bEmployeeExcelVo.class, null, excelName, "xls");
                response.setErrorFileUrl(exportUrl);
            } catch (Exception e) {
                logger.info("品牌端人员导入失败信息文件生成失败");
                logger.info(e.getMessage(),e);
            }
        }

        response.setSuccessNum(response.getTotalNum()-response.getErrorNum());
//        logger.info("品牌端人员返回数据1313213:{}",JSON.toJSONString(response));
        return new RestResponse<>(response);
    }

    /**
     *
     * @param list 原数据
     * @param account 账号
     * @param errorMsg 错误信息
     * @return
     */
    public List<B2bEmployeeExcelVo> setErrorsMsg(List<B2bEmployeeExcelVo> list,String account,String errorMsg){
        list.forEach(vo->{
            if(StringUtils.isBlank(vo.getErrorMsg()) && vo.getAccount().equals(account)){
                vo.setErrorMsg(errorMsg);
            }
        });
        return list;
    }

    /**
     * 校验数据
     * @param list 原数据
     * @return
     */
    public List<B2bEmployeeExcelVo> verifyData(List<B2bEmployeeExcelVo> list, Map<String,List<RoleExpandRespDto>>  roleMap,
                                            Map<String, List<UserOrgizationRespDto>> userOrgMap,Long orgId){

        for (B2bEmployeeExcelVo vo : list) {
            if (StringUtils.isBlank(vo.getAccount())){
                vo.setErrorMsg("登录账号不能为空");
                continue;
            }

            if(userOrgMap.isEmpty() || !userOrgMap.containsKey(vo.getAccount())){
                vo.setErrorMsg("登录账号错误");
                continue;
            }
            List<UserOrgizationRespDto> userOrgizationVos = userOrgMap.get(vo.getAccount());
            List<UserOrgizationRespDto> userOrgList = userOrgizationVos.stream().filter(e -> orgId.compareTo(e.getOrgId()) == 0).collect(Collectors.toList());
            if(CollectionUtils.isEmpty(userOrgList)){
                vo.setErrorMsg("登录账号所属组织错误");
                continue;
            }
            vo.setUserId(userOrgList.get(0).getUserId());
            vo.setOrgId(orgId);
            if (StringUtils.isBlank(vo.getUserName())){
                vo.setErrorMsg("姓名不能为空");
                continue;
            }
            if (StringUtils.isBlank(vo.getPosition())){
                vo.setErrorMsg("职位不能为空");
                continue;
            }
            if (StringUtils.isBlank(vo.getEmployeeType())){
                vo.setErrorMsg("人员类型不能为空");
                continue;
            }
            if (!("内部".equals(vo.getEmployeeType()) || "外部".equals(vo.getEmployeeType()))){
                vo.setErrorMsg("人员类型只能为'内部'或'外部'");
                continue;
            }
            if (StringUtils.isNoneBlank(vo.getRoleName())){
                if(roleMap.isEmpty() || !roleMap.containsKey(vo.getRoleName())){
                    vo.setErrorMsg("角色错误");
                    continue;
                }
                List<RoleExpandRespDto> roleExpandEos = roleMap.get(vo.getRoleName());
                List<RoleExpandRespDto> orgRoleList = roleExpandEos.stream()
                        .filter(e -> (Objects.nonNull(e.getBelongOrgId()) && e.getBelongOrgId().compareTo(vo.getOrgId()) == 0) || e.getRoleType() == 1)
                        .collect(Collectors.toList());
                if(CollectionUtils.isEmpty(orgRoleList)){
                    vo.setErrorMsg("角色所属组织错误");
                    continue;
                }
                EmployeeRoleReqDto employeeRoleReqDto = new EmployeeRoleReqDto();
                employeeRoleReqDto.setRoleId(orgRoleList.get(0).getId());
                employeeRoleReqDto.setRoleCode(orgRoleList.get(0).getCode());
                vo.setRole(employeeRoleReqDto);
            }
            if(StringUtils.isNotEmpty(vo.getPhoneNumber())){
                if(!CommonUtil.checkPhoneNum(vo.getPhoneNumber())){
                    vo.setErrorMsg("手机号格式不正确;");
                    continue;
                }
            }

        }

        return list;
    }

    /**
     * 合并数据（合并重复的数据）
     * @param list 原数据
     * @return
     */
    private List<B2bEmployeeExcelVo> mergeData(List<B2bEmployeeExcelVo> list) {
        if(CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        //根据账户进行分组
        Map<String, List<B2bEmployeeExcelVo>> employeeGroupMap = list.stream().collect(Collectors.groupingBy(B2bEmployeeExcelVo::getAccount));
        List<B2bEmployeeExcelVo> resultVoList = Lists.newArrayList();
        for (String account : employeeGroupMap.keySet()) {
            List<B2bEmployeeExcelVo> employeeExcelVos = employeeGroupMap.get(account);
            B2bEmployeeExcelVo vo = new B2bEmployeeExcelVo();
            CubeBeanUtils.copyProperties(vo,employeeExcelVos.get(0));
            //合并多个角色
            List<EmployeeRoleReqDto> roleList = employeeExcelVos.stream().filter(v -> Objects.nonNull(v.getRole())).map(B2bEmployeeExcelVo::getRole)
                    .collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(EmployeeRoleReqDto:: getRoleId))), ArrayList::new));
            if(CollectionUtils.isNotEmpty(roleList)){
                vo.setRoleList(roleList);
            }

            resultVoList.add(vo);
        }
        return resultVoList;


    }


}
