/**
 * @(#) IItemQueryApi.java 1.0 2018-08-22
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.item.api.base.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.*;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.response.*;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author 黑马
 * @since 1.0.0
 */
@Api(tags = {"商品中心：商品服务"})
@FeignClient(name = "${yundt.cube.center.item.api.name:yundt-cube-center-item}",
//        contextId = "com-yunxi-bundle-item-api-query-IItemQueryApi",
        contextId = "com-dtyunxi-yundt-cube-item-api-base-query-IItemQueryApi",
        path = "/v1/item",
        url = "${yundt.cube.center.item.api:}")
public interface IItemQueryApi {


    @PostMapping(value = "/item-lib/sku/page")
    @ApiOperation(value = "分页查询商品库商品sku列表", notes = "分页查询商品库商品sku列表")
    RestResponse<PageInfo<ItemSkuExtRespDto>> queryItemLibSkuByPage(@RequestBody ItemReqDto itemReqDto,
                                                                    @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                    @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 查询sku分页
     *
     * @param filter 请求dto
     * @return 分页
     */
    @GetMapping(value = "/on-self/page")
    @ApiOperation(value = "查询sku分页", notes = "分页查询商品列表，filter=ItemReqDto")
    RestResponse<PageInfo<ItemOnShelfRespDto>> findItemOnSelf(@RequestParam("filter") String filter,
                                                              @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                              @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);


    /**
     * 分页查询商品列表  filter -> ItemReqDto
     *
     * @param filter   查询条件
     * @param pageNum  页码  [json转dto]
     * @param pageSize 每页记录数
     * @return 返回分页ItemDto列表
     */
    @GetMapping(value = "/list")
    @ApiOperation(value = "分页查询商品列表", notes = "分页查询商品列表，filter=ItemReqDto")
    RestResponse<PageInfo<ItemRespDto>> queryByPage(@RequestParam("filter") String filter,
                                                    @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                    @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);


    @GetMapping(value = "/itemlib/page")
    @ApiOperation(value = "分页查询商品库商品列表", notes = "分页查询商品库商品列表")
    RestResponse<PageInfo<ItemRespDto>> queryItemLibByPage(@ModelAttribute ItemReqDto itemReqDto,
                                                           @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                           @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    //post方式重载
    @PostMapping(value = "/itemlib/search")
    @ApiOperation(value = "分页查询商品库商品列表", notes = "分页查询商品库商品列表")
    RestResponse<PageInfo<ItemRespDto>> searchItemLibByPage(@RequestBody ItemReqDto itemReqDto,
                                                            @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                            @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);



    @GetMapping(value = "/page")
    @ApiOperation(value = "分页查询上架商品列表", notes = "分页查询上架商品列表", response = ItemShelfQueryRespDto.class)
    RestResponse<PageInfo<ItemShelfQueryRespDto>> queryShelfItemByPage(@ModelAttribute ItemShelfQueryReqDto itemShelfQueryReqDto,
                                                                       @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                       @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    @PostMapping(value = "/pageByPost")
    @ApiOperation(value = "分页查询上架商品列表", notes = "分页查询上架商品列表", response = ItemShelfQueryRespDto.class)
    RestResponse<PageInfo<ItemShelfQueryRespDto>> queryShelfItemByPageByPost(@RequestBody ItemShelfQueryReqDto itemShelfQueryReqDto,
                                                                             @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                             @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);



    @GetMapping(value = "/{shopId}/{itemId}/skus")
    @ApiOperation(value = "查询店铺下指定上架商品包含sku列表", notes = "查询店铺下指定上架商品包含sku列表", response = ItemShelfSkuRespDto.class)
    RestResponse<List<ItemShelfSkuRespDto>> queryShelfItemSkuList(@PathVariable("shopId") Long shopId, @PathVariable("itemId") Long itemId, Integer busType);


    @GetMapping(value = "/shelf/count")
    @ApiOperation(value = "统计店铺上下架商品的数量", notes = "统计店铺上下架商品的数量", response = ItemShelfCountRespDto.class)
    RestResponse<ItemShelfCountRespDto> countShelfItem(@ModelAttribute ItemShelfQueryReqDto itemShelfQueryReqDto);

    /**
     * 根据商品ID查询商品详情
     *
     * @param id    主键  备注: /{id}/include/{field} :sku,attributes,medias
     * @param field 显示项：sku,attributes,medias task表示查询SKU,商品属性,多媒体,定时任务
     * @param dirId 目录ID
     * @return 返回ItemDetailDto列表
     */
    @GetMapping(value = "/{id}/included/{field}")
    @ApiOperation(value = "根据商品ID查询商品详情", notes = "根据商品ID查询商品详情,\nfield 显示项：sku,attributes,medias,task表示查询SKU,商品属性,多媒体,定时任务。\ndirId传值表明查询该目录的上架数据，不传则查询商品ID的对应的上架数据。")
    RestResponse<ItemDetailRespDto> queryItemDetail(@PathVariable("id") Long id,
                                                    @PathVariable(value = "field", required = false) String field,
                                                    @RequestParam(value = "dirId", required = false) Long dirId);

    /**
     * 根据关键字查询商品
     *
     * @param itemSearchReqDto 关键字
     * @return 返回ItemKeywordDto列表
     */
    @GetMapping("/search")
    @ApiOperation(value = "根据关键字查询商品", notes = "请使用/query")
    RestResponse<ItemSearchRespDto> queryByKeyword(@ModelAttribute ItemSearchReqDto itemSearchReqDto);


    /**
     * 根据关键字查询商品，数据隔离
     *
     * @param filter   查询入参
     * @param pageNum  当前页
     * @param pageSize 页大小
     * @return 返回ItemKeywordDto列表
     */
    @Deprecated
    @GetMapping("/query")
    @ApiOperation(value = "根据关键字查询商品", notes = "filter=ItemSearchQueryReqDto={\n" +
            "\"keyword\":\"关键字\",\n" +
            "\"brandId\":\"品牌ID\",\n" +
            "\"type\":\"商品类型\",\n" +
            "\"prop\":\"属性,例子：颜色:红色;存储:8G\",\n" +
            "\"dirIds\":\"目录/类目ID集\",\n" +
            "\"sellerId\":\"商户ID\",\n" +
            "\"shopId\":\"店铺ID\"，\n" +
            "\"busType\":\"业务类型：0 普通； 1 社区团购  默认0\"，\n" +
            "\"orderByDesc\":\"按该值降序排列查询(目前支持，sales、price)，不可和orderBy同时使用，同时使用时默认使用orderByDesc\"，\n" +
            "\"orderBy\":\"按该值升序排列查询(目前支持，sales、price)，不可和orderByDesc同时使用，同时使用时默认使用orderByDesc\"，\n" +
            "\"maxPrice\":\"最大价格\"，\n" +
            "\"minPrice\":\"最小价格\"，\n" +
            "\"pageNum\":\"当前页\",\n" +
            "\"pageSize\":\"页大小\",\n" +
            "\"instanceId\":\"实例ID\",\n" +
            "\"tenantId\":\"租户ID\"\n" +
            "}")
    RestResponse<ItemSearchRespDto> queryByKeyword(@RequestParam("filter") String filter,
                                                   @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                   @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);


    /**
     * 根据标签查询商品列表
     *
     * @param tagIds   标签ID
     * @param pageNum  指定页码
     * @param pageSize 指定每页返回的对象个数
     * @param filter   过滤条件
     * @return 统一返回响应对象，业务数据data为标签Dto分页列表
     */
    @GetMapping(value = "/tag/{tagIds}")
    @ApiOperation(value = "根据标签查询商品列表", notes = "根据标签查询商品列表，filter=RequestDto")
    RestResponse<PageInfo<ItemRespDto>> queryItemByTagId(@NotNull(message = "标签id不能为空") @PathVariable("tagIds") String tagIds,
                                                         @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                         @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize,
                                                         @RequestParam(name = "filter", required = false) String filter
    );

    /**
     * 通过品牌id查询商品列表
     *
     * @param pageNum  分页页码
     * @param pageSize 分页行数
     * @param brandIds 品牌Id
     * @param filter   过滤条件
     * @return 返回分页ItemDto列表
     */
    @GetMapping(value = "/brand/{brandIds}")
    @ApiOperation(value = "通过品牌品牌id查询商品列表", notes = "通过目录id查询商品列表 \tu\n id:品牌id，filter=RequestDto")
    RestResponse<PageInfo<ItemRespDto>> queryItemByBrandId(@NotNull @PathVariable("brandIds") String brandIds,
                                                           @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                           @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize,
                                                           @RequestParam(name = "filter", required = false) String filter
    );

    /**
     * 通过目录id查询商品列表  filter -> RequestDto
     *
     * @param pageNum  分页页码
     * @param pageSize 分页行数
     * @param dirIds   目录Id
     * @param filter   过滤条件
     * @return 返回分页ItemDto列表
     */
    @GetMapping(value = "/dir/{dirIds}")
    @ApiOperation(value = "通过目录id查询商品列表", notes = "通过目录id查询商品列表 \tu\n id:目录id，filter=RequestDto")
    RestResponse<PageInfo<ItemRespDto>> queryItemByDirId(@NotNull @PathVariable("dirIds") String dirIds,
                                                         @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                         @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize,
                                                         @RequestParam(name = "filter", required = false) String filter);

    /**
     * 通过目录ID深度查询商品列表
     *
     * @param itemSearchReqDto 商品深度查询入参Dto
     * @param pageNum          当前页
     * @param pageSize         页大小
     * @return 返回分页ItemDto列表
     */
    @PostMapping("/dir/depth")
    @ApiOperation(value = "通过目录ID深度查询商品列表", notes = "查询dirIds及其所有子目录的上架商品，filter=ItemReqDto")
    RestResponse<ItemSearchRespDto> queryItemByDirIdDepth(@RequestBody ItemSearchQueryReqDto itemSearchReqDto,
                                                          @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                          @RequestParam(value = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 通过商品id查询商品属性列表
     *
     * @param id       商品id
     * @param pageNum  指定页
     * @param pageSize 页大小
     * @return 返回分页ItemDto列表
     */
    @GetMapping(value = "/{id}/attributes/list")
    @ApiOperation(value = "通过商品id查询商品属性列表", notes = "通过商品id查询商品属性列表 \t\n id:商品id")
    RestResponse<PageInfo<ItemAttributesRespDto>> queryItemAttrById(@NotNull(message = "商品id不能为空") @PathVariable("id") Long id,
                                                                    @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                    @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据商品id查询规格列表
     *
     * @param itemId 商品id
     * @return 规格列表
     */
    @GetMapping(value = "/{itemId}/sku/list")
    @ApiOperation(value = "通过商品id查询规格列表", notes = "通过商品id查询规格列表 \t\n itemId:商品id")
    RestResponse<List<ItemSkuRespDto>> queryItemSkuByItemId(@NotNull @PathVariable("itemId") Long itemId);

    /**
     * 根据商品id查询多媒体信息
     *
     * @param itemId 商品id
     * @return 多媒体信息
     */
    @GetMapping(value = "/{itemId}/medias/list")
    @ApiOperation(value = "通过商品id查询多媒体信息列表", notes = "通过商品id查询多媒体信息列表 \t\n itemId:商品id")
    RestResponse<List<ItemMediasRespDto>> queryItemMediasByItemId(@NotNull @PathVariable("itemId") Long itemId);

    /**
     * 根据skuId查询商品列表
     *
     * @param skuIds id
     * @return 返回商品，sku信息
     */
    @GetMapping(value = "/list/{skuIds}")
    @ApiOperation(value = "根据skuId查询商品列表", notes = "根据skuId查询商品列表,skuId支持传复数 \n skuIds:规格ID集合")
    RestResponse<ItemSkuListRespDto> queryItemBySkuId(@NotNull @PathVariable("skuIds") List<Long> skuIds);

    /**
     * 根据商品ID查询商品价格信息
     *
     * @param itemId 商品ID
     * @return 统一返回响应对象，业务数据data为商品价格列表
     */
    @GetMapping(value = "/{itemId}/price")
    @ApiOperation(value = "根据商品ID查询商品价格信息", notes = "根据商品ID查询商品价格信息 \n itemId:商品ID")
    RestResponse<List<ItemPriceRespDto>> queryItemPriceByItemId(@NotNull @PathVariable("itemId") Long itemId);

    /**
     * 根据商品id查询该商品所有的上架目录信息
     *
     * @param filter 查询条件
     * @return 商品和目录的关系
     */
    @GetMapping("/dir/list")
    @ApiOperation(value = "根据商品id查询该商品所有的上架目录信息", notes = "根据商品id查询该商品所有的上架目录信息 \n filter=ItemDirQueryReqDto")
    RestResponse<List<ItemDirQueryRespDto>> queryItemDir(@RequestParam("filter") String filter);

    /**
     * 根据条件查询商品的上架关系
     *
     * @param filter 查询条件
     * @return 商品和目录的关系
     */
    @GetMapping("/dir")
    @ApiOperation(value = "根据条件查询商品的上架关系", notes = "根据条件查询商品的上架关系 \n filter=DirItemRelativeReqDto")
    RestResponse<List<DirItemRelativeQueryRespDto>> queryItemDirRelative(@RequestParam("filter") String filter);

    /**
     * 查询商品定时上下架计划
     *
     * @param filter   查询条件
     * @param pageNum  当前页
     * @param pageSize 页大小
     * @return 商品定时计划
     */
    @GetMapping("/shelf/task/page")
    @ApiOperation(value = "查询商品定时上下架计划", notes = "filter=ShelfTaskQueryReqDto")
    RestResponse<PageInfo<ShelfTaskQueryRespDto>> queryShelfTask(@RequestParam("filter") String filter,
                                                                 @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                 @RequestParam(value = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 检查商品是否允许下单
     *
     * @param filter
     * @return
     */
    @GetMapping("/allow/order")
    @ApiOperation(value = "检查商品是否允许下单", notes = "请求入参是集合，集合元素是ItemAllowOrderReqDto。filter=new ArrayList<ItemAllowOrderReqDto>()")
    RestResponse<ItemAllowOrderQueryRespDto> checkItemAllowOrder(@RequestParam("filter") String filter);

    /**
     * 根据规格编码code查询sku
     *
     * @param codes 规格编码列表
     * @return
     */
    @GetMapping(value = "/{codes}")
    @ApiOperation(value = "根据规格编码code查询sku", notes = "根据规格编码code查询sku")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "codes", value = "规格编码列表", dataType = "String", paramType = "path")
    })
    RestResponse<ItemSkuCodeQueryRespDto> querySkuByCode(@PathVariable("codes") String codes);

    /**
     * 根据购物车商品ID进行失效商品校验
     *
     * @param itemIds 商品ID
     * @param filter  过滤条件
     * @return 失效商品ID
     */
    @GetMapping(value = "/batch/{itemIds}")
    @ApiOperation(value = "根据购物车商品ID进行失效商品校验", notes = "根据购物车商品ID进行失效商品校验，filter=RequestDto")
    RestResponse<List<ItemRespDto>> checkShoppingCartItemInvalid(@PathVariable(name = "itemIds") String itemIds,
                                                                 @RequestParam("filter") String filter);

    /**
     * 根据商品ID查询单位时间内商品销量
     *
     * @param itemId 商品ID
     * @param filter 过滤条件
     * @return 对应数据
     */
    @GetMapping("/stat/sales/{itemId}")
    @ApiOperation(value = "根据商品ID查询单位时间内商品销量", notes = "根据商品ID查询单位时间内商品销量,filter=RequestDto.class传入租户和实例")
    RestResponse<ItemStatRespDto> queryItemSalesCountByItemId(@PathVariable("itemId") Long itemId,
                                                              @RequestParam("filter") String filter);

    /**
     * 根据商品ID列表批量查询商品
     *
     * @param itemIds 商品id列表
     * @return 商品列表数据
     * @author wu.sheng
     * @date 2019/11/25 14:11
     */
    @GetMapping(value = "/ids")
    @ApiOperation(value = "根据商品id列表批量查询商品", notes = "批量查询，多个逗号隔开")
    RestResponse<List<ItemRespDto>> queryByItemIds(@RequestParam("itemIds") String itemIds);

    /**
     * 根据商品ID列表批量查询商品
     *
     */
    @PostMapping(value = "/queryByItemIdsOnPost")
    @ApiOperation(value = "根据商品id列表批量查询商品", notes = "根据商品id列表批量查询商品")
    RestResponse<List<ItemRespDto>> queryByItemIdsOnPost(@RequestBody List<String> itemIds);

    /**
     * 根据商品ID列表批量查询商品
     *
     * @param itemIds 商品id列表
     * @return 商品列表数据
     * @author wu.sheng
     * @date 2019/11/25 14:11
     */
    @PostMapping(value = "/new/ids")
    @ApiOperation(value = "根据商品id列表批量查询商品", notes = "批量查询，多个逗号隔开")
    RestResponse<List<ItemRespDto>> queryNewByItemIds(@RequestParam("itemIds") String itemIds);

    /**
     * 商品售罄列表
     *
     * @param itemReq 查询属性
     * @param page    当前页
     * @param size    页大小
     * @return 带分页信息的商品IDS
     */
    @GetMapping("sale-out/page")
    @ApiOperation(value = "根据商品ID列表批量查询商品", notes = "根据商品ID列表批量查询商品,filter=ItemReqDto")
    RestResponse<PageInfo<Long>> querySaleOut(@RequestParam("filter") String itemReq, @RequestParam("page") int page, @RequestParam("size") int size);

    /**
     * 统计某个状态数量
     *
     * @param filter
     * @return
     */
    @GetMapping("/count")
    @ApiOperation(value = "统计商品的数量", notes = "统计商品的数量")
    RestResponse<Long> queryCount(@RequestParam("filter") String filter);

    /**
     * 查询商品上架列表
     *
     * @param filter
     * @param pageNum  指定页
     * @param pageSize 页大小
     * @return 返回分页ItemShelfRespDto列表
     */
    @GetMapping(value = "/shelf/list")
    @ApiOperation(value = "查询商品上架列表", notes = "查询商品上架列表 \t\n ,filter=ShelfReqDto")
    RestResponse<PageInfo<ItemShelfRespDto>> queryItemShelf(@RequestParam("filter") String filter,
                                                            @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                            @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);


    @GetMapping(value = "item-id/list")
    @ApiOperation(value = "查询商品id列表", notes = "查询商品id列表,filter=ItemReqDto")
    RestResponse<List<Long>> queryItemIds(@RequestParam("filter") String filter);


    @GetMapping(value = "/{itemId}/{skuId}")
    @ApiOperation(value = "根据商品id和skuId查询商品及sku信息", notes = "走缓存，根据商品id和skuId查询sku信息，返回商品基本信息，sku基本信息")
    RestResponse<ItemRespDto> queryItemAndSku(@PathVariable("itemId") Long itemId, @PathVariable("skuId") Long skuId);

    @PostMapping(value = "/getSkuDetail")
    @ApiOperation(value = "获取sku详情", notes = "获取sku详情")
    RestResponse<ItemSkuDetailRespDto> getSkuDetail(@RequestBody ItemSkuReqDto itemSkuReqDto);

    @GetMapping(value = "/getSkuCostPrice")
    @ApiOperation(value = "获取sku成本价", notes = "获取sku成本价")
    RestResponse<BigDecimal> getSkuCostPrice(@RequestParam("skuId") Long skuId);
    /**
     * @param branchQueryItemReq
     * @return
     */
    @PostMapping(value = "/detail/branch")
    @ApiOperation(value = "批量查询商品信息（性能优化订单专用接口，其他勿用）", notes = "批量查询商品信息（性能优化订单专用接口，其他勿用）")
    RestResponse<BranchItemRespDto> branchQueryItem(@RequestBody BranchQueryItemReqDto branchQueryItemReq);

    @GetMapping(value = "/detail/{id}")
    @ApiOperation(value = "根据商品ID查询商品详情(移植于应用)", notes = "根据商品ID查询商品详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "商品id", paramType = "path", dataType = "Int", required = true),
            @ApiImplicitParam(name = "shopId", value = "店铺id", paramType = "query", dataType = "Long")
    })
    RestResponse<ItemDetailBaseInfoRespDto> detail(@PathVariable("id") long id, @RequestParam(value = "shopId", required = false, defaultValue = "0") long shopId);


    @GetMapping(value = "/freightTemplate/count/{freightTemplateId}")
    @ApiOperation(value = "根据运费模板查询使用了该运费模板的上架商品数量", notes = "根据运费模板查询使用了该运费模板的上架商品数量")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "freightTemplateId", value = "运费模板ID", paramType = "path", dataType = "Long", required = true),
    })
    RestResponse<Integer> countByFreightTemplateId(@PathVariable("freightTemplateId") Long freightTemplateId);


    @GetMapping(value = "/pointsDeduction/shelfIds")
    @ApiOperation(value = "查询上架商品的积分扣减规则信息", notes = "查询上架商品的积分扣减规则信息")
    RestResponse<List<PointsDeductionRespDto>> queryShelfSkuPointsDeduction(@NotNull @RequestParam("shelfIds") Set<Long> shelfIds);


    @GetMapping(value = "/onShelf/list")
    @ApiOperation(value = "查询商品上架信息", notes = "查询商品上架信息")
    RestResponse<List<ShelfQueryRespDto>> queryShelfList(@ModelAttribute ShelfQueryReqDto shelfQueryReqDto);

    @GetMapping(value = "/{shopId}/{itemId}/{itemBusType}/bundle")
    @ApiOperation(value = "查询组合商品上架明细", notes = "查询组合商品上架明细")
    RestResponse<List<BundleItemRespDto>> queryBundleItemDetail(@PathVariable("shopId") Long shopId,
                                                                @PathVariable("itemId") Long itemId,
                                                                @PathVariable("itemBusType") Integer itemBusType);

    @PostMapping(value = "/sku/list")
    @ApiOperation(value = "批量查询商品sku基本信息", notes = "批量查询商品sku基本信息")
    RestResponse<List<ItemSkuRespDto>> queryItemSkus(@RequestBody List<Long> skuIds);

    @GetMapping(value = "/{itemId}/bundle")
    @ApiOperation(value = "查询组合商品基本信息", notes = "查询组合商品基本信息，单表查询")
    RestResponse<List<BundleItemRespDto>> queryBundleItem(@PathVariable("itemId") Long itemId);

    @GetMapping(value = "/bundle/list")
    @ApiOperation(value = "查询组合商品的", notes = "查询组合商品基本信息，单表查询")
    RestResponse<List<BundleItemRespDto>> queryBundleItemList(@ModelAttribute BundleItemReqDto bundleItemReqDto);

    @GetMapping(value = "/{itemId}/baseProp")
    @ApiOperation(value = "查询商品的基础属性信息", notes = "查询商品的基础属性信息")
    RestResponse<List<PropNameRespDto>> queryItemBasePropList(@PathVariable("itemId") Long itemId);

    @GetMapping(value = "/queryByItemCode/{itemCode}")
    @ApiOperation(value = "根据商品编码查询商品信息", notes = "根据商品编码查询商品信息")
    RestResponse<ItemRespDto> queryByItemCode(@PathVariable("itemCode") String itemCode);
    /**
     * 通过商品id集合查询商品属性列表
     *
     * @param itemIdList  商品id集合
     * @return 返回ItemAttributesRespDto列表
     */
    @PostMapping(value = "/queryItemAttrByItemIdList")
    @ApiOperation(value = "通过商品id集合查询商品属性列表", notes = "通过商品id集合查询商品属性列表")
    RestResponse<List<ItemAttributesRespDto>> queryItemAttrByItemIdList(@RequestBody List<Long> itemIdList);

    @PostMapping(value = "/queryByItemLongCodes")
    @ApiOperation(value = "根据商品长编码集合查询商品信息", notes = "根据商品长编码集合查询商品信息")
    RestResponse<List<ItemRespDto>> queryByItemLongCodes(@RequestBody List<String> itemCodes);

    @PostMapping(value = "/queryItemMediasByItemIds")
    @ApiOperation(value = "根据商品id查询商品多媒体信息", notes = "根据商品id查询商品多媒体信息")
    RestResponse<Map<Long, String>> queryItemMediasByItemIds(@RequestBody List<Long> itemIds);

    @PostMapping(value = "/queryItemSkuRelByCodes")
    @ApiOperation(value = "根据商品编码查询与sku的关系", notes = "根据商品编码查询与sku的关系")
    RestResponse<List<ItemSkuRelRespDto>> queryItemSkuRelByCodes(@RequestBody List<String> itemCodes);

    @PostMapping(value = "/queryItemSkuByItemCodes")
    @ApiOperation(value = "根据商品编码查询与sku的关系-多字段返回", notes = "根据商品编码查询与sku的关系")
    RestResponse<List<ItemSkuRelRespDto>> queryItemSkuByItemCodes(@RequestBody List<String> itemCodes);

    @PostMapping(value = "/queryItemSkuBySkuCodes")
    @ApiOperation(value = "根据商品编码查询与sku的关系-SkuCodes", notes = "根据商品编码查询与sku的关系")
    RestResponse<List<ItemSkuRelRespDto>> queryItemSkuBySkuCodes(@RequestBody List<String> skuCodes);

    /**
     * 根据条件批量查询商品详情
     * @param itemDetailReqDto
     * @return
     */
    @PostMapping(value = "/list/detail")
    @ApiOperation(value = "根据条件批量查询商品详情", notes = "根据条件批量查询商品详情")
    RestResponse<List<ItemDetailRespDto>> listItemDetail(@RequestBody ItemDetailReqDto itemDetailReqDto);
}