/*
 * @(#)IDirectoryItemApi.java 1.0 2018-08-21
 *
 * 			Copyright (c) 2017, YUNXI. All rights reserved.
 * 			YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.item.api.base;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.*;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.DirectoryCreateReqDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.DirectoryReqDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.DirectorySortUpdateDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.RootDirectoryReqDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 目录服务
 *
 * @author 诺克
 * @since 2.0.0
 */
@Api(tags = {"商品中心：目录服务"})
@FeignClient(name = "${yundt.cube.center.item.api.name:yundt-cube-center-item}",
        contextId = "com-dtyunxi-yundt-cube-center-item-api-base-IDirectoryApi",
        path = "/v1/dir",
        url = "${yundt.cube.center.item.api:}")
public interface IDirectoryApi {

    /**
     * 新增根目录
     *
     * @param rootDirectoryReqDto 目录请求对象
     * @return 统一返回响应对象
     */
    @PostMapping(value = "/root")
    @ApiOperation(value = "新增目录接口")
    RestResponse<Long> addRootDirectory(@Valid @RequestBody RootDirectoryReqDto rootDirectoryReqDto);

    /**
     * 新增目录项
     *
     * @param directoryReqDto 目录项对象
     * @return 返回执行成功的条数
     */
    @PostMapping(value = "")
    @ApiOperation(value = "新增目录项", notes = "新增目录项，要保证目录存在")
    RestResponse<Long> addDirectory(@Validated @RequestBody DirectoryReqDto directoryReqDto);

    /**
     * 批量初始化类目
     *
     * @param directoryCreateReqDtos 批量创建类目的请求dto
     * @return
     */
    @PostMapping(value = "/addDir/list")
    @ApiOperation(value = "批量初始化类目", notes = "批量初始化类目")
    RestResponse<Void> addDirList(@Validated @RequestBody List<DirectoryCreateReqDto> directoryCreateReqDtos);

    /**
     * 修改根目录
     *
     * @param rootDirectoryReqDto 目录请求对象
     * @return 统一返回业务对象
     */
    @PutMapping(value = "/root")
    @ApiOperation(value = "修改目录", notes = "目录id")
    RestResponse<Void> modifyRootDirectory(@Validated @RequestBody RootDirectoryReqDto rootDirectoryReqDto);

    /**
     * 修改目录项
     *
     * @param directoryReqDto 目录请求对象
     * @return 统一返回响应对象
     */
    @PutMapping("")
    @ApiOperation(value = "修改目录项")
    RestResponse<Void> modifyDirectory(@Validated @RequestBody DirectoryReqDto directoryReqDto);

    /**
     * 修改目录状态
     *
     * @param id
     * @param status
     * @return RestResponse
     */
    @PutMapping(value = "/{id}/status/{status}")
    @ApiOperation(value = "修改目录状态", notes = "修改目录状态，status=0禁用 1启用")
    RestResponse<Void> modifyDirStatus(@PathVariable("id") Long id, @PathVariable("status") Integer status);

    /**
     * 合并目录内容， 仅限叶子目录
     *
     * @param targetDirId
     * @param sourceDirIds
     * @return
     */
    @PutMapping(value = "/{targetDirId}/merge/dir/{sourceDirIds}")
    @ApiOperation(value = "合并目录", notes = "目录id，targetDirId=目标目录ID，sourceDirIds=被合并目录ID，不支持复数")
    RestResponse<Void> mergeDirectory(@PathVariable("targetDirId") Long targetDirId, @PathVariable("sourceDirIds") String sourceDirIds);

    /**
     * 目录排序（用于sort是规律的情况）
     *
     * @param id                     目录ID
     * @param directorySortUpdateDto 排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)
     * @return 操作结果
     */
    @PutMapping("/sort/{id}")
    @ApiOperation(value = "目录排序(如果sort不是规律的使用另一个接口)", notes = "sortType=排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)")
    RestResponse<Void> sortDirectory(@NotNull(message = "目录ID不允许为空") @PathVariable("id") Long id,
                                     @RequestBody DirectorySortUpdateDto directorySortUpdateDto);

    /**
     * 目录排序（兼容sort 不规律的情况）
     *
     * @param id                     目录ID
     * @param directorySortUpdateDto 排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)
     * @return 操作结果
     */
    @PutMapping("/compatible/sort/{id}")
    @ApiOperation(value = "目录排序(兼容sort 不规律的情况)", notes = "sortType=排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)")
    RestResponse<Void> compatSortDirectory(@NotNull(message = "目录ID不允许为空") @PathVariable("id") Long id,
                                           @RequestBody DirectorySortUpdateDto directorySortUpdateDto);


    /**
     * 目录排序
     *
     * @param directorySortUpdateDto 目录位移操作Dto
     * @return 操作结果
     */
    @PutMapping("/sort")
    @ApiOperation(value = "目录排序-指定交换位置的目录", notes = "sourceDirId=原目录ID,targetDirId=位移目录ID")
    RestResponse<Void> sortDirectory(@RequestBody DirectorySortUpdateDto directorySortUpdateDto);

    /**
     * 初始化目录树， 该接口不对外开放
     *
     * @return
     */
    @PutMapping(produces = "application/json", value = "/init")
    RestResponse<Void> initDirTree();

    /**
     * 删除目录项
     *
     * @param id 目录（项）ID
     * @return 统一返回响应对象
     */
    @DeleteMapping(value = "/{id}")
    @ApiOperation(value = "删除目录项,根据目录ID删除", notes = "删除目录项,根据目录ID删除 \n id:目录ID")
    RestResponse<Void> removeDirectory(@NotNull @PathVariable("id") Long id);

    /**
     * 删除目录索引
     *
     * @param indexId 目录id
     * @return 统一返回业务对象
     */
    @DeleteMapping(value = "/root/{indexId}")
    @ApiOperation(value = "删除目录索引", notes = "目录索引id")
    RestResponse<Void> removeRootDirectory(@NotNull @PathVariable("indexId") Long indexId);

    /**
     * 通过RootId删除根目录
     *
     * @param rootId 目录id
     * @return 统一返回业务对象
     */
    @DeleteMapping(value = "/root/by/{rootId}")
    @ApiOperation(value = "删除目录索引(通过rootId)", notes = "rootId")
    RestResponse<Void> removeRootDirectoryByRootId(@NotNull @PathVariable("rootId") Long rootId);


    /**
     * 保存类目下的属性组信息
     *
     * @param dirId
     * @param propGroupReqDtos
     * @return
     */
    @PostMapping("/{dirId}/propGroup")
    @ApiOperation(value = "保存类目下的属性组信息", notes = "保存类目下的属性组信息")
    RestResponse<Void> saveDirPropGroup(@PathVariable("dirId") Long dirId, @RequestBody List<PropGroupReqDto> propGroupReqDtos);

    /**
     * 增加类目下的属性组
     *
     * @param dirId
     * @param propGroupIds
     * @return
     */
    @PostMapping("/{dirId}/propGroup/add")
    @ApiOperation(value = "增加类目下的属性组", notes = "增加类目下的属性组")
    RestResponse<Void> addDirPropGroup(@PathVariable("dirId") Long dirId, @RequestBody Set<Long> propGroupIds);

    /**
     * 修改类目下的属性组用途
     *
     * @param dirId
     * @param propGroupId
     * @param usage
     * @return
     */
    @PostMapping("/{dirId}/{propGroupId}/usage")
    @ApiOperation(value = "修改类目下的属性组用途", notes = "修改类目下的属性组用途")
    RestResponse<Void> updateDirPropGroupUsage(@PathVariable("dirId") Long dirId, @PathVariable("propGroupId") Long propGroupId,
                                               @RequestParam(name = "usage", required = false) Integer usage);

    /**
     * 删除类目下的属性组
     *
     * @param dirId
     * @param propGroupIds
     * @return
     */
    @DeleteMapping("/{dirId}/propGroup")
    @ApiOperation(value = "删除类目下的属性组", notes = "删除类目下的属性组")
    RestResponse<Void> removeDirPropGroup(@PathVariable("dirId") Long dirId, @RequestBody Set<Long> propGroupIds);

    /**
     * 上移或者下移类目下的属性组顺序
     *
     * @param dirId
     * @param propGroupId
     * @param action
     * @return
     */
    @PostMapping("/{dirId}/{propGroupId}/{action}")
    @ApiOperation(value = "上移或者下移类目下的属性组顺序", notes = "上移或者下移类目下的属性组顺序")
    RestResponse<Void> updateDirPropGroupSort(@PathVariable("dirId") Long dirId, @PathVariable("propGroupId") Long propGroupId,
                                            @PathVariable("action") String action);

    /**
     * 复制类目下的属性组到指定的类目下
     *
     * @param sourceDirId
     * @param targetDirId
     * @return
     */
    @PostMapping("/{sourceDirId}/propGroup/{targetDirId}")
    @ApiOperation(value = "复制类目下的属性组到指定的类目下", notes = "复制类目下的属性组到指定的类目下")
    RestResponse<Void> copyDirPropGroup(@PathVariable("sourceDirId") Long sourceDirId, @PathVariable("targetDirId") Long targetDirId);


    @PostMapping(value = "/initShopDir")
    @ApiOperation(value = "初始化店铺类目", notes = "list")
    RestResponse<String> initShopDir(@RequestBody List<CommanderInfoMQDto> list);

}
