/*
 * @(#)IItemApi.java 1.0 2018-08-22
 *
 * 			Copyright (c) 2017, YUNXI. All rights reserved.
 * 			YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.item.api.base;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.*;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.response.ItemRespDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import javax.validation.constraints.NotNull;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * 商品接口
 *
 * @author 诺克 双清
 * @since 2.0.0
 */

@Api(tags = {"商品中心：商品服务"})
@FeignClient(name = "${yundt.cube.center.item.api.name:yundt-cube-center-item}",
        contextId="com-dtyunxi-yundt-cube-center-item-api-item-IItemApi",
        path = "/v1/item",
        url = "${yundt.cube.center.item.api:}")
public interface IItemApi {

    /**
     * 批量上架（添加）商品目录以及sku
     * （若商品是上架状态，则直接添加目录关联，若商品是下架状态，则把改为上架状态添加目录关联）
     *
     * @param itemOnShelfReqListDto 添加上架目录列表
     * @return 操作结果
     */
    @PostMapping(value = "/shelf")
    @ApiOperation(value = "上架", notes = "支持单个/批量上架")
    RestResponse<Void> onShelfItem(@Validated @NotNull @RequestBody ItemOnShelfReqListDto itemOnShelfReqListDto);


    /**
     * 批量下架商品
     *
     * @param itemOffShelfReqListDto 下架的商品列表
     * @return 操作结果
     */
    @PutMapping(value = "/shelf")
    @ApiOperation(value = "下架", notes = "支持单个/批量下架商品")
    RestResponse<Void> offShelfItem(@RequestBody ItemOffShelfReqListDto itemOffShelfReqListDto);

    /**
     * 商品发布
     *
     * @param itemPublishReqDto 商品发布的dto，现在仅有商品id，改个商品状态
     * @return 操作结果
     */
    @PutMapping (value = "/publish")
    @ApiOperation(value = "商品发布", notes = "商品发布")
    RestResponse<Void> publish(@RequestBody ItemPublishReqDto itemPublishReqDto);
    /**
     * 商品审核
     * @param itemAuditReqDto 商品审核信息
     * @return 操作结果
     */
    @PutMapping (value = "/audit")
    @ApiOperation(value = "商品审核", notes = "商品审核")
    RestResponse<Void> audit(@RequestBody ItemAuditReqDto itemAuditReqDto);
    /**
     * 删除商品(以及商品相关属性)
     *
     * @param ids 商品id
     * @return 返回long
     */
    @DeleteMapping(value = "/{ids}")
    @ApiOperation(value = "删除商品", notes = "根据id删除商品，以及商品相关属性 \t\n ids：商品Id，复数用逗号分隔")
    RestResponse<Void> removeItem(@NotNull @PathVariable(value = "ids") String ids);


    /**
     * 删除上架信息
     *
     * @param ids
     * @return
     */
    @DeleteMapping(value = "/itemShelf/{ids}")
    @ApiOperation(value = "删除上架信息", notes = "根据id删除上架信息，复数用逗号分隔")
    RestResponse<Void> removeItemShelf(@NotNull @PathVariable(value = "ids") String ids);

    /**
     * 批量下架商品
     *
     * @param itemIds 下架的商品id列表
     * @return 操作结果
     */
    @Deprecated
    @DeleteMapping(value = "/shelf/{itemIds}")
    @ApiOperation(value = "批量下架商品", notes = "批量下架商品，优先考虑使用/shelf做商品下架 \n itemIds:商品ID，复数用逗号分隔")
    RestResponse<Void> offShelfItem(@PathVariable(name = "itemIds") String itemIds);

    /**
     * 给商品打标签
     *
     * @param requestItemTags 商品标签Dto列表
     * @return ResponseDto<Long> 统一返回响应对象，业务数据data为属性名ID
     */
    @PostMapping(value = "/tag")
    @ApiOperation(value = "给商品打标签", notes = "给商品打标签")
    RestResponse<Void> addItemTag(@Validated @RequestBody List<ItemTagReqDto> requestItemTags);


    /**
     * 创建已上架商品跟目录的关系
     *
     * @param itemId    商品ID
     * @param dirIds    目录ID
     * @param shopId    店铺ID
     * @param channelId 渠道ID
     * @return 操作结果
     */
    @PostMapping("/{itemId}/dir/{dirIds}")
    @ApiOperation(value = "创建已上架商品跟目录的关系，不推荐使用，可使用商品上架功能上架到其他目录", notes = "创建已上架商品跟目录的关系")
    RestResponse<Void> addItemDir(@NotNull(message = "id不允许为空") @PathVariable("itemId") Long itemId,
                                  @NotNull(message = "目录id不允许为空") @PathVariable("dirIds") String dirIds,
                                  @RequestParam("shopId") Long shopId,
                                  @RequestParam("channelId") Long channelId);



    /**
     * 商品下架
     *
     * @param dirItemIds 目录关联关系id列表
     * @return 操作结果
     */
    @DeleteMapping(value = "/shelf/relation/{dirItemIds}")
    @ApiOperation(value = "商品下架，不推荐使用，推荐使用正常下架接口", notes = "指定商品跟目录关系id进行下架，dirItemIds=商品上架目录ID，复数用逗号分隔")
    RestResponse<Void> deleteDirItemByIds(@NotNull @RequestParam(name = "dirItemIds") String dirItemIds);

    /**
     * 删除商品标签（物理删除）
     *
     * @param id     商品ID
     * @param tagIds 标签ID
     * @return RestResponse  统一返回响应对象
     */
    @DeleteMapping(value = "/{id}/tag/{tagIds}")
    @ApiOperation(value = "删除商品标签", notes = "移除商品标签 \t\n id:商品id \t\n tagId: 标签id")
    RestResponse<Void> removeItemTag(@NotNull @PathVariable(name = "id") Long id,
                                     @NotNull @PathVariable(name = "tagIds") String tagIds);

    /**
     * 根据购物车商品ID进行失效商品校验
     *
     * @param itemIds 商品ID
     * @param filter  过滤条件
     * @return 失效商品ID
     * @See IItemQueryApi#checkShoppingCartItemInvalid
     */
    @Deprecated
    @GetMapping(value = "/{itemIds}")
    @ApiOperation(value = "根据购物车商品ID进行失效商品校验", notes = "根据购物车商品ID进行失效商品校验，itemIds=商品ID，filter=RequestDto")
    RestResponse<List<ItemRespDto>> checkShoppingCartItemInvalid(@PathVariable(name = "itemIds") String itemIds,
                                                                 @RequestParam("filter") String filter);

    /**
     * 修改定时上下架任务
     *
     * @param shelfTaskUpdateReqDtos 定时任务dto列表
     * @return 操作结果
     */
    @PutMapping(value = "/shelf/task")
    @ApiOperation(value = "修改定时上下架任务", notes = "修改定时上下架任务")
    RestResponse<Void> modifyShelfTask(@RequestBody List<ShelfTaskUpdateReqDto> shelfTaskUpdateReqDtos);

    /**
     * 删除定时上下架任务
     *
     * @param ids        定时任务id 列表
     * @param instanceId 实例ID
     * @param tenantId   租户ID
     * @return 操作结果
     */
    @DeleteMapping(value = "/shelf/task/{ids}")
    @ApiOperation(value = "删除定时上下架任务", notes = "删除定时上下架任务，ids=定时任务ID，instanceId=实例ID，tenantId=租户ID")
    RestResponse<Void> removeShelfTask(@PathVariable("ids") List<Long> ids,
                                       @RequestParam("instanceId") Long instanceId,
                                       @RequestParam("tenantId") Long tenantId);

    /**
     * 设置单位时间内商品的销售数量
     *
     * @param itemStatReqDto 请求参数
     * @return 操作结果
     */
    @PostMapping("/stat/sales")
    @ApiOperation(value = "设置单位时间内商品的销售数量", notes = "设置单位时间内商品的销售数量")
    RestResponse<Void> addOrModifyItemSalesCount(@RequestBody ItemStatReqDto itemStatReqDto);


    /**
     * 批量修改商品状态
     *
     * @param  modifyStatusReqDto 批量修改商品状态
     * @return 操作结果
     */
    @PutMapping(value = "/status")
    @ApiOperation(value = "批量修改商品状态", notes = "批量修改商品状态")
    RestResponse<Void> batchModifyItemStatus(@RequestBody ItemModifyStatusReqDto modifyStatusReqDto);


    /**
     * 手动调接口同步到es
     *
     * @param
     * @return 手动调接口同步到es
     */
    @GetMapping(value = "/newSyncEs/es")
    @ApiOperation(value = "手动调接口同步到es", notes = "手动调接口同步到es")
    RestResponse<Void> newSyncEs(ShelfQueryReqDto shelfQueryReqDto);

    /**
     * 手动调接口同步到es
     *
     * @param
     * @return 手动调接口同步到es
     */
    @GetMapping(value = "/sync/es")
    @ApiOperation(value = "手动调接口同步到es", notes = "手动调接口同步到es")
    RestResponse<Void> syncEs();

    /**
     * 新增商品
     *
     * @param changeApplyDto
     * @return 操作结果
     */
    @PostMapping(value = "/add")
    @ApiOperation(value = "新增", notes = "新增")
    RestResponse<Long> addItem(@Validated @NotNull @RequestBody ItemChangeApplyDto changeApplyDto);

    /**
     * 修改商品
     *
     * @param itemReqDto
     * @return 操作结果
     */
    @PostMapping(value = "/modify")
    @ApiOperation(value = "修改", notes = "修改")
    RestResponse<Void> modifyItem(@Validated @NotNull @RequestBody ItemReqDto itemReqDto);

    /**
     * 批量修改上架信息的店铺授权标识
     *
     * @param  modifyShopAuthReqDto 批量修改上架信息的店铺授权标识
     * @return 操作结果
     */
    @PutMapping(value = "/shelf/shop-auth")
    @ApiOperation(value = "批量修改上架信息的店铺授权标识", notes = "批量修改上架信息的店铺授权标识")
    RestResponse<Void> batchModifyShelfShopAuth(@RequestBody ShelfModifyShopAuthReqDto modifyShopAuthReqDto);

}
