/*
 * @(#)IDirectoryItemQueryApi.java 1.0 2018-08-31
 *
 * 			Copyright (c) 2017, YUNXI. All rights reserved.
 * 			YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.item.api.base.query;

import com.dtyunxi.cube.commons.dto.TreeDto;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.DirTreeReqDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.DirectoryReqDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.request.RootDirectoryReqDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.response.DirIndexRespDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.response.DirectoryItemRespDto;
import com.dtyunxi.yundt.cube.center.item.api.base.dto.response.PropGroupRespDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 目录项查询Api
 *
 * @author 诺克
 * @since 2.0.0
 */
@Api(tags = {"商品中心：目录服务接口"})
@FeignClient(name = "${yundt.cube.center.item.api.name:yundt-cube-center-item}",
        contextId = "com-dtyunxi-yundt-cube-center-item-api-base-query-IDirectoryQueryApi",
        path = "/v1/dir",
        url = "${yundt.cube.center.item.api:}")
public interface IDirectoryQueryApi {

    /**
     * 根据目录id查询目录
     *
     * @param id 目录Id
     * @return 目录
     */
    @GetMapping(value = "/{id}")
    @ApiOperation(value = "根据目录id查询目录", notes = "根据目录id查询目录信息及引用目录信息")
    RestResponse<DirectoryItemRespDto> queryDirById(@PathVariable("id") Long id);

    /**
     * 根据商品id、店铺id查询商品店铺类目
     * @param itemId
     * @param shopId
     * @param busType
     * @param dirType
     * @return
     */
    @GetMapping(value = "/queryShopDir")
    @ApiOperation(value = "根据商品id、店铺id查询商品店铺类目", notes = "商品店铺类目查询接口")
    RestResponse<List<DirectoryItemRespDto>> queryShopDirList(@RequestParam("itemId") Long itemId,
                                                    @RequestParam("shopId") Long shopId,
                                                    @RequestParam("busType") Integer busType,
                                                    @RequestParam("dirType") String dirType);

    /**
     * 根据条件查询目录 filter -> DirectoryReqDto
     *
     * @param filter 查询条件
     * @return 目录集合
     */
    @GetMapping("")
    @ApiOperation(value = "根据条件查询目录", notes = "目录查询接口,filter=DirectoryReqDto")
    RestResponse<List<DirectoryItemRespDto>> queryDirectoryItem(@RequestParam("filter") String filter);

    /**
     * 根据条件查询目录树 filter -> RootDirectoryReqDto
     *
     * @param filter 查询条件
     * @return 目录树集合
     */
    @GetMapping("/tree")
    @ApiOperation(value = "根据条件查询目录树,不会返回目录索引信息", notes = "目录查询接口:rootId:根目录ID集合：{dirUsage=?},filter=RootDirectoryReqDto")
    RestResponse<List<TreeDto<DirectoryItemRespDto>>> queryDirectoryTreeByFilter(@RequestParam("filter") String filter);

    /**
     * 根据rootId查询目录树 filter -> BaseReqDto
     *
     * @param id     目录ID
     * @param filter 数据过滤
     * @return 目录树
     */
    @GetMapping("/{id}/tree")
    @ApiOperation(value = "根据rootId查询目录树", notes = "目录查询接口:rootId:根目录ID集合：{rootId=?}，filter=DirectoryReqDto")
    RestResponse<TreeDto<DirectoryItemRespDto>> queryDirectoryTree(@NotNull @PathVariable("id") Long id,
                                                                   @RequestParam("filter") String filter);

    /**
     * 根据入参查询目录索引，再根据目录索引查询树结构
     *
     * @param filter 查询条件
     * @return 目录树集合
     */
    @GetMapping("/tree/detail")
    @ApiOperation(value = "根据条件查询目录树，会返回目录索引信息", notes = "返回的目录树会附带目录索引的信息，filter=RootDirectoryReqDto")
    RestResponse<List<TreeDto<DirIndexRespDto>>> queryDirTreeDetail(@RequestParam("filter") String filter);

    /**
     * 根据目录id列表批量查询类目
     *
     * @param dirIds 类目id列表
     * @return 目录列表
     * @author wu.sheng
     * @date 2019/11/25 14:58
     */
    @GetMapping(value = "/ids")
    @ApiOperation(value = "根据目录id列表批量查询目录", notes="批量查询，多个逗号隔开")
    RestResponse<List<DirectoryItemRespDto>> queryDirByIds(@RequestParam("dirIds") String dirIds);


    /**
     * 查询类目下的属性组信息
     *
     * @param dirId
     * @return
     */
    @GetMapping("{dirId}/propGroup")
    @ApiOperation(value = "查询类目下的属性组信息", notes = "查询类目下的属性组信息")
    RestResponse<List<PropGroupRespDto>> queryPropGroupListByDirId(@PathVariable("dirId") Long dirId, @RequestParam(value = "usage", required = false) Integer usage);


    /**
     * 查询类目的基本信息
     *
     * @param id
     * @return
     */
    @GetMapping(value = "/base/ids")
    @ApiOperation(value = "查询目录基本信息", notes = "查询目录基本信息")
    RestResponse<DirectoryItemRespDto> queryBaseInfo(@PathVariable("id") Long id);

    /**
     * 根据条件查询目录树
     *
     * @param dirTreeReqDto
     * @return
     */
    @PostMapping("/base/tree")
    @ApiOperation(value = "根据条件查询目录树,不会返回目录索引信息", notes = "根据条件查询目录树,不会返回目录索引信息，有缓存")
    RestResponse<List<TreeDto<DirectoryItemRespDto>>> queryDirTree(@RequestBody DirTreeReqDto dirTreeReqDto);
    @GetMapping(value = "/getfront/byId")
    @ApiOperation(value = "根据后台id查找前台类目")
    RestResponse<List<TreeDto<DirectoryItemRespDto>>>queryDirectoryTreeById(@RequestParam("id")Long id,
                                                                            @RequestParam("dirUsage")String dirUsage,
                                                                            @RequestParam(value = "shopId",required = false)Long shopId);

    @GetMapping(value = "/getDirId/byBackId")
    @ApiOperation(value = "根据后台id查找前台类目")
    RestResponse<List<Long>> queryListDirById(@RequestParam("id")Long id,
                                               @RequestParam("dirUsage")String dirUsage);

    @GetMapping(value = "/getDirId/byBackIdOrShopDirId")
    @ApiOperation(value = "根据后台id查找前台类目和店铺类目")
    RestResponse<List<Long>> queryFontDirIdsByBackId(@RequestParam("id")Long id, @RequestParam("dirUsage")String dirUsage, @RequestParam(value = "shopId",required = false)Long shopId);

    @GetMapping(value = "/getBack/byId")
    @ApiOperation(value = "根据前台、店铺id查找后台类目名称")
    List<Long> selectBackNameById(@RequestParam("id") Long id);


    /**
     * 批量查询层级信息，返回格式如下[一级，二级，当前类目层级]
     * @param ids
     * @return
     */
    @PostMapping(value = "/levels")
    @ApiOperation(value = "批量查询目录层级信息", notes = "批量查询目录层级信息,响应格式list[list[一级，二级，当前类目]]")
    RestResponse<List<List<DirectoryItemRespDto>>> queryLevels(@RequestBody List<Long> ids);

    /**
     * 查询指定目录id的下级目录(包含自身)
     * @author 惜笔 | 林五弟
     * @date 2021/9/23 18:31
     * @param directoryReqDto           目录ids和租户，实例
     * @return com.dtyunxi.rest.RestResponse<java.util.List < java.lang.Long>>
     */
    @PostMapping(value = "/sub-dir-list")
    @ApiOperation(value = "查询指定目录id的下级目录(包含自身)", notes = "查询指定目录id的下级目录(包含自身)，返回所有的目录id")
    RestResponse<List<Long>> queryLinkAndChildDirIdListByDirId(@RequestBody DirectoryReqDto directoryReqDto);
}
