package com.dtyunxi.yundt.cube.center.inventory.utils;

import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * BigDecimal 工具类
 * @author wufan
 */
public class BigDecimalUtils {

    public static final BigDecimal ONE_HUNDRED = BigDecimalUtils.parse("100");

    /**
     * 加法 a + b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static BigDecimal add (BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.add(b);
    }

    /**
     * 减法 a - b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static BigDecimal subtract (BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.subtract(b);
    }

    /**
     * 乘法 a * b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static BigDecimal multiply(BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.multiply(b);
    }

    /**
     * 除法 a / b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static BigDecimal divide (BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        if (eqZero(b)) {
           return BigDecimal.ZERO;
        }
        return a.divide(b, 0, RoundingMode.HALF_UP);
    }

    /**
     * 除法 a / b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static BigDecimal divide (BigDecimal a, BigDecimal b, int scale) {
        a = check(a);
        b = check(b);
        if (eqZero(b)) {
            return BigDecimal.ZERO;
        }
        return a.divide(b, scale, RoundingMode.HALF_UP);
    }

    /**
     * 等于 a = b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static Boolean equals(BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.compareTo(b) == 0;
    }

    /**
     * 不等于 a != b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static Boolean notEquals(BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return !equals(a, b);
    }

    /**
     * 大于 a > b
     * @param a 入参
     * @param b 入参
     * @return BigDecimal
     */
    public static Boolean gt (BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.compareTo(b) > 0;
    }

    /**
     * a <= 0
     * @param a 入参
     * @return Boolean
     */
    public static Boolean leZero (BigDecimal a) {
        a = check(a);
        return le(a, BigDecimal.ZERO);
    }

    /**
     * a < 0
     * @param a 入参
     * @return Boolean
     */
    public static Boolean ltZero (BigDecimal a) {
        a = check(a);
        return lt(a, BigDecimal.ZERO);
    }

    /**
     * a == 0
     * @param a 入参
     * @return Boolean
     */
    public static Boolean eqZero (BigDecimal a) {
        a = check(a);
        return equals(a, BigDecimal.ZERO);
    }

    /**
     * a >= 0
     * @param a 入参
     * @return Boolean
     */
    public static Boolean geZero (BigDecimal a) {
        a = check(a);
        return ge(a, BigDecimal.ZERO);
    }

    /**
     * a > 0
     * @param a 入参
     * @return Boolean
     */
    public static Boolean gtZero (BigDecimal a) {
        a = check(a);
        return gt(a, BigDecimal.ZERO);
    }

    /**
     * 小于 a < b
     * @param a 入参
     * @return Boolean
     */
    public static Boolean lt (BigDecimal a, BigDecimal b) {
        a = check(a);
        b = check(b);
        return a.compareTo(b) < 0;
    }

    /**
     * 大于等于 a >= b
     * @param a 入参
     * @param b 入参
     * @return Boolean
     */
    public static Boolean ge (BigDecimal a, BigDecimal b) {
        return equals(a, b) || gt(a, b);
    }

    /**
     * 小于等于 a <= b
     * @param a 入参
     * @param b 入参
     * @return Boolean
     */
    public static Boolean le (BigDecimal a, BigDecimal b) {
        return equals(a, b) || lt(a, b);
    }

    /**
     * 检查空值 if (a == null) a = BigDecimal.ZERO
     * @param a 入参
     * @return Boolean
     */
    public static BigDecimal check (BigDecimal a) {
        return null == a ? BigDecimal.ZERO : a;
    }

    /**
     * 取绝对值
     * @param a 入参
     * @return Boolean
     */
    public static BigDecimal abs (BigDecimal a) {
        a = check(a);
        return a.abs();
    }

    /**
     * 取负
     * @param a 入参
     * @return BigDecimal
     */
    public static BigDecimal negate(BigDecimal a) {
        a = check(a);
        return multiply(a, parse("-1"));
    }

    /**
     * 字符串转换 BigDecimal
     * @param a 入参
     * @return BigDecimal
     */
    public static BigDecimal parse (String a) {
        if (StringUtils.isBlank(a)) {
            return BigDecimal.ZERO;
        }
        return new BigDecimal(a);
    }
}
