package com.tcbj.framework.dto.util;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.List;


/**
 * Json工具类,封装了jackson框架爱
 * @author huanghj2
 *
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public final class Jsons {
 	
	private Jsons() { }
	private static JsonFactory jsonFactory = new JsonFactory();
	
	private static ObjectMapper mapper = null;
	
	static {
		jsonFactory.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
		jsonFactory.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
		mapper = new ObjectMapper(jsonFactory);
		mapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);//设置反序列化时遇到未知属性时不报错
	}	
 
	/**
	 * 将json转成java bean
	 * @param <T> - 多态类型
	 * @param json - json字符串
	 * @param clazz - java bean类型(Class)
	 * @return - java bean对象
	 */
	public static <T> T toBean(String json, Class clazz) {
		
		T rtv = null;
		try { 
			rtv = (T) mapper.readValue(json, clazz); 
		} catch (Exception ex) {   
			String cause = String.format("json to bean error:[%s],json:[%s]", ex.getMessage(), json); 
			throw new IllegalArgumentException(cause, ex);
		}
		return (T)rtv;		
	}

	public static <T> T toBean(String json, TypeReference<T> clazz) {

		T rtv = null;
		try {
			rtv = (T) mapper.readValue(json, clazz);
		} catch (Exception ex) {
			String cause = String.format("json to bean error:[%s],json:[%s]", ex.getMessage(), json);
			throw new IllegalArgumentException(cause, ex);
		}
		return (T)rtv;
	}

	/**
	 * 将输入流转成java bean
	 * @param <T> - 多态类型
	 * @param in - 输入流
	 * @param clazz - java bean类型(Class)
	 * @return - java bean对象
	 */
	public static <T> T toBean(InputStream in, Class clazz) {
		
		T rtv = null;
		try {  
			rtv = (T)mapper.readValue(in, clazz); 
		} catch (Exception ex) {  
			throw new IllegalArgumentException("json-stream to bean error", ex);
		}
		return (T)rtv;		
	}
	
	/**
	 * 将java bean转成json
	 * @param bean - java bean
	 * @return - json 字符串
	 */
	public static String toJson(Object bean) {
		
		String rtv = null;
		try { 
			rtv = mapper.writeValueAsString(bean);
		} catch (Exception ex) {  
			String cause = String.format("bean to json error:[%s],bean-class:[%s]", ex.getMessage(), bean.getClass()); 
			throw new IllegalArgumentException(cause, ex);
		}
		return rtv;		
	}
	
	/**
	 * 将Json字符串转为List JavaBean 
	 * @param json	json格式字符串
	 * @param clazz	java bean类型(Class)
	 * @return java bean对象
	 */
	public static <T> List<T> toListBean(String json,Class<T> clazz){
		List<T>  result = null;
		try {
			result = mapper.readValue(json, getCollectionType(List.class, clazz));
		} catch (Exception ex) {
			String cause = String.format("json to bean error:[%s],json:[%s]", ex.getMessage(), json); 
			throw new IllegalArgumentException(cause, ex);
		}
		return result;
	}

	private static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
		return mapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
	}
}