package com.dtyunxi.tcbj.api.biz.inventory;

import com.dtyunxi.tcbj.api.biz.inventory.dto.QueryExternalSystemCodeReq;
import com.dtyunxi.tcbj.api.biz.inventory.dto.QueryExternalSystemCodeResp;
import com.dtyunxi.tcbj.api.biz.inventory.dto.ExternalSystemCodeDto;
import lombok.extern.log4j.Log4j2;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Log4j2
public class ItemQuery2025Service {

    @Resource
    private JdbcTemplate jdbcTemplate;

    public QueryExternalSystemCodeResp queryItemExternalCode(QueryExternalSystemCodeReq req) {
        log.info("根据仓库编码和SAP商品编码查询外围系统编码，请求参数：{}", req);

        String warehouseCode = req.getWarehouseCode();
        List<String> skuCodes = req.getSkuCodes();
        if (warehouseCode == null) {
            return QueryExternalSystemCodeResp.builder()
                    .externalSystemCodeList(Collections.emptyList())
                    .build();
        }

        log.info("查询的仓库编码：{}，SAP商品编码列表：{}", warehouseCode, skuCodes);

        List<ExternalSystemCodeDto> externalSystemCodeList = queryExternalSystemCodeFromDatabase(warehouseCode, skuCodes);

        log.info("查询完成，返回结果数量：{}", externalSystemCodeList.size());

        return QueryExternalSystemCodeResp.builder()
                .externalSystemCodeList(externalSystemCodeList)
                .build();
    }

    /**
     * 根据仓库编码和SAP商品编码查询外围系统编码
     */
    private List<ExternalSystemCodeDto> queryExternalSystemCodeFromDatabase(String warehouseCode, List<String> skuCodes) {
        StringBuilder sqlBuilder = new StringBuilder();
        sqlBuilder.append("SELECT auth.id AS authId, ")
                  .append("       shop.code AS tenantId, ")
                  .append("       shop.name AS tenantName, ")
                  .append("       warehouse.warehouse_code AS warehouseCode, ")
                  .append("       warehouse.warehouse_name AS warehouseName, ")
                  .append("       sku.code AS skuCode, ")
                  .append("       auth.external_code AS externalCode ")
                  .append("FROM it_item_auth auth ")
                  .append("INNER JOIN it_item_sku sku ON sku.id = auth.business_id AND sku.dr = 0 ")
                  .append("INNER JOIN sc_shop shop ON shop.id = auth.target_id AND target_type = '2' AND shop.dr = 0 ")
                  .append("INNER JOIN cs_logic_warehouse warehouse ON warehouse.organization_id = shop.organization_id AND warehouse.dr = 0 ")
                  .append("WHERE auth.dr = 0 ")
                  .append("  AND warehouse.warehouse_code = ? ");
        
        // 如果skuCodes不为null且不为空，才拼接SKU条件
        if (skuCodes != null && !skuCodes.isEmpty()) {
            String inSql = skuCodes.stream()
                    .map(skuCode -> "'" + skuCode + "'")
                    .collect(Collectors.joining(","));
            sqlBuilder.append("  AND sku.code IN (").append(inSql).append(")");
        }

        String sql = sqlBuilder.toString();

        return jdbcTemplate.query(sql, new Object[]{warehouseCode}, (rs, rowNum) -> {
            ExternalSystemCodeDto dto = new ExternalSystemCodeDto();
            dto.setAuthId(rs.getLong("authId"));
            dto.setTenantId(rs.getString("tenantId"));
            dto.setTenantName(rs.getString("tenantName"));
            dto.setWarehouseCode(rs.getString("warehouseCode"));
            dto.setWarehouseName(rs.getString("warehouseName"));
            dto.setSkuCode(rs.getString("skuCode"));
            dto.setExternalCode(rs.getString("externalCode"));
            return dto;
        });
    }
}
