/**
 * @(#)IResourceApi.java 1.0 June 17, 2018
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api;


import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.ButtonDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.MenuDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.RegisterChannelDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.UserGroupDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.ResourceImportRespDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;

/*
 * 用户资源管理接口涉及内容：
 * 用户使用应用时创建用户实例，这个用户就是应用实例的创建者，我们可以：
 * 1、通过指定的应用实例和创建者，为其新增注册渠道，记录相应的信息（如：渠道名称，渠道编码，渠道类型等）
 * 2、修改某条指定的注册渠道
 * 3、删除某条指定的注册渠道
 *
 * 4、用户组的相关操作跟以上类似
 * */

/**
 * 用户资源管理API接口
 *
 * @author 伊泽
 * @since 1.0.0
 */
@Api(tags = {"用户中心：资源管理"})
@FeignClient(contextId="com-dtyunxi-yundt-cube-center-user-api-IResourceApi",path = "/v1/resource", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IResourceApi {


    /**
     * 新增us_resource资源
     *
     * @param reqDto us_resource资源请求对象
     * @return 处理结果
     */
    @PostMapping("")
    @ApiOperation(value = "新增资源", notes = "新增资源")
    @Capability(capabilityCode = "user.resource.add-resource")
    RestResponse<Long> addResource(@RequestBody @Valid ResourceCreateReqDto reqDto);

    /**
     * 修改us_resource资源
     *
     * @param reqDto us_resource资源请求对象
     * @return 处理结果
     */
    @PutMapping("")
    @ApiOperation(value = "修改资源", notes = "根据id或code修改资源")
    @Capability(capabilityCode = "user.resource.modify-resource")
    RestResponse<Void> modifyResource(@RequestBody ResourceUpdateReqDto reqDto);

    /**
     * 删除us_resource资源
     *
     * @param ids us_resource资源删除数据ID
     * @return 处理结果
     */
    @DeleteMapping("/{ids}")
    @ApiOperation(value = "删除资源", notes = "删除资源,多个id使用英文逗号分隔")
    @Capability(capabilityCode = "user.resource.remove-resource")
    RestResponse<Void> removeResource(@PathVariable("ids") String ids);

    /**
     * 菜单排序
     *
     * @param sortReqDto 排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)
     * @return 操作结果
     */
    @PutMapping("/sort")
    @ApiOperation(value = "菜单排序", notes = "sortType=排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)")
    @Capability(capabilityCode = "user.resource.sort-resource-menu")
    RestResponse<Void> sortResourceMenu(@RequestBody SortReqDto sortReqDto);

    /**
     * 导入资源
     * @param dtoList 批量导入资源集合
     * @return
     */
    @PostMapping("/import")
    @ApiOperation(value = "导入资源", notes = "导入资源")
    @Capability(capabilityCode = "user.resource.import-resources")
    RestResponse<ResourceImportRespDto> importResources(@RequestBody @Valid List<ResourceImportReqDto> dtoList);

    /**
     * 注册资源
     * @param resourceRegisterDto
     * @return
     */
    @PostMapping("/register")
    @ApiOperation(value = "注册资源", notes = "注册资源")
    @Capability(capabilityCode = "user.resource.register-resources")
    RestResponse<Void> registerResources(@RequestBody @Valid ResourceRegisterDto resourceRegisterDto);


    /**
     * 批量添加注册渠道，分配ID；
     *
     * @param instanceId     应用实例Id
     * @param userId         创建者Id
     * @param channelDtoList 批量注册渠道
     * @return List
     */
    @RequestMapping(value = "/user/{userId}/channel", produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "新增注册渠道", notes = "新增注册渠道")
    @Capability(capabilityCode = "user.resource.add-register-channel")
    RestResponse<List<Long>> addRegisterChannel(@NotNull @RequestParam(value = "instanceId") Long instanceId,
                                                @NotNull @PathVariable("userId") Long userId,
                                                @RequestBody List<RegisterChannelDto> channelDtoList);

    /**
     * 修改注册渠道
     *
     * @param registerChannelDto 注册通道Dto
     * @param channelId          注册渠道的ID
     * @return Long
     */
    @RequestMapping(value = "/channel/{channelId}", produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "修改注册渠道", notes = "修改注册渠道")
    @Capability(capabilityCode = "user.resource.update-register-channel")
    RestResponse<Long> updateRegisterChannel(@NotNull @PathVariable("channelId") Long channelId,
                                             @RequestBody RegisterChannelDto registerChannelDto);

    /**
     * 删除注册渠道
     *
     * @param channelId 注册渠道的Id
     * @return Long
     */
    @RequestMapping(value = "/channel/{channelId}", produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "删除注册渠道", notes = "删除注册渠道")
    @Capability(capabilityCode = "user.resource.delete-register-channel")
    RestResponse<Long> deleteRegisterChannel(@NotNull @PathVariable("channelId") Long channelId);

    /**
     * 新增用户组
     *
     * @param instanceId   应用实例的ID
     * @param userId       创建者Id
     * @param userGroupDto 用户组
     * @return Long
     */
    @RequestMapping(value = "/user/{userId}/user-group", produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "新增用户组", notes = "新增用户组")
    @Capability(capabilityCode = "user.resource.add-user-group")
    RestResponse<Long> addUserGroup(@NotNull @RequestParam(value = "instanceId") Long instanceId,
                                    @NotNull @PathVariable("userId") Long userId,
                                    @RequestBody UserGroupDto userGroupDto);

    /**
     * 修改用户组
     *
     * @param userGroupDto 用户组
     * @param groupId      用户组的ID
     * @return Long
     */
    @RequestMapping(value = "/user-group/{groupId}", produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "修改用户组", notes = "修改用户组")
    @Capability(capabilityCode = "user.resource.update-user-group")
    RestResponse<Long> updateUserGroup(@NotNull @PathVariable("groupId") Long groupId, @RequestBody UserGroupDto userGroupDto);

    /**
     * 删除用户组
     *
     * @param groupId 用户组的ID
     * @return Long
     */
    @RequestMapping(value = "/user-group/{groupId}", produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "删除用户组", notes = "删除用户组")
    @Capability(capabilityCode = "user.resource.delete-user-group")
    RestResponse<Long> deleteUserGroup(@NotNull @PathVariable("groupId") Long groupId);

    /**
     * 添加按钮信息,传入应用实例id，按钮信息表保存按钮的名称，编码，状态，所属应用的id信息
     * 如果要限制应用实例下的资源范围，可以通过套餐管理，参考套餐相关API
     *
     * @param instanceId 应用实例ID
     * @param menuId     菜单Id
     * @param buttonDto  按钮信息
     * @return Long 新增的按钮ID
     */
    @Deprecated
    @RequestMapping(value = "/menu/{menuId}/button", method = RequestMethod.POST)
    @ApiOperation(value = "添加按钮信息, 需要有菜单才能添加按钮", notes = "如果要限制应用实例下的资源范围，可以通过套餐管理，参考套餐相关API")
    @Capability(capabilityCode = "user.resource.add-button")
    RestResponse<Long> addButton(@NotNull @RequestParam(value = "instanceId") Long instanceId,
                                 @PathVariable("menuId") Long menuId,
                                 @RequestBody ButtonDto buttonDto);

    /**
     * 根据按钮id修改按钮信息，按钮的名称，编码，状态；
     * 修改按钮信息
     *
     * @param id        按钮id
     * @param buttonDto 按钮信息
     * @return long
     */
    @RequestMapping(value = "/button/{id}", method = RequestMethod.PUT)
    @ApiOperation(value = "修改按钮信息", notes = "修改按钮信息")
    @Capability(capabilityCode = "user.resource.modify-button")
    RestResponse<Long> modifyButton(@NotNull @PathVariable("id") Long id,
                                    @RequestBody ButtonDto buttonDto);

    /**
     * 根据按钮id删除按钮信息
     * 删除按钮信息
     *
     * @param instanceId 应用实例ID
     * @param id         按钮ID
     * @return void
     */
    @RequestMapping(value = "/button/{id}", method = RequestMethod.DELETE)
    @ApiOperation(value = "删除按钮信息", notes = "删除按钮信息")
    @Capability(capabilityCode = "user.resource.remove-button")
    RestResponse<Void> removeButton(@NotNull @RequestParam(value = "instanceId") Long instanceId, @NotNull @PathVariable("id") Long id);

    /**
     * 传入一个节点，包括菜单的基本信息，parentId为空则为根节点，否则为子节点，传入的children可空可不为空。
     * 添加菜单信息
     * 如果要限制应用实例下的资源范围，可以通过套餐管理，参考套餐相关API
     *
     * @param instanceId 应用实例ID
     * @param menuDto    菜单
     * @return Long 新增的菜单ID
     */
    @Deprecated
    @RequestMapping(value = "/menu", method = RequestMethod.POST)
    @ApiOperation(value = "添加菜单信息", notes = "如果要限制应用实例下的资源范围，可以通过套餐管理，参考套餐相关API")
    @Capability(capabilityCode = "user.resource.add-menu")
    RestResponse<Long> addMenu(@NotNull @RequestParam(value = "instanceId") Long instanceId, @RequestBody MenuDto menuDto);

    /**
     * 根据菜单id修改菜单信息，包括父节点以及基本信息。
     *
     * @param id      菜单id
     * @param menuDto 菜单
     * @return Long
     */
    @RequestMapping(value = "/menu/{id}", method = RequestMethod.PUT)
    @ApiOperation(value = "修改菜单信息", notes = "修改菜单信息")
    @Capability(capabilityCode = "user.resource.modify-menu")
    RestResponse<Long> modifyMenu(@NotNull @PathVariable("id") Long id, @RequestBody MenuDto menuDto);

    /**
     * 删除菜单信息
     *
     * @param instanceId 应用实例Id
     * @param id         菜单ID
     * @return void
     */
    @RequestMapping(value = "/menu/{id}", method = RequestMethod.DELETE)
    @ApiOperation(value = "删除菜单信息", notes = "删除菜单信息, 菜单下存在子节点时, 无法删除")
    @Capability(capabilityCode = "user.resource.remove")
    RestResponse<Void> remove(@NotNull @RequestParam(value = "instanceId") Long instanceId, @NotNull @PathVariable("id") Long id);

    /**
     * 解除用户组角色
     */
    @RequestMapping(value = {"/user-group/{groupId}/roles/{roleIds}"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "解除用户组角色", notes = "解除用户组角色")
    @Capability(capabilityCode = "user.resource.remove-user-roles")
    RestResponse<Void> removeUserRoles(@PathVariable("groupId") Long groupId, @PathVariable("roleIds") Long[] roleIds);

}
