/**
 * @(#)IUserInfoQueryApi.java 1.0 June 13, 2018
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api.query;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.AddressDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.BillInfoDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.ContactsInfoDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.DeviceDto;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 用户信息查询API接口
 *
 * @author 伊泽
 * @since 1.0.0
 */
@Api(tags = {"用户中心：用户信息查询"})
@FeignClient(contextId="com-dtyunxi-yundt-cube-center-user-api-query-IUserInfoQueryApi",path = "/v1/user-info", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IUserInfoQueryApi {

    /**
     * 根据地址ID查询地址的信息
     *
     * @param addressId 地址ID
     * @param filter    tenantId, instanceId通过filter传递
     * @return AddressDto
     */
    @RequestMapping(value = "/address/{addressId}", produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询地址信息", notes = "根据地址Id查询指定的地址信息 \t\n addressId:用户地址ID，filter里传userId可以走分库键，提高查询效率")
    @Capability(capabilityCode = "user.user-info.query-address-by-id")
    RestResponse<AddressDto> queryAddressById(@NotNull @PathVariable("addressId") Long addressId, @RequestParam("filter") String filter);

    /**
     * 根据开票ID查询开票的信息
     *
     * @param billInfoId 开票ID
     * @param filter     tenantId, instanceId通过filter传递
     * @return BillInfoDto
     */
    @RequestMapping(value = "/bill/{billInfoId}", produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询开票信息", notes = "根据开票Id查询指定的开票信息\t\n billInfoId:用户开票ID，filter里传userId可以走分库键，提高查询效率")
    @Capability(capabilityCode = "user.user-info.query-bill-info-by-id")
    RestResponse<BillInfoDto> queryBillInfoById(@NotNull @PathVariable("billInfoId") Long billInfoId, @RequestParam("filter") String filter);


    @RequestMapping(value = "/bill/list", method = RequestMethod.GET)
    @ApiOperation(value = "查询开票信息列表", notes = "查询开票信息列表（分页）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "example", value = "example=BillInfoDto，查询条件json字符串,例如 {username:\"张三\"}，,该参数内容会合并到filter参数", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "filter=BillInfoDto，过滤条件", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    @Capability(capabilityCode = "user.user-info.query-bill-info-by-example")
    RestResponse<PageInfo<BillInfoDto>> queryBillInfoByExample(@RequestParam(name = "example", required = false) String example,
                                                     @RequestParam("filter") String filter,
                                                     @RequestParam("pageNum") Integer pageNum,
                                                     @RequestParam("pageSize") Integer pageSize);

    /**
     * 根据联系人ID查询联系人的信息
     *
     * @param contactId 联系人ID
     * @param filter    tenantId, instanceId通过filter传递
     * @return ContactsInfoDto
     */
    @RequestMapping(value = "/contact/{contactId}", produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询联系人", notes = "根据联系人Id查询指定的联系人信息\t\n contactId:用户联系人ID，filter里传userId可以走分库键，提高查询效率")
    @Capability(capabilityCode = "user.user-info.query-contact-by-id")
    RestResponse<ContactsInfoDto> queryContactById(@NotNull @PathVariable("contactId") Long contactId, @RequestParam("filter") String filter);



    /**
     * 查询用户的联系人列表
     *
     * @param userId   用户ID
     * @param example  查询条件json字符串
     * @param filter   查询条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @param sort     排序字段
     * @return PageInfo
     */
    @RequestMapping(value = "/{userId}/contact/list", method = RequestMethod.GET)
    @ApiOperation(value = "查询用户的联系人列表", notes = "查询某个用户下的联系人列表（分页）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "Long", paramType = "path", required = true),
            @ApiImplicitParam(name = "example", value = "查询条件json字符串,例如 {username:\"张三\"}，,该参数内容会合并到filter参数", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "过滤条件", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "sort", value = "排序字段，格式如:age_desc,name_asc", dataType = "string", paramType = "query")})
    @Capability(capabilityCode = "user.user-info.query-contacts-by-example")
    RestResponse<PageInfo<ContactsInfoDto>> queryContactsByExample(@NotNull @PathVariable("userId") Long userId,
                                                                   @RequestParam(name = "example", required = false) String example,
                                                                   @RequestParam("filter") String filter,
                                                                   @RequestParam("pageNum") Integer pageNum,
                                                                   @RequestParam("pageSize") Integer pageSize,
                                                                   @RequestParam(name = "sort", required = false) String sort);

    /**
     * 查询用户所拥有的设备信息
     *
     * @param userId 用户ID
     * @return 设备信息列表
     */
    @RequestMapping(value = "{userId}/device/list", method = RequestMethod.GET)
    @ApiOperation(value = "查询用户所拥有的设备信息", notes = "根据用户id查询设备信息")
    @Capability(capabilityCode = "user.user-info.query-device-by-id")
    RestResponse<List<DeviceDto>> queryDeviceById(@PathVariable("userId") Long userId);
}
