/**
 * @(#) IUserQueryApi.java 1.1.0 2018-06-13 Copyright (c) 2018, YUNXI. All rights reserved. YUNXI
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api.query;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.request.UserQueryDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.TenantManagerRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.UserRoleDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.BindingsQueryRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.vo.UserVo;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.List;


/**
 * 用户账户查询API接口
 *
 * @author 吉米
 * @since 1.0.0
 */
@Api(tags = {"用户中心：用户账户查询服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-query-IUserQueryApi", path = "/v1/user", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IUserQueryApi {
    /**
     * 根据登录名查询用户
     *
     * @param loginStr   登录名
     * @param instanceId 实例Id
     * @param filter     tenantId, instanceId通过filter传递
     * @return RestResponse&lt;UserDto&gt;
     */
    @RequestMapping(value = {"/{loginStr}/app/{instanceId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据登录名查询用户", notes = "根据登录名查询用户，loginStr：登录名 \t\n instanceId：实例ID， filter：该查询参数存帐号域domain信息 \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-login-str")
    RestResponse<UserDto> queryByLoginStr(@PathVariable("loginStr") String loginStr, @PathVariable("instanceId") Long instanceId, @RequestParam("filter") String filter);

    /**
     * 根据登录名、登录名类型和数据范围查询一个用户信息
     * loginNameType值为LoginNameType<br>
     * 且User的Status为1的
     *
     * @param loginName     登录名
     * @param loginNameType 登录名类型，1 name,2 phone,3 email
     * @param tenantId      租户ID
     * @param instanceId    实例ID
     * @return 用户信息Dto
     */
    @Deprecated
    @RequestMapping(value = {"/loginName/{loginNameType}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据登录名、登录名类型和数据范围查询一个用户信息", notes = "根据登录名、登录名类型和数据范围查询一个用户信息 \t\n loginName 登录名，loginNameType 登录名类型，1 name,2 phone,3 email，tenantId 租户ID，instanceId 实例ID \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-login-name-and-type")
    RestResponse<UserDto> queryByLoginNameAndType(@RequestParam("loginName") String loginName, @PathVariable("loginNameType") Integer loginNameType, @RequestParam(value = "tenantId", required = false) Long tenantId, @RequestParam(value = "instanceId", required = false) Long instanceId);

    /**
     * 根据登录名、登录名类型和数据范围查询一个用户信息
     * loginNameType值为LoginNameType<br>
     * 且User的Status为1的
     *
     * @param userByLoginNameTypeQueryDto 请求dto
     * @return 用户信息Dto
     */
    @RequestMapping(value = {"/loginName/by-loginNameType"}, method = RequestMethod.GET)
    @ApiOperation(value = "根据登录名、登录名类型和数据范围查询一个用户信息", notes = "根据登录名、登录名类型和数据范围查询一个用户信息 \t\n loginName 登录名，loginNameType 登录名类型，1 name,2 phone,3 email，tenantId 租户ID，instanceId 实例ID \t\n 只返回用户信息, domain 账号域")
    @Capability(capabilityCode = "user.user.query-by-login-name-and-type")
    RestResponse<UserDto> queryByLoginNameAndType(@SpringQueryMap UserByLoginNameTypeQueryDto userByLoginNameTypeQueryDto);


    /**
     * 根据手机号查询用户
     *
     * @param phone  手机号
     * @param filter tenantId, instanceId通过filter传递
     * @return RestResponse&lt;UserDto&gt;
     */
    @RequestMapping(value = {"/phone/{phone}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据手机号查询用户", notes = "根据手机号查询用户 \t\n phone：手机号，filter：tenantId, instanceId,domain通过filter传递 \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-phone")
    RestResponse<UserDto> queryByPhone(@PathVariable("phone") String phone, @RequestParam(value = "filter", required = false) String filter);

    /**
     * 查询用户详情
     *
     * @param userId 用户ID
     * @param filter tenantId, instanceId通过filter传递
     * @return UserDto
     */
    @RequestMapping(value = {"/{userId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询用户详情", notes = "查询用户详情 \t\n userId：用户ID，filter：查询参数值无效 \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-id")
    RestResponse<UserDto> queryById(@PathVariable("userId") Long userId, @RequestParam(name = "filter", required = false) String filter);

    /**
     * 查询用户详情，请求头需设置Access-Token的值
     *
     * @return UserDto
     */
    @RequestMapping(value = {"/current-login-user"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询用户详情，请求头需设置Access-Token的值", notes = "查询用户详情 \t\n 请求头需设置Access-Token的值 \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-id")
    RestResponse<UserDto> queryById();

    /**
     * 通过id查询应用信息
     * <p>
     * 2.5.20.0起 res支持 employee
     *
     * @param userId 用户id
     * @param res    表示查询地址信息、个人信息、角色列表、发票信息、组织信息
     * @param filter tenantId, instanceId通过filter传递
     * @return UserVo
     */
    @ApiImplicitParams({@ApiImplicitParam(name = "userId", value = "用户id", paramType = "path"),
            @ApiImplicitParam(name = "res", value = "可选值: address: 地址信息 、personInfo: 个人信息、role:角色信息、billInfo:发票信息、organizationInfo: 组织信息, employee:员工信息", dataType = "String[]", paramType = "path"),
            @ApiImplicitParam(name = "filter", value = "tenantId, instanceId通过filter传递", paramType = "query")})
    @RequestMapping(value = "/{userId}/info/{res}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @ApiOperation(value = "查询用户详细信息", notes = "查询用户信息，根据传参res带出相应的资源信息，\t\n 根据需要返回用户信息及用户相关信息")
    @Capability(capabilityCode = "user.user.query-by-id-and-resource")
    RestResponse<UserVo> queryByIdAndResource(@PathVariable("userId") Long userId, @PathVariable("res") String[] res, @RequestParam(value = "filter", required = false, defaultValue = "{}") String filter);

    /**
     * 分页查询用户列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = {"/list"}, produces = "application/json", method = RequestMethod.GET)
    @ApiImplicitParams({@ApiImplicitParam(name = "example", value = "example=UserDto，查询条件json字符串," +
            "例如 {name:\"App1\"},该参数内容会合并到filter参数,不建议使用此参数", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "filter=UserDto，查询条件json字符串,例如 {name:\"App1\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),})
    @ApiOperation(value = "分页查询用户列表", notes = "分页查询用户列表，返回用户及用户个人信息")
    @Capability(capabilityCode = "user.user.query-by-example")
    RestResponse<PageInfo<UserDto>> queryByExample(@RequestParam(name = "example", required = false) String example,
                                                   @RequestParam("filter") String filter,
                                                   @RequestParam("pageNum") Integer pageNum,
                                                   @RequestParam("pageSize") Integer pageSize);


    /**
     * 分页查询用户列表
     *
     * @param userBatchQueryReqDto 查询条件
     * @param pageNum              起始页
     * @param pageSize             每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = {""}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "分页查询用户列表", notes = "分页查询用户列表，返回用户及用户个人信息")
    @Capability(capabilityCode = "user.user.query")
    RestResponse<PageInfo<UserRespDto>> query(@SpringQueryMap UserBatchQueryReqDto userBatchQueryReqDto,
                                              @RequestParam("pageNum") Integer pageNum,
                                              @RequestParam("pageSize") Integer pageSize);


    /**
     * 分页查询用户列表
     *
     * @param filter   tenantId, instanceId通过filter传递
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = {"/page"}, produces = "application/json", method = RequestMethod.GET)
    @ApiImplicitParams({@ApiImplicitParam(name = "filter", value = "filter=UserDto，查询条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @ApiOperation(value = "分页查询用户列表（通用方法）", notes = "分页查询用户列表的通用方法,返回用户及用户个人信息，filter为查询条件的json字符串，内容如下：\n" +
            "tenantId：租户ID\n" +
            "instanceId: 实例ID\n" +
            "filters:查询条件，由property、operator、value组成，分别代表属性、运算符(eq | ne | gt | lt | ge | le | like | in | isNull | isNotNull )、属性值\n" +
            "reses: 查询关联的对象，如us_user关联的us_address信息\n" +
            "orderBy：升序字段,多个字段时逗号隔开\n" +
            "orderByDesc：降序字段,多个字段时逗号隔开\n" +
            "---------------------------------------------------------------------------------------------------\n" +
            "示例：\n" +
            "{\n" +
            "\t\"tenantId\": 1,\n" +
            "\t\"instanceId\": 1,\n" +
            "\t\"filters\": [{\n" +
            "\t\t\t\"property\": \"userName\",\n" +
            "\t\t\t\"operator\": \"like\",\n" +
            "\t\t\t\"value\": \"%liu%\"\n" +
            "\t\t},\n" +
            "\t\t{\n" +
            "\t\t\t\"property\": \"phone\",\n" +
            "\t\t\t\"operator\": \"eq\",\n" +
            "\t\t\t\"value\": \"158xxxxxx\"\n" +
            "\t\t}\n" +
            "\t],\n" +
            "\t\"orderBy\": \"createTime\",\n" +
            "\t\"orderByDesc\": \"id\",\n" +
            "\t\"reses\": [\"address \", \"personInfo \", \"billInfo\"]\n" +
            "}")
    @Capability(capabilityCode = "user.user.query-by-page")
    RestResponse<PageInfo<UserDto>> queryByPage(@RequestParam("filter") String filter,
                                                @RequestParam("pageNum") Integer pageNum,
                                                @RequestParam("pageSize") Integer pageSize);


    /**
     * 分页查询租户列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = {"/tenant"}, produces = "application/json", method = RequestMethod.GET)
    @ApiImplicitParams({@ApiImplicitParam(name = "example", value = "example=TenantDto，查询条件json字符串，,该参数内容会合并到filter参数", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "filter=TenantDto，查询条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),})
    @ApiOperation(value = "分页查询租户列表", notes = "分页查询租户列表")
    @Capability(capabilityCode = "user.user.query-tenant-by-example")
    RestResponse<PageInfo<TenantDto>> queryTenantByExample(@RequestParam(name = "example", required = false) String example,
                                                           @RequestParam("filter") String filter,
                                                           @RequestParam("pageNum") Integer pageNum,
                                                           @RequestParam("pageSize") Integer pageSize);


    @RequestMapping(value = {"/{userId}/tenant"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据用户ID查询租户信息", notes = "根据用户ID查询租户信息")
    @Capability(capabilityCode = "user.user.query-tenant-by-user-id")
    RestResponse<TenantDto> queryTenantByUserId(@PathVariable("userId") Long userId);


    /**
     * 查询登录账号归属私有租户信息（供xLightning使用）
     * @return
     */
    @RequestMapping(value = {"/tenant/private"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "查询登录账号归属私有租户信息（供xLightning使用）", notes = "查询登录账号归属私有租户信息（供xLightning使用）")
    @Capability(capabilityCode = "user.user.query-private-tenant")
    RestResponse<TenantDto> queryPrivateTenant(@SpringQueryMap PrivateTenantReqDto privateTenantReqDto);


    @RequestMapping(value = {"/tenant/{id}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据租户ID查询租户信息", notes = "根据租户ID查询租户信息")
    @Capability(capabilityCode = "user.user.query-tenant-by-id")
    RestResponse<TenantDto> queryTenantById(@PathVariable("id") Long tenantId);

    @RequestMapping(value = {"/tenant/by-instance-id/{instanceId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据应用ID查询租户信息，同时返回应用信息", notes = "根据应用ID查询租户信息，同时返回应用信息")
    @Capability(capabilityCode = "user.user.query-tenant-by-instance-id")
    RestResponse<TenantDto> queryTenantByInstanceId(@PathVariable("instanceId") Long instanceId);


    /**
     * 根据角色编码分页查询用户列表
     *
     * @param roleCode 角色编码
     * @param example  查询条件json字符串
     * @param filter   查询过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = {"/by/role/{roleCode}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiImplicitParams({@ApiImplicitParam(name = "roleCode", value = "角色编码", dataType = "string", paramType = "path"),
            @ApiImplicitParam(name = "example", value = "example=UserDto，查询条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "filter=UserDto，查询条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),})
    @ApiOperation(value = "根据角色编码分页查询用户列表", notes = "根据角色编码分页查询用户列表，返回用户及用户个人信息")
    @Capability(capabilityCode = "user.user.query-by-role-code")
    RestResponse<PageInfo<UserDto>> queryByRoleCode(@PathVariable("roleCode") String roleCode,
                                                    @RequestParam(name = "example", required = false) String example,
                                                    @RequestParam("filter") String filter,
                                                    @RequestParam("pageNum") Integer pageNum,
                                                    @RequestParam("pageSize") Integer pageSize);


    /**
     * 根据用户ids查询用户列表
     *
     * @param ids    多个用户id
     * @param filter tenantId, instanceId通过filter传递
     * @return UserDtoList
     */
    @RequestMapping(value = {"/by/{ids}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据多个用户id查询用户列表", notes = "根据多个用户id查询用户列表，ids：多个id逗号连接，最多不超过1000个，filter：该查询参数值无效。 \t\n 只返回用户信息")
    @Capability(capabilityCode = "user.user.query-by-id-list")
    RestResponse<List<UserDto>> queryByIdList(@PathVariable("ids") String ids, @RequestParam("filter") String filter);


    /**
     * fixme Deprecated的要说明替代方法
     *
     * @param openId
     * @param thirdparyType 第三方类型 参考: com.dtyunxi.yundt.cube.center.user.api.constant.ThirdPlatform
     * @return
     * @see com.dtyunxi.yundt.cube.center.user.api.constant.ThirdPlatform
     */
    @Deprecated
    @RequestMapping(value = {"/openId/{openId}/type/{thirdparyType}"}, method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "openId", value = "第三方应用ID", dataType = "string", paramType = "path"),
            @ApiImplicitParam(name = "thirdparyType", value = "第三方类型：1 微信， 2 QQ， 3 新浪， 4 支付宝， 5 钉钉,参考: com.dtyunxi.yundt.cube.center.user.api.constant.ThirdPlatform\"", dataType = "integer", paramType = "path")})
    @ApiOperation(value = "查询第三方绑定的用户", notes = "根据openId和类型查询第三方绑定的用户")
    @Capability(capabilityCode = "user.user.query-binding-user")
    RestResponse<UserDto> queryBindingUser(@PathVariable("openId") String openId, @PathVariable("thirdparyType") Integer thirdparyType);

    /**
     * 查询分享注册人信息
     *
     * @param filter   过滤条件json字符串
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return 注册人信息
     */
    @RequestMapping(value = {"/relationInfo"}, method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "filter=RelationInfoDto，过滤条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @ApiOperation(value = "查询分享注册人信息", notes = "根据过滤条件查询分享注册人信息")
    @Capability(capabilityCode = "user.user.query-relation-info")
    RestResponse<PageInfo<RelationInfoDto>> queryRelationInfo(@RequestParam("filter") String filter, @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize);


    /**
     * 多租户多实例场景下，根据openid查询是否已经绑定
     *
     * @param openId        openId或者unionId
     * @param thirdparyType 第三方类型1 微信， 2 QQ， 3 新浪， 4 支付宝， 5 钉钉
     * @return 用户信息Dto
     * @see com.dtyunxi.yundt.cube.center.user.api.query.IUserQueryApi#queryBindingUserByThirdId
     */
    @GetMapping("/multi-instance-tenant/openId/{openId}/type/{thirdparyType}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "openId", value = "第三方应用的用户ID，例如openId或者unionId", dataType = "string", paramType = "path"),
            @ApiImplicitParam(name = "thirdparyType", value = "第三方类型：1 微信， 2 QQ， 3 新浪， 4 支付宝， 5 钉钉,参考: com.dtyunxi.yundt.cube.center.user.api.constant.ThirdPlatform\"", dataType = "integer", paramType = "path"),
            @ApiImplicitParam(name = "filter", value = "租户ID、实例ID", dataType = "String", paramType = "query", required = true)
    })
    @ApiOperation(value = "查询第三方绑定的用户", notes = "根据租户ID、实例ID、openId和第三方类型查询第三方绑定的用户，只返回用户信息")
    @Capability(capabilityCode = "user.user.query-binding-user-multi-instance-tenant")
    @Deprecated
    RestResponse<UserDto> queryBindingUserMultiInstanceTenant(@PathVariable("openId") String openId, @PathVariable("thirdparyType") Integer thirdparyType,
                                                              @RequestParam("filter") String filter);


    /**
     * 多租户多实例场景下，根据openid查询是否已经绑定
     *
     * @param filter 查询条件 com.dtyunxi.yundt.cube.center.user.api.dto.request.ThirdBindingQueryReq的json字符串
     * @return 用户信息Dto
     */
    @GetMapping("/third")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "过滤条件,如:租户ID、实例ID",
                    dataTypeClass = ThirdBindingQueryReq.class,
                    required = true,
                    defaultValue = "{instanceId\":1,\"openId\":\"openid\",\"queryType\":0,\"tenantId\":1,\"thirdType\":1,\"unionId\":\"unionId\",\"domain\":\"domain\"}"

            )
    })
    @ApiOperation(value = "查询第三方绑定的用户", notes = "根据租户ID、实例ID、openId和第三方类型查询第三方绑定的用户")
    @Capability(capabilityCode = "user.user.query-binding-user-by-third-id")
    RestResponse<UserDto> queryBindingUserByThirdId(@RequestParam("filter") String filter);

    @GetMapping(value = {"/third/bindings"})
    @ApiOperation(value = "根据用户id查询绑定的第三方账号信息")
    RestResponse<List<BindingsQueryRespDto>> queryBindingByUserId(@SpringQueryMap BindingsQueryReqDto bindingsQueryReqDto);
    /**
     * 分页查询用户组信息，当用户组名称groupName不为空时，用户组名称为模糊查询
     *
     * @param filter   查询条件json字符串，
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return 用户组信息
     */
    @GetMapping("/user-group/page")
    @ApiImplicitParams({@ApiImplicitParam(name = "filter", value = "查询条件json字符串", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @ApiOperation(value = "分页查询用户组信息", notes = "分页查询用户组信息，当用户组名称groupName不为空时，用户组名称为模糊查询")
    @Capability(capabilityCode = "user.user.query-user-group-by-page")
    RestResponse<PageInfo<UserGroupDto>> queryUserGroupByPage(@RequestParam("filter") String filter,
                                                              @RequestParam("pageNum") Integer pageNum,
                                                              @RequestParam("pageSize") Integer pageSize);

    /**
     * 查询用户所属用户组,用户和用户组是多对多关系，本接口不查询自子用户组
     *
     * @param userId 用户id
     * @Author : 萧玄
     * @Date: 10:30 2019/3/26
     */
    @GetMapping("{userId}/user-group")
    @ApiImplicitParams({@ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", paramType = "path")})
    @ApiOperation(value = "查询用户所属用户组", notes = "根据用户id查询用户所属用户组,用户和用户组是多对多关系，本接口不查询自子用户组")
    @Capability(capabilityCode = "user.user.find-user-group-of-user")
    RestResponse<List<UserGroupDto>> findUserGroupOfUser(@PathVariable("userId") Long userId);

    /**
     * 根据用户组类型查询组信息
     *
     * @param type     1为普通用户组, 2为通知组
     * @param tenantId tenantId
     * @return RestResponse<List < UserGroupDto>>
     */
    @GetMapping("/user-group/{tenantId}")
    @ApiOperation(value = "根据用户组类型查询组信息", notes = "根据用户组类型查询组信息, 1为普通用户组, 2为通知组")
    @Capability(capabilityCode = "user.user.query-by-group-type")
    RestResponse<List<UserGroupDto>> queryByGroupType(@RequestParam("type") Integer type, @PathVariable("tenantId") Long tenantId);

    /**
     * 根据用户组类型查询用户信息
     *
     * @param type     1为普通用户组, 2为通知组
     * @param tenantId tenantId
     * @return
     */
    @GetMapping("/{tenantId}/user/type")
    @ApiOperation(value = "根据用户组类型查询用户信息", notes = "根据用户组类型查询用户信息, 1为普通用户组, 2为通知组")
    @Capability(capabilityCode = "user.user.query-user-by-group-type")
    RestResponse<List<UserRespDto>> queryUserByGroupType(@RequestParam("type") Integer type, @PathVariable("tenantId") Long tenantId);

    /**
     * 分页查询用户及员工部分信息
     *
     * @param filter   查询条件json字符串,filter=UserEmployeeQueryReqDto
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return 用户及员工部分信息列表
     */
    @GetMapping("/employee/page")
    @ApiImplicitParams({@ApiImplicitParam(name = "filter", value = "查询条件json字符串,filter=UserEmployeeQueryReqDto", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @ApiOperation(value = "分页查询用户及员工部分信息", notes = "分页查询用户及员工部分信息 \t\n userName:用户账号 \t\n phone:手机号 \t\n employeeNo:工号 \t\n employeeName:姓名 \t\n domain:用户域")
    @Capability(capabilityCode = "user.user.query-page-by-user-and-employee-info")
    RestResponse<PageInfo<UserEmployeeQueryRespDto>> queryPageByUserAndEmployeeInfo(@RequestParam("filter") String filter,
                                                                                    @RequestParam("pageNum") Integer pageNum,
                                                                                    @RequestParam("pageSize") Integer pageSize);

    /**
     * 根据用户组id查询用户列表
     *
     * @param userGroupId 用户组id
     * @return RestResponse<List < UserRespDto>>
     */
    @GetMapping("/{userGroupId}/user")
    @ApiOperation(value = "根据用户组id查询用户列表", notes = "根据用户组id查询用户列表")
    @Capability(capabilityCode = "user.user.query-user-by-group-id")
    RestResponse<List<UserRespDto>> queryUserByGroupId(@PathVariable("userGroupId") Long userGroupId);

    /**
     * 分页查询用户列表，同时返回用户是否拥有某个角色权限
     *
     * @param roleId   角色编码
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return 用户列表
     */
    @RequestMapping(value = {"/page/role/{roleId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "分页查询用户列表，同时返回用户是否拥有某个角色权限", notes = "分页查询用户列表，同时返回用户是否拥有某个角色权限")
    @ApiImplicitParams({@ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @Capability(capabilityCode = "user.user.query-by-page-and-role-id")
    RestResponse<PageInfo<UserRoleDto>> queryByPageAndRoleId(@NotNull(message = "角色ID不允许为空") @PathVariable("roleId") Long roleId,
                                                             @SpringQueryMap UserQueryDto userQueryDto,
                                                             @RequestParam("pageNum") Integer pageNum,
                                                             @RequestParam("pageSize") Integer pageSize);

    /**
     * 分页查询未关联某个角色的用户列表
     *
     * @param roleId   角色id
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return 用户列表
     */
    @RequestMapping(value = {"/page/by-role/{roleId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "分页查询未关联某个角色的用户列表", notes = "分页查询未关联某个角色的用户列表")
    @Capability(capabilityCode = "user.user.query-by-page-and-role-id")
    RestResponse<PageInfo<UserRoleDto>> queryNotRelateRoleUser(@NotNull(message = "角色ID不允许为空") @PathVariable("roleId") Long roleId,
                                                             @SpringQueryMap UserQueryDto userQueryDto,
                                                             @RequestParam("pageNum") Integer pageNum,
                                                             @RequestParam("pageSize") Integer pageSize);

    /**
     * 根据条件分页查询租户下用户列表
     *
     * @param tenantId 租户id
     * @param account  用户账号
     * @param userName 用户名
     * @param phone    手机号
     * @param email    邮箱
     * @param pageNum  当前页
     * @param pageSize 每页记录数
     * @return 用户分页
     */
    @RequestMapping(value = {"/page/user/{tenantId}"}, produces = "application/json", method = RequestMethod.GET)
    @ApiOperation(value = "根据条件分页查询租户下用户列表", notes = "根据条件分页查询租户下用户列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query")})
    @Capability(capabilityCode = "user.user.query-user")
    RestResponse<PageInfo<UserDto>> queryUser(@PathVariable("tenantId") Long tenantId,
                                              @RequestParam(value = "account", required = false) String account,
                                              @RequestParam(value = "userName", required = false) String userName,
                                              @RequestParam(value = "phone", required = false) String phone,
                                              @RequestParam(value = "email", required = false) String email,
                                              @RequestParam("pageNum") Integer pageNum,
                                              @RequestParam("pageSize") Integer pageSize);


    @RequestMapping(value = "/overview/{userId}", method = RequestMethod.GET)
    @ApiOperation(value = "查询租户概览", notes = "查询租户概览")
    @Capability(capabilityCode = "user.user.query-tenant-overview")
    RestResponse<TenantManagerRespDto> queryTenantOverview(@PathVariable("userId") Long userId);

    /**
     * 根据地址信息查询用户id(去重)列表
     *
     * @param addressQueryReqDto 查询Dto
     * @param pageNum            当前页
     * @param pageSize           每页记录数
     * @return 地址列表分页信息
     */
    @GetMapping("/address")
    @ApiOperation(value = "根据地址信息查询用户id(去重)列表(注：因历史原因，本接口只返回用户id信息)", notes = "根据地址信息查询用户id(去重)列表(注：因历史原因，本接口只返回用户id信息)")
    @Capability(capabilityCode = "user.user.query-address")
    RestResponse<PageInfo<AddressDto>> queryAddress(@SpringQueryMap AddressQueryReqDto addressQueryReqDto,
                                                    @RequestParam("pageNum") Integer pageNum,
                                                    @RequestParam("pageSize") Integer pageSize);

    /**
     * 查询用户已授权的应用
     *
     * @param userId 用户id
     * @return 应用信息
     */
    @GetMapping("/{userId}/instance/relate")
    @ApiOperation(value = "查询用户已授权的应用", notes = "查询用户已授权的应用")
    @Capability(capabilityCode = "user.user.query-user-instance-relation")
    RestResponse<PageInfo<AppInstanceDto>> queryUserInstanceRelation(@PathVariable("userId") Long userId,
                                                                     @RequestParam("pageNum") Integer pageNum,
                                                                     @RequestParam("pageSize") Integer pageSize);

    /**
     * 查询是否需要提示用户修改密码
     *
     * @param userId 用户id
     * @return true 需要修改密码；false 不需要
     */
    @GetMapping("/{userId}/prompt")
    @ApiOperation(value = "查询是否需要提示用户修改密码", notes = "查询是否需要提示用户修改密码")
    @Capability(capabilityCode = "user.user.is-need-modify-password")
    RestResponse<Boolean> isNeedModifyPassword(@PathVariable("userId") Long userId);

    /**
     * 根据用户id查询用户默认收货地址
     *
     * @param userId 用户id
     * @return
     */
    @GetMapping("/{userId}/address/default")
    @ApiOperation(value = "根据用户id查询用户默认收货地址", notes = "根据用户id查询用户默认收货地址")
    RestResponse<AddressDto> queryDefaultAddress(@PathVariable("userId") Long userId);

    /**
     * 分页查询用户收货地址
     *
     * @param addressQueryReqDto 查询Dto
     * @param pageNum            当前页
     * @param pageSize           每页记录数
     * @return 地址列表分页信息
     */
    @GetMapping("/address/by-page")
    @ApiOperation(value = "分页查询用户收货地址", notes = "分页查询用户收货地址")
    @Capability(capabilityCode = "user.user.query-address")
    RestResponse<PageInfo<AddressDto>> queryUserAddress(@SpringQueryMap AddressQueryReqDto addressQueryReqDto,
                                                    @RequestParam("pageNum") Integer pageNum,
                                                    @RequestParam("pageSize") Integer pageSize);
}
