/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Supplier;
import io.grpc.EquivalentAddressGroup;
import io.grpc.LoadBalancer;
import io.grpc.Status;
import io.grpc.internal.BackoffPolicy;
import io.grpc.internal.ClientTransport;
import io.grpc.internal.ClientTransportFactory;
import io.grpc.internal.ConnectionClientTransport;
import io.grpc.internal.DelayedClientTransport;
import io.grpc.internal.FailingClientTransport;
import io.grpc.internal.GrpcUtil;
import io.grpc.internal.InUseStateAggregator;
import io.grpc.internal.LogExceptionRunnable;
import io.grpc.internal.ManagedClientTransport;
import io.grpc.internal.WithLogId;
import java.net.SocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
final class TransportSet
implements WithLogId {
    private static final Logger log = Logger.getLogger(TransportSet.class.getName());
    private static final ClientTransport SHUTDOWN_TRANSPORT = new FailingClientTransport(Status.UNAVAILABLE.withDescription("TransportSet is shutdown"));
    private final Object lock = new Object();
    private final EquivalentAddressGroup addressGroup;
    private final String authority;
    private final String userAgent;
    private final BackoffPolicy.Provider backoffPolicyProvider;
    private final Callback callback;
    private final ClientTransportFactory transportFactory;
    private final ScheduledExecutorService scheduledExecutor;
    private final Executor appExecutor;
    @GuardedBy(value="lock")
    private int nextAddressIndex;
    @GuardedBy(value="lock")
    private BackoffPolicy reconnectPolicy;
    @GuardedBy(value="lock")
    private final Stopwatch connectingTimer;
    @Nullable
    @GuardedBy(value="lock")
    private ScheduledFuture<?> reconnectTask;
    @GuardedBy(value="lock")
    private final Collection<ManagedClientTransport> transports = new ArrayList<ManagedClientTransport>();
    private final InUseStateAggregator<ManagedClientTransport> inUseStateAggregator = new InUseStateAggregator<ManagedClientTransport>(){

        @Override
        Object getLock() {
            return TransportSet.this.lock;
        }

        @Override
        void handleInUse() {
            TransportSet.this.callback.onInUse(TransportSet.this);
        }

        @Override
        void handleNotInUse() {
            TransportSet.this.callback.onNotInUse(TransportSet.this);
        }
    };
    @Nullable
    @GuardedBy(value="lock")
    private ConnectionClientTransport pendingTransport;
    private final LoadBalancer<ClientTransport> loadBalancer;
    @GuardedBy(value="lock")
    private boolean shutdown;
    @Nullable
    private volatile ManagedClientTransport activeTransport;

    TransportSet(EquivalentAddressGroup addressGroup, String authority, String userAgent, LoadBalancer<ClientTransport> loadBalancer, BackoffPolicy.Provider backoffPolicyProvider, ClientTransportFactory transportFactory, ScheduledExecutorService scheduledExecutor, Supplier<Stopwatch> stopwatchSupplier, Executor appExecutor, Callback callback) {
        this.addressGroup = (EquivalentAddressGroup)Preconditions.checkNotNull((Object)addressGroup, (Object)"addressGroup");
        this.authority = authority;
        this.userAgent = userAgent;
        this.loadBalancer = loadBalancer;
        this.backoffPolicyProvider = backoffPolicyProvider;
        this.transportFactory = transportFactory;
        this.scheduledExecutor = scheduledExecutor;
        this.connectingTimer = (Stopwatch)stopwatchSupplier.get();
        this.appExecutor = appExecutor;
        this.callback = callback;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    final ClientTransport obtainActiveTransport() {
        Runnable runnable;
        ManagedClientTransport savedTransport = this.activeTransport;
        if (savedTransport != null) {
            return savedTransport;
        }
        Object object = this.lock;
        synchronized (object) {
            savedTransport = this.activeTransport;
            if (savedTransport != null) {
                return savedTransport;
            }
            if (this.shutdown) {
                return SHUTDOWN_TRANSPORT;
            }
            DelayedClientTransport delayedTransport = new DelayedClientTransport(this.appExecutor);
            this.transports.add(delayedTransport);
            delayedTransport.start(new BaseTransportListener(delayedTransport));
            savedTransport = this.activeTransport = delayedTransport;
            runnable = this.startNewTransport(delayedTransport);
        }
        if (runnable != null) {
            runnable.run();
        }
        return savedTransport;
    }

    @CheckReturnValue
    @GuardedBy(value="lock")
    private Runnable startNewTransport(DelayedClientTransport delayedTransport) {
        Preconditions.checkState((this.reconnectTask == null ? 1 : 0) != 0, (Object)"Should have no reconnectTask scheduled");
        if (this.nextAddressIndex == 0) {
            this.connectingTimer.reset().start();
        }
        List<SocketAddress> addrs = this.addressGroup.getAddresses();
        SocketAddress address = addrs.get(this.nextAddressIndex++);
        if (this.nextAddressIndex >= addrs.size()) {
            this.nextAddressIndex = 0;
        }
        ConnectionClientTransport transport = this.transportFactory.newClientTransport(address, this.authority, this.userAgent);
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "[{0}] Created {1} for {2}", new Object[]{this.getLogId(), transport.getLogId(), address});
        }
        this.pendingTransport = transport;
        this.transports.add(transport);
        return transport.start(new TransportListener(transport, delayedTransport, address));
    }

    @CheckReturnValue
    @GuardedBy(value="lock")
    private Runnable scheduleBackoff(final DelayedClientTransport delayedTransport, final Status status) {
        Preconditions.checkState((this.reconnectTask == null ? 1 : 0) != 0, (Object)"previous reconnectTask is not done");
        if (this.reconnectPolicy == null) {
            this.reconnectPolicy = this.backoffPolicyProvider.get();
        }
        long delayMillis = this.reconnectPolicy.nextBackoffMillis() - this.connectingTimer.elapsed(TimeUnit.MILLISECONDS);
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "[{0}] Scheduling backoff for {1} ms", new Object[]{this.getLogId(), delayMillis});
        }
        class EndOfCurrentBackoff
        implements Runnable {
            EndOfCurrentBackoff() {
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    delayedTransport.endBackoff();
                    boolean shutdownDelayedTransport = false;
                    Runnable runnable = null;
                    boolean hasPendingStreams = delayedTransport.hasPendingStreams();
                    Object object = TransportSet.this.lock;
                    synchronized (object) {
                        TransportSet.this.reconnectTask = null;
                        if (hasPendingStreams) {
                            runnable = TransportSet.this.startNewTransport(delayedTransport);
                        } else {
                            TransportSet.this.activeTransport = null;
                            shutdownDelayedTransport = true;
                        }
                    }
                    if (runnable != null) {
                        runnable.run();
                    }
                    if (shutdownDelayedTransport) {
                        delayedTransport.setTransportSupplier(new Supplier<ClientTransport>(){

                            public ClientTransport get() {
                                return TransportSet.this.obtainActiveTransport();
                            }
                        });
                        delayedTransport.shutdown();
                    }
                }
                catch (Throwable t) {
                    log.log(Level.WARNING, "Exception handling end of backoff", t);
                }
            }
        }
        this.reconnectTask = this.scheduledExecutor.schedule(new LogExceptionRunnable(new EndOfCurrentBackoff()), delayMillis, TimeUnit.MILLISECONDS);
        return new Runnable(){

            @Override
            public void run() {
                delayedTransport.startBackoff(status);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    final void shutdown() {
        ConnectionClientTransport savedPendingTransport;
        ManagedClientTransport savedActiveTransport;
        boolean runCallback = false;
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                return;
            }
            this.shutdown = true;
            savedActiveTransport = this.activeTransport;
            savedPendingTransport = this.pendingTransport;
            this.activeTransport = null;
            if (this.transports.isEmpty()) {
                runCallback = true;
                Preconditions.checkState((this.reconnectTask == null ? 1 : 0) != 0, (Object)"Should have no reconnectTask scheduled");
            }
        }
        if (savedActiveTransport != null) {
            savedActiveTransport.shutdown();
        }
        if (savedPendingTransport != null) {
            savedPendingTransport.shutdown();
        }
        if (runCallback) {
            this.callback.onTerminated(this);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void shutdownNow(Status reason) {
        ArrayList<ManagedClientTransport> transportsCopy;
        this.shutdown();
        Iterator iterator = this.lock;
        synchronized (iterator) {
            transportsCopy = new ArrayList<ManagedClientTransport>(this.transports);
        }
        for (ManagedClientTransport transport : transportsCopy) {
            transport.shutdownNow(reason);
        }
    }

    @GuardedBy(value="lock")
    private void cancelReconnectTask() {
        if (this.reconnectTask != null) {
            this.reconnectTask.cancel(false);
            this.reconnectTask = null;
        }
    }

    @Override
    public String getLogId() {
        return GrpcUtil.getLogId(this);
    }

    static abstract class Callback {
        Callback() {
        }

        public void onTerminated(TransportSet ts) {
        }

        public void onAllAddressesFailed() {
        }

        public void onConnectionClosedByServer(Status status) {
        }

        public void onInUse(TransportSet ts) {
        }

        public void onNotInUse(TransportSet ts) {
        }
    }

    private class TransportListener
    extends BaseTransportListener {
        private final SocketAddress address;
        private final DelayedClientTransport delayedTransport;

        public TransportListener(ManagedClientTransport transport, DelayedClientTransport delayedTransport, SocketAddress address) {
            super(transport);
            this.address = address;
            this.delayedTransport = delayedTransport;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void transportReady() {
            boolean savedShutdown;
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "[{0}] {1} for {2} is ready", new Object[]{TransportSet.this.getLogId(), this.transport.getLogId(), this.address});
            }
            super.transportReady();
            Object object = TransportSet.this.lock;
            synchronized (object) {
                savedShutdown = TransportSet.this.shutdown;
                TransportSet.this.reconnectPolicy = null;
                TransportSet.this.nextAddressIndex = 0;
                if (TransportSet.this.shutdown) {
                    Preconditions.checkState((TransportSet.this.activeTransport == null ? 1 : 0) != 0, (Object)"Unexpected non-null activeTransport");
                } else if (TransportSet.this.activeTransport == this.delayedTransport) {
                    Preconditions.checkState((TransportSet.this.pendingTransport == this.transport ? 1 : 0) != 0, (Object)"transport mismatch");
                    TransportSet.this.activeTransport = this.transport;
                    TransportSet.this.pendingTransport = null;
                }
            }
            this.delayedTransport.setTransport(this.transport);
            this.delayedTransport.shutdown();
            if (savedShutdown) {
                this.transport.shutdown();
            }
            TransportSet.this.loadBalancer.handleTransportReady(TransportSet.this.addressGroup);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void transportShutdown(Status s) {
            boolean allAddressesFailed = false;
            boolean closedByServer = false;
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "[{0}] {1} for {2} is being shutdown with status {3}", new Object[]{TransportSet.this.getLogId(), this.transport.getLogId(), this.address, s});
            }
            super.transportShutdown(s);
            Runnable runnable = null;
            Object object = TransportSet.this.lock;
            synchronized (object) {
                if (TransportSet.this.activeTransport == this.transport) {
                    TransportSet.this.activeTransport = null;
                    closedByServer = !TransportSet.this.shutdown;
                } else if (TransportSet.this.activeTransport == this.delayedTransport) {
                    if (TransportSet.this.nextAddressIndex == 0) {
                        allAddressesFailed = true;
                        runnable = TransportSet.this.scheduleBackoff(this.delayedTransport, s);
                    } else {
                        runnable = TransportSet.this.startNewTransport(this.delayedTransport);
                    }
                }
            }
            if (runnable != null) {
                runnable.run();
            }
            TransportSet.this.loadBalancer.handleTransportShutdown(TransportSet.this.addressGroup, s);
            if (allAddressesFailed) {
                TransportSet.this.callback.onAllAddressesFailed();
            }
            if (closedByServer) {
                TransportSet.this.callback.onConnectionClosedByServer(s);
            }
        }

        @Override
        public void transportTerminated() {
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "[{0}] {1} for {2} is terminated", new Object[]{TransportSet.this.getLogId(), this.transport.getLogId(), this.address});
            }
            super.transportTerminated();
            Preconditions.checkState((TransportSet.this.activeTransport != this.transport ? 1 : 0) != 0, (Object)"activeTransport still points to the delayedTransport. Seems transportShutdown() was not called.");
        }
    }

    private class BaseTransportListener
    implements ManagedClientTransport.Listener {
        protected final ManagedClientTransport transport;

        public BaseTransportListener(ManagedClientTransport transport) {
            this.transport = transport;
        }

        @Override
        public void transportReady() {
        }

        @Override
        public void transportInUse(boolean inUse) {
            TransportSet.this.inUseStateAggregator.updateObjectInUse(this.transport, inUse);
        }

        @Override
        public void transportShutdown(Status status) {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void transportTerminated() {
            boolean runCallback = false;
            TransportSet.this.inUseStateAggregator.updateObjectInUse(this.transport, false);
            Object object = TransportSet.this.lock;
            synchronized (object) {
                TransportSet.this.transports.remove(this.transport);
                if (TransportSet.this.shutdown && TransportSet.this.transports.isEmpty()) {
                    if (log.isLoggable(Level.FINE)) {
                        log.log(Level.FINE, "[{0}] Terminated", TransportSet.this.getLogId());
                    }
                    runCallback = true;
                    TransportSet.this.cancelReconnectTask();
                }
            }
            if (runCallback) {
                TransportSet.this.callback.onTerminated(TransportSet.this);
            }
        }
    }
}

