package com.dtyunxi.cube.biz.commons.utils;

import com.alibaba.fastjson.JSON;
import com.dtyunxi.cube.biz.commons.annotation.ExcelColumnProperty;
import com.dtyunxi.util.DateUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.PropertyDescriptor;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * 报表工具类
 * @author li.jundong
 * @date: 2019/7/4 10:11
 */
public class ExcelUtil {

    private static final Logger logger = LoggerFactory.getLogger(ExcelUtil.class);

    private static final Integer BATCH_NUM = 100;
    /**
     * 数据转为excel报表，并输出InputStream
     * @author: li.jundong
     * @date: 2019/7/4 14:27
     * @param objects
     * @return:byte[]
     */
    public static InputStream createExcel(List<? extends Object> objects){
        return createExcel(null, objects);
    }

    /**
     * 创建一个多sheet的Excel
     * @param sheetNames    sheet名称数组
     * @param dataLists 每个sheet对应的数据List集合
     * @return
     */
    public static InputStream createExcel(String[] sheetNames, Object... dataLists){

        if (0 == sheetNames.length || sheetNames.length != dataLists.length){
            //参数异常，sheet名称要跟数据数组一一对应
            return null;
        }

        //创建一个HSSFWorkbook，对应一个Excel文件
        HSSFWorkbook wb = new HSSFWorkbook();

        for (int i = 0; i < sheetNames.length; i++){
            //创建sheet
            HSSFSheet sheet = sheetNames[i] == null ? wb.createSheet() : wb.createSheet(sheetNames[i]);

            //循环sheet数据集合数组
            if (null != dataLists[i]){

                List<Object> dataList = (List<Object>) dataLists[i];
                if (!dataList.isEmpty()){

                    List<String> titles = getTitles(dataList.get(0));
                    logger.info("导出excel标题, titles={}", JSON.toJSONString(titles));

                    // 第三步，在sheet中添加表头第0行,注意老版本poi对Excel的行数列数有限制
                    HSSFRow row = sheet.createRow(0);

                    //创建标题
                    for(int j = 0; j < titles.size(); j++){
                        HSSFCell cell = row.createCell(j);
                        cell.setCellValue(titles.get(j));
                    }

                    //获取表格内容
                    List<List<Object>> datas = getDatas(dataList);  //一个sheet中每一行的数据集合
                    logger.info("导出excel总行数, datas={}", datas.size());

                    //插入整个sheet的表格数据
                    for (int j = 0; j < datas.size(); j++){

                        //创建行数
                        row = sheet.createRow(j+1);
                        for(int k = 0; k < datas.get(j).size(); k++){
                            //插入一行中的每一列数据
                            HSSFCell cell = row.createCell(k);

                            Object o = datas.get(j).get(k);
                            if(o != null){
                                //遇到时间对象，要做格式转换
                                if (o instanceof Date) {
                                    cell.setCellValue(DateUtil.format((Date) o, "yyyy-MM-dd HH:mm:ss"));
                                } else {
                                    cell.setCellValue(o.toString());
                                }
                            }
                        }
                    }
                }
            }
        }

        ByteArrayOutputStream out = null;
        try {
            out = new ByteArrayOutputStream();
            wb.write(out);
            return new ByteArrayInputStream(out.toByteArray());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 数据转为excel报表，并输出InputStream
     * @author: li.jundong
     * @date: 2019/7/4 14:27
     * @param: * @param sheetName 工作表名称
     * @param objects
     * @return:byte[]
     */
    public static InputStream createExcel(String sheetName, List<? extends Object> objects){
        if(CollectionUtils.isEmpty(objects)){
            return null;
        }
        Object object = objects.get(0);
        List<String> titles = getTitles(object);
        logger.info("导出excel标题, titles={}", JSON.toJSONString(titles));

        // 第一步，创建一个HSSFWorkbook，对应一个Excel文件
        HSSFWorkbook wb = new HSSFWorkbook();

        // 第二步，在workbook中添加一个sheet,对应Excel文件中的sheet
        final HSSFSheet sheet = sheetName == null ? wb.createSheet() : wb.createSheet(sheetName);

        // 第三步，在sheet中添加表头第0行,注意老版本poi对Excel的行数列数有限制
        HSSFRow row = sheet.createRow(0);

        // 第四步，创建单元格，并设置值表头 设置表头居中
//        HSSFCellStyle style = wb.createCellStyle();
//        style.setAlignment(HSSFCellStyle.ALIGN_CENTER); // 创建一个居中格式

        //创建标题
        for(int i = 0; i < titles.size(); i++){
            HSSFCell cell = row.createCell(i);
            cell.setCellValue(titles.get(i));
        }

        //获取表格内容
        List<List<Object>> datas = getDatas(objects);
        logger.info("导出excel总行数, datas={}", datas.size());

        //总数据量
        int size = datas.size();
        //总分页数，不整除+1, 比如10000条数据就是10页，10001就是11页
        int totalPage = size % BATCH_NUM == 0 ? size / BATCH_NUM : ((size / BATCH_NUM) + 1);

        //线程执行任务的计数器，初始大小为totalPage
        CountDownLatch latch = new CountDownLatch(totalPage);

        for(int i = 0; i < datas.size(); i++){
            int index = i + 1;

            //假如totalPage=10，index=10000，则totalPage - (index % BATCH_NUM) == 1为假, index % BATCH_NUM == 0为真
            if(index % BATCH_NUM == 0){
                //假如index=10000，start = 9000
                int start = index - BATCH_NUM;

                //多线程生成row
//                ExecutorUtils.execute(() -> {
                    int currentIndex = index;
                    batchCreateRow(sheet, latch, start, currentIndex, datas);
                    //每个任务执行完就countDown一次
                    latch.countDown();
//                });
            }else {
                //如果是最后一条数据
                if(index == size){
                    //假如index=10001，start = 10000
                    int start = (index / BATCH_NUM) * BATCH_NUM;

                    //多线程生成row
//                    ExecutorUtils.execute(() -> {
                        int currentIndex = index;
                        batchCreateRow(sheet, latch, start, currentIndex, datas);
                        //每个任务执行完就countDown一次
                        latch.countDown();
//                    });
                }
            }
        }

        ByteArrayOutputStream out = null;
        try {
            //阻塞，直到计数器的值为0，才让主线程往下执行
            latch.await();

            out = new ByteArrayOutputStream();
            wb.write(out);
            return new ByteArrayInputStream(out.toByteArray());
        } catch (Exception e) {
            logger.error("excel转换成输入流error, message={}", e);
        }
        return null;
    }

    /**
     * 获取excel标题
     * @author: li.jundong
     * @date: 2019/7/4 10:26
     * @param: * @param object
     * @return:java.util.List<java.lang.String>
     */
    private static List<String> getTitles(Object object){
        Class clazz = object.getClass();
        Field[] fields = clazz.getDeclaredFields();

        //key=index, value=columnName
        SortedMap<Integer, String> columnMap = new TreeMap<Integer, String>();

        for(Field field: fields){
            field.setAccessible(true);
            try {
                //获取字段名
                String property = field.getName();
                PropertyDescriptor pd = new PropertyDescriptor(field.getName(), clazz);

                //获取字段注解上的操作类型
                ExcelColumnProperty excelColumnProperty = field.getAnnotation(ExcelColumnProperty.class);
                if(excelColumnProperty != null){
                    columnMap.put(Integer.valueOf(excelColumnProperty.index()), excelColumnProperty.columnName());
                }
            } catch (Exception e){
                e.printStackTrace();
            }
        }
        return new ArrayList<>(columnMap.values());
    }

    /**
     * 转换为报表填充数据
     * @author: li.jundong
     * @date: 2019/7/4 14:26
     * @param: * @param objects
     * @return:java.util.List<java.util.List<java.lang.Object>>
     */
    private static List<List<Object>> getDatas(List<? extends Object> objects){
        List<List<Object>> datas = new ArrayList<>();

        List<Future<ArrayList<Object>>> futureList = new ArrayList<>(objects.size());

        for (Object object: objects){
            //多线程把对象反射成object
            Future<ArrayList<Object>> future = ExecutorUtils.submit(new Callable<ArrayList<Object>>() {
                @Override
                public ArrayList<Object> call() {
                    Class clazz = object.getClass();
                    Field[] fields = clazz.getDeclaredFields();

                    SortedMap<Integer, Object> columnValueMap = new TreeMap<Integer, Object>();
                    for (Field field : fields) {
                        field.setAccessible(true);
                        try {
                            //获取字段名
                            PropertyDescriptor pd = new PropertyDescriptor(field.getName(), clazz);
                            //获取get方法
                            Method method = pd.getReadMethod();
                            //获取到方法的值
                            Object value = method.invoke(object);

                            //获取字段注解上的操作类型
                            ExcelColumnProperty excelColumnProperty = field.getAnnotation(ExcelColumnProperty.class);
                            if (excelColumnProperty != null) {
                                columnValueMap.put(Integer.valueOf(excelColumnProperty.index()), value);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                    ArrayList<Object> list = new ArrayList<>(columnValueMap.values());
                    return list;
                }
            });
            futureList.add(future);
        }

        futureList.forEach(future ->{
            try {
                datas.add(future.get());
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ExecutionException e) {
                e.printStackTrace();
            }
        });
        return datas;
    }

    /**
     * sheet.createRow是线程不安全的，这里加上锁
     * @author li.jundong
     * @date 2019/7/27
     * @param sheet
     * @param rownum
     * @return org.apache.poi.hssf.usermodel.HSSFRow
     */
    public static synchronized HSSFRow createRow(HSSFSheet sheet, int rownum) {
        return sheet.createRow(rownum);
    }

    /**
     * 批量生成行
     * @author li.jundong
     * @date 2019/7/29
     * @param sheet
     * @param latch
     * @param start
     * @param datas
     * @return void
     */
    public static void batchCreateRow(HSSFSheet sheet, CountDownLatch latch, int start, int end, List<List<Object>> datas){
        for(int b = start; b < end; b++){
            //获取每一行的数据
            List<Object> values = datas.get(b);

            //sheet.createRow是线程不安全的，这里加上锁
            int index = b + 1;
            HSSFRow hssfRow = createRow(sheet, index);

            for (int j = 0; j < values.size(); j++) {
                //将内容按顺序赋给对应的列对象
                Object o = values.get(j);
                try {
                    //这里一定要指定type，type=value的数据类型，不然多线程情况下，会对整个HSSFCell进行调整，不是线程安全的
                    HSSFCell cell = hssfRow.createCell(j, HSSFCell.CELL_TYPE_STRING);
                    if(o != null){
                        if (o instanceof Date) {
                            cell.setCellValue(DateUtil.format((Date) o, "yyyy-MM-dd HH:mm:ss"));
                        } else {
                            cell.setCellValue(o.toString());
                        }
                    } else {
                        cell.setCellValue("");
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    logger.error("生成单元格error, 第{}行, 第{}列, value={}, message={}", index, j, JSON.toJSONString(o), e.getMessage(), e);
                }
            }
        }
    }
}
