/**
 * @(#)IApplicationApi.java 1.0 May 17, 2018
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.AppPurchaseDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.AppRegisterDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.ButtonDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.MenuDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.AppInstanceCreateReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.AppInstanceMenuIncrDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.ResourceUpdateReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.SortReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.vo.AppRegisterVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 用户应用服务API接口
 *
 * @author 南枫
 * @since 1.0.0
 */
@Api(tags = {"用户中心：应用模板和应用服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-IApplicationApi", path = "/v1/application", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IApplicationApi {

    /**
     * 注册应用模板
     *
     * @param appRegisterDto 注册应用模板
     * @return AppRegisterVo
     */
    @RequestMapping(value = "", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.POST)
    @ApiOperation(value = "注册应用模板", notes = "注册应用模板")
    @Capability(capabilityCode = "user.application.register")
    RestResponse<AppRegisterVo> register(@RequestBody AppRegisterDto appRegisterDto);

    /**
     * 通过appId修改应用模板基本信息
     *
     * @param appId          应用id
     * @param appRegisterDto 应用注册
     * @return void
     */
    @RequestMapping(value = "/{appId}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.PUT)
    @ApiOperation(value = "修改应用模板", notes = "通过appId修改应用模板基本信息")
    @Capability(capabilityCode = "user.application.modify-app")
    RestResponse<Void> modifyApp(@PathVariable("appId") Long appId, @RequestBody AppRegisterDto appRegisterDto);

    /**
     * 通过应用id和status，修改应用模板状态，启用或者禁用
     *
     * @param appId  应用模板id
     * @param status 状态：1，启用，2禁用
     * @return Long
     */
    @RequestMapping(value = "/{appId}/status/{status}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.PUT)
    @ApiOperation(value = "修改应用状态", notes = "通过appId修改应用状态status")
    @Capability(capabilityCode = "user.application.modify")
    RestResponse<Long> modify(@PathVariable("appId") Long appId, @PathVariable("status") Integer status);

    /**
     * 删除应用模板
     *
     * @param appIds 应用模板Id
     * @return void
     */
    @RequestMapping(value = "", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.DELETE)
    @ApiOperation(value = "删除应用模板", notes = "根据应用模板id删除应用模板")
    @Capability(capabilityCode = "user.application.remove-app")
    RestResponse<Void> removeApp(@RequestParam("appIds") List<Long> appIds);


    /**
     * 应用模板重绑菜单
     *
     * @param appCode   应用模板编码
     * @param menuCodes 菜单列表
     * @return Boolean
     */
    @RequestMapping(value = "/app-template/menus", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.POST)
    @ApiOperation(value = "应用模板重绑菜单", notes = "应用模板重绑菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "appCode", value = "应用模板编码", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCodes", value = "资源菜单编码列表", paramType = "query", allowMultiple = true, required = true)
    })
    @Capability(capabilityCode = "user.application.app-rebind-menus")
    RestResponse<Long> appRebindMenus(@RequestParam("appCode") String appCode, @RequestParam("menuCodes") List<String> menuCodes);


    /**
     * 应用模板绑定或解绑单个菜单
     *
     * @param appCode  应用模板编码
     * @param menuCode 菜单列表
     * @return Boolean
     */
    @PostMapping(value = "/app-template/menu", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "应用模板绑定或解绑单个菜单", notes = "应用模板绑定或解绑单个菜单")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "type", value = "操作类型:1绑定,2解绑", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用模板编码", dataType = "string", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCode", value = "资源菜单编码", dataType = "string", paramType = "query", required = true)
    })
    @Capability(capabilityCode = "user.application.app-rebind-menu")
    RestResponse<Boolean> appRebindMenu(@RequestParam("type") Integer type, @RequestParam("appCode") String appCode, @RequestParam("menuCode") String menuCode);

    /**
     * 应用模板重绑按钮
     *
     * @param appCode     应用模板编码
     * @param buttonCodes 按钮列表
     * @return Boolean
     */
    @PostMapping(value = "/app-template/buttons", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "应用模板重绑按钮", notes = "应用模板重绑按钮")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "appCode", value = "应用模板编码", dataType = "string", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCode", value = "菜单编码：不为空时代表在当前菜单下重绑按钮;为空时代表在当前应用模板下重绑按钮", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "buttonCodes", value = "资源按钮编码列表", paramType = "query", allowMultiple = true, required = true)
    })
    @Capability(capabilityCode = "user.application.app-rebind-buttons")
    RestResponse<Long> appRebindButtons(@RequestParam("appCode") String appCode, @RequestParam(value = "menuCode", required = false) String menuCode, @RequestParam("buttonCodes") List<String> buttonCodes);

    /**
     * 应用模板绑定或解绑单个按钮
     *
     * @param appCode    应用模板编码
     * @param buttonCode 按钮
     * @return Boolean
     */
    @PostMapping(value = "/app-template/button", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "应用模板绑定或解绑单个按钮", notes = "应用模板绑定或解绑单个按钮")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "type", value = "操作类型:1绑定,2解绑", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "appCode", value = "应用模板编码", dataType = "string", paramType = "query", required = true),
            @ApiImplicitParam(name = "buttonCode", value = "资源按钮编码", dataType = "string", paramType = "query", required = true)
    })
    @Capability(capabilityCode = "user.application.app-rebind-button")
    RestResponse<Boolean> appRebindButton(@RequestParam("type") Integer type, @RequestParam("appCode") String appCode, @RequestParam("buttonCode") String buttonCode);

    /**
     * 添加应用
     *
     * @param reqDto 应用请求dto
     * @return 返回应用id
     */
    @PostMapping(value = "/appInstance", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "添加应用", notes = "添加应用")
    @Capability(capabilityCode = "user.application.add-app-instance")
    RestResponse<Long> addAppInstance(@RequestBody @Valid AppInstanceCreateReqDto reqDto);

    /**
     * 应用重新初始化
     */
    @PostMapping(value = "/appInstance/init", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "应用重新初始化菜单按钮", notes = "应用重新初始化菜单按钮")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCodes", value = "资源菜单编码列表", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "buttonCodes", value = "资源按钮编码列表", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "updateOld", value = "是否对旧的菜单或按钮列表有且新的菜单或按钮列表也有的元素进行更新(默认是true)", paramType = "query", dataType = "boolean"),
            @ApiImplicitParam(name = "deleteOld", value = "是否对旧的菜单或按钮列表有且新的菜单或按钮列表没有的元素进行删除(默认是true)", paramType = "query", dataType = "boolean")
    })
    @Capability(capabilityCode = "user.application.init-app-instance")
    RestResponse<Void> initAppInstance(@RequestParam("instanceId") Long instanceId,
                                       @RequestParam(value = "menuCodes", required = false, defaultValue = "") List<String> menuCodes,
                                       @RequestParam(value = "buttonCodes", required = false, defaultValue = "") List<String> buttonCodes,
                                       @RequestParam(value = "updateOld", required = false, defaultValue = "true") Boolean updateOld,
                                       @RequestParam(value = "deleteOld", required = false, defaultValue = "true") Boolean deleteOld);

    /**
     * 应用增量初始化资源为菜单按钮
     *
     * @param appInstanceMenuIncrDto
     * @return
     */
    @PostMapping(value = "/appInstance/incr/resource")
    @ApiOperation(value = "应用增量初始化资源为菜单按钮", notes = "应用增量初始化资源为菜单按钮")
    @Capability(capabilityCode = "user.application.incremental-app-instance-resources")
    RestResponse<Void> incrementalAppInstanceResources(@RequestBody AppInstanceMenuIncrDto appInstanceMenuIncrDto);

    /**
     * 通过appId修改应用信息
     *
     * @param instanceId     应用编码
     * @param appPurchaseDto 应用
     * @return Long
     */
    @RequestMapping(value = "/appInstance/{instanceId}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.PUT)
    @ApiOperation(value = "修改应用信息", notes = "通过appId修改应用信息")
    @Capability(capabilityCode = "user.application.modify-app-ins-info")
    RestResponse<Long> modifyAppInsInfo(@PathVariable("instanceId") Long instanceId, @RequestBody AppPurchaseDto appPurchaseDto);

    /**
     * 通过应用id和status，修改应用状态，启用或者禁用
     *
     * @param instanceId 应用编码
     * @param status     状态：1，启用，2禁用
     * @return Long
     */
    @RequestMapping(value = "/appInstance/{instanceId}/status/{status}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.PUT)
    @ApiOperation(value = "修改应用状态", notes = "通过appInstanceId修改应用状态status")
    @Capability(capabilityCode = "user.application.modify-app-ins")
    RestResponse<Long> modifyAppIns(@PathVariable("instanceId") Long instanceId, @PathVariable("status") Integer status);

    /**
     * 删除应用
     *
     * @param instanceId 应用Id
     * @return void
     */
    @DeleteMapping(value = "/appInstance/{instanceId}", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "删除应用", notes = "根据应用id删除应用，以及绑定的资源")
    @Capability(capabilityCode = "user.application.remove-app-ins")
    RestResponse<Void> removeAppIns(@PathVariable("instanceId") Long instanceId);

    /**
     * 删除应用菜单按钮
     */
    @DeleteMapping(value = "/appInstance/menu-button", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "删除应用菜单按钮", notes = "删除应用菜单按钮")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCodes", value = "资源菜单编码列表", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "buttonCodes", value = "资源按钮编码列表", paramType = "query", allowMultiple = true)
    })
    @Capability(capabilityCode = "user.application.remove-instance-menu-button")
    RestResponse<Void> removeInstanceMenuButton(@RequestParam("instanceId") Long instanceId,
                                                @RequestParam(value = "menuCodes", required = false, defaultValue = "") List<String> menuCodes,
                                                @RequestParam(value = "buttonCodes", required = false, defaultValue = "") List<String> buttonCodes);

    /**
     * 更新菜单和按钮的父级
     */
    @PutMapping(value = "/appInstance/parent", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "更新菜单和按钮的父级", notes = "更新菜单和按钮的父级")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query", required = true),
            @ApiImplicitParam(name = "parentCode", value = "父级编码", paramType = "query", required = true),
            @ApiImplicitParam(name = "menuCodes", value = "资源菜单编码列表", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "buttonCodes", value = "资源按钮编码列表", paramType = "query", allowMultiple = true)
    })
    @Capability(capabilityCode = "user.application.update-instance-menu-button-parent")
    RestResponse<Void> updateInstanceMenuButtonParent(@RequestParam("instanceId") Long instanceId, @RequestParam("parentCode") String parentCode,
                                                      @RequestParam(value = "menuCodes", required = false, defaultValue = "") List<String> menuCodes,
                                                      @RequestParam(value = "buttonCodes", required = false, defaultValue = "") List<String> buttonCodes);

    /**
     * 更新菜单和按钮的父级
     */
    @PutMapping(value = "/app-instance/buttons", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "更新应用按钮", notes = "更新应用按钮")
    @Capability(capabilityCode = "user.application.update-instance-buttons")
    RestResponse<Void> updateInstanceButtons(@RequestBody List<ButtonDto> dtos);

    /**
     * 应用模板绑定按钮信息
     * 再 {@link IApplicationApi#appRebindButtons(java.lang.String, java.lang.String, java.util.List)}
     * 或绑定到该应用模板
     *
     * @param appId      应用模板id
     * @param buttonDtos 按钮信息
     * @return Boolean
     * @deprecated 2.6.21.0, 应用模板菜单按钮, 可通过 {@link com.dtyunxi.yundt.cube.center.user.api.IApplicationApi#appRebindButtons(String, String, List)} 进行绑定
     */
    @Deprecated
    @RequestMapping(value = "/{appId}/{menuId}/buttons", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.POST)
    @ApiOperation(value = "应用模板绑定按钮信息", notes = "应用模板绑定按钮信息，新增按钮")
    @Capability(capabilityCode = "user.application.app-bind-button")
    RestResponse<Boolean> appBindButton(@PathVariable("appId") Long appId,
                                        @PathVariable("menuId") Long menuId,
                                        @RequestBody List<ButtonDto> buttonDtos);

    /**
     * 应用模板解除绑定按钮信息
     * <p>
     * 再 {@link IApplicationApi#appRebindButtons(java.lang.String, java.lang.String, java.util.List)}
     * 或绑定到该应用模板
     *
     * @param appId     应用id
     * @param menuId    菜单Id
     * @param buttonIds 按钮id集合
     * @return Boolean
     * @deprecated 2.6.21.0, 应用模板菜单按钮, 可通过 {@link com.dtyunxi.yundt.cube.center.user.api.IApplicationApi#appRebindButtons(String, String, List)} 进行绑定
     */
    @Deprecated
    @RequestMapping(value = "/{appId}/{menuId}/buttons", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.DELETE)
    @ApiOperation(value = "应用解除绑定按钮信息", notes = "应用解除绑定按钮信息，同时会删除按钮，如果按钮被应用套餐引用则无法删除")
    @Capability(capabilityCode = "user.application.app-un-bind-button")
    RestResponse<Boolean> appUnBindButton(@PathVariable("appId") Long appId,
                                          @PathVariable("menuId") Long menuId,
                                          @RequestParam("buttonIds") List<Long> buttonIds);

    /**
     * 应用模板绑定菜单信息
     * 再 {@link IApplicationApi#appRebindMenus(String, List)}
     * 或绑定到该应用模板
     *
     * @param appId   应用模板id
     * @param menuDto 菜单信息，多个树结构
     * @return Boolean
     * @deprecated 2.6.21.0, 应用模板菜单按钮通过{@link com.dtyunxi.yundt.cube.center.user.api.IApplicationApi#appRebindMenus(String, List)}进行绑定
     */
    @Deprecated
    @RequestMapping(value = "/{appId}/menus", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.POST)
    @ApiOperation(value = "应用模板绑定菜单信息", notes = "应用模板绑定菜单信息")
    @Capability(capabilityCode = "user.application.app-bind-menu")
    RestResponse<Long> appBindMenu(@PathVariable("appId") Long appId, @RequestBody MenuDto menuDto);

    /**
     * 修改应用模板菜单信息
     *
     * @param appId   应用模板id
     * @param id      按钮id
     * @param menuDto 菜单信息，多个树结构
     * @return Boolean
     * @deprecated 2.6.21.0, 应用模板菜单按钮修改应使用 {@link com.dtyunxi.yundt.cube.center.user.api.IResourceApi#modifyResource(ResourceUpdateReqDto)}
     */
    @Deprecated
    @RequestMapping(value = "/{appId}/menu/{id}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.PUT)
    @ApiOperation(value = "修改应用菜单信息", notes = "修改应用菜单信息，应用如果为禁用状态无法修改，菜单如果被应用套餐引用无法禁用")
    @Capability(capabilityCode = "user.application.modify-app-menu")
    RestResponse<Boolean> modifyAppMenu(@PathVariable("appId") Long appId, @NotNull @PathVariable("id") Long id, @RequestBody MenuDto menuDto);

    /**
     * 应用模板解除绑定菜单信息
     * <p>
     * 再 {@link IApplicationApi#appRebindMenus(String, List)}
     * 或绑定到该应用模板
     *
     * @param appId  应用模板id
     * @param menuId 菜单code
     * @return Boolean
     * @deprecated 2.6.21.0, 解绑应用模板菜单应通过 {@link com.dtyunxi.yundt.cube.center.user.api.IApplicationApi#appRebindMenus(String, List)}
     */
    @Deprecated
    @RequestMapping(value = "/{appId}/menus", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.DELETE)
    @ApiOperation(value = "应用模板解除绑定菜单信息", notes = "应用模板解除绑定菜单信息，会删除菜单信息，应用模板如果为禁用状态无法解绑，菜单如果被应用模板套餐引用也无法解绑")
    @Capability(capabilityCode = "user.application.app-un-bind-menu")
    RestResponse<Boolean> appUnBindMenu(@PathVariable("appId") Long appId, @RequestParam("menuId") Long menuId);

    /**
     * 应用实例初始化菜单信息
     *
     * @param id 应用实例id
     * @return Boolean
     * @deprecated 2.6.21.0, 应用初始化 {@link com.dtyunxi.yundt.cube.center.user.api.IApplicationApi#initAppInstance}
     */
    @Deprecated
    @RequestMapping(value = "/instance/{id}/menus", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.POST)
    @ApiOperation(value = "应用实例初始化菜单信息", notes = "应用实例初始化菜单信息：从应用的菜单复制一份")
    @Capability(capabilityCode = "user.application.app-instance-initialization-menus")
    RestResponse<Boolean> appInstanceInitializationMenus(@PathVariable("id") Long id);


    /**
     * 启用菜单
     *
     * @param menuId 套餐主键ID
     * @param appId  应用ID
     * @deprecated 2.6.21.0, 应用模板菜单按钮可以通过修改资源菜单按钮来修改 {@link com.dtyunxi.yundt.cube.center.user.api.IResourceApi#modifyResource}
     */
    @Deprecated
    @PutMapping("/menu/enable/{menuId}")
    @ApiOperation(value = "启用菜单", notes = "启用菜单")
    @Capability(capabilityCode = "user.application.enable-menu")
    RestResponse<Void> enableMenu(@PathVariable("menuId") Long menuId, @RequestParam("appId") Long appId);

    /**
     * 禁用菜单
     *
     * @param menuId 套餐主键
     * @param appId  应用ID
     * @deprecated 2.6.21.0, 应用模板菜单按钮可以通过修改资源菜单按钮来修改 {@link com.dtyunxi.yundt.cube.center.user.api.IResourceApi#modifyResource}
     */
    @Deprecated
    @PutMapping("/menu/disable/{menuId}")
    @ApiOperation(value = "禁用菜单", notes = "禁用菜单，菜单如果被应用套餐引用则无法禁用")
    @Capability(capabilityCode = "user.application.disable-menu")
    RestResponse<Void> disableMenu(@PathVariable("menuId") Long menuId,
                                   @RequestParam("appId") Long appId);

    /**
     * 菜单排序
     *
     * @param id         菜单ID
     * @param sortReqDto 排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)
     * @return 操作结果
     */
    @PutMapping("/menu/sort/{id}")
    @ApiOperation(value = "菜单排序", notes = "sortType=排序类型(BOTTOM：置底，INCR：升序，DECR：降序，TOP：置顶)")
    @Capability(capabilityCode = "user.application.sort-menu")
    RestResponse<Void> sortMenu(@NotNull(message = "菜单ID不允许为空") @PathVariable("id") Long id,
                                @RequestBody @Validated SortReqDto sortReqDto);


    /**
     * 创建应用实例：不做任何检测，简单创建记录
     *
     * @param reqDto 应用请求dto
     * @return 返回应用id
     */
    @PostMapping(value = "/instance", produces = {"application/json;charset=UTF-8"})
    @ApiOperation(value = "创建应用实例", notes = "创建应用实例：不做任何检测，简单创建记录")
    @Capability(capabilityCode = "user.application.create-app-instance")
    RestResponse<Long> createAppInstance(@RequestBody @Valid AppInstanceCreateReqDto reqDto);

}
