/**
 * @(#) IAccessQueryApi.java 1.0 2018-06-20
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api.query;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.RoleAccessDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.RoleDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.ApplicationAndRelatedRolesRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.RoleDetailAndAccessRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.UserRoleReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.*;
import com.dtyunxi.yundt.cube.center.user.api.dto.vo.UserAccessVo;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 用户权限查询服务API接口
 *
 * @author 黑马
 * @since 1.0.0
 */
@Api(tags = {"用户中心：权限查询服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-query-IAccessQueryApi", path = "/v1/access", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IAccessQueryApi {
    /**
     * 根据角色ID查询角色
     *
     * @param roleId 角色ID
     * @param filter filter
     * @return RoleAccessDto
     */
    @GetMapping(value = {"/role/{roleId}"}, produces = "application/json")
    @ApiOperation(value = "根据角色ID查询角色权限", notes = "根据角色ID查询角色权限，filter=暂未使用")
    @Capability(capabilityCode = "user.access.query-role-by-id")
    RestResponse<RoleAccessDto> queryRoleById(@PathVariable("roleId") Long roleId, @RequestParam("filter") String filter);

    /**
     * 根据角色ID查询角色详情
     *
     * @param roleId 角色ID
     * @return RoleDto 角色详情
     */
    @GetMapping(value = {"/role/id/{roleId}"}, produces = "application/json")
    @ApiOperation(value = "根据角色ID查询角色详情")
    @Capability(capabilityCode = "user.access.query-role-by-role-id")
    RestResponse<RoleDto> queryRoleByRoleId(@PathVariable("roleId") Long roleId);

    /**
     * 根据实例ID+角色code，或者实例ID+角色名称查询角色详情
     *
     * @param instanceId 实例ID
     * @param code       角色code
     * @param name       角色名称
     * @return RoleDto   角色详情
     */
    @GetMapping(value = {"/{instanceId}/role"}, produces = "application/json")
    @ApiOperation(value = "根据实例ID+角色code，或者实例ID+角色名称查询角色详情")
    @Capability(capabilityCode = "user.access.query-role-by-code-or-name")
    RestResponse<RoleDto> queryRoleByCodeOrName(@PathVariable("instanceId") Long instanceId,
                                                @RequestParam(value = "code", required = false) String code,
                                                @RequestParam(value = "name", required = false) String name);

    /**
     * 分页查询角色列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @GetMapping(value = {"/roles"}, produces = "application/json")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "example", value = "查询条件json字符串,例如 {\"code\": \"tenant\"}", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "查询条件", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),
    })
    @ApiOperation(value = "分页查询角色列表", notes = "分页查询角色列表，example和filter会合并，合并后只支持code和name模糊查询，自动带上%%，不需要手动传入\t\n" +
            "如需查询角色关联的用户及用户组个数，filter应设置key=countRelationUserAndGroupNum并且value=true")
    @Capability(capabilityCode = "user.access.query-page-role")
    RestResponse<PageInfo<RoleDto>> queryPageRole(
            @RequestParam("example") String example,
            @RequestParam("filter") String filter,
            @RequestParam("pageNum") Integer pageNum,
            @RequestParam("pageSize") Integer pageSize);

    /**
     * 用户角色权限查询
     *
     * @param instanceId 应用ID
     * @param userId     用户Id
     * @param filter     tenantId, instanceId通过filter传递
     * @return UserAccessVo
     */
    @ApiOperation(value = "用户角色权限查询", notes = "用户角色权限查询，filter=暂未使用")
    @GetMapping(value = {"/{instanceId}/user/{userId}"}, produces = "application/json")
    @Capability(capabilityCode = "user.access.query-user-access")
    RestResponse<UserAccessVo> queryUserAccess(@PathVariable("instanceId") Long instanceId, @PathVariable("userId") Long userId, @RequestParam("filter") String filter);


    /**
     * 用户角色权限资源查询
     *
     * @param userId 用户Id
     * @return UserAccessVo
     */
    @ApiOperation(value = "用户角色权限资源查询", notes = "用户角色权限资源查询，" +
            "返回结构与/v1/access/{instanceId}/user/{userId}不一致，构建了菜单树" +
            ",filter :{ \"withNoFirstNode\"=1} 表示菜单树不要第一级节点，第一级节点不等于根节点code=0")
    @GetMapping(value = {"/user/{userId}/resources"}, produces = "application/json")
    @Capability(capabilityCode = "user.access.query-user-access-resources")
    RestResponse<UserAccessRespDto> queryUserAccessResources(@PathVariable("userId") Long userId, @RequestParam(value = "filter", required = false) String filter);

    /**
     * 用户组角色权限查询
     *
     * @param instanceId  应用ID
     * @param userGroupId 用户组Id
     * @param filter      tenantId, instanceId通过filter传递
     * @return UserAccessVo
     */
    @ApiOperation(value = "用户组角色权限查询", notes = "用户组角色权限查询，filter=暂未使用")
    @GetMapping(value = {"/{instanceId}/user-group/{userGroupId}"}, produces = "application/json")
    @Capability(capabilityCode = "user.access.query-user-group-access")
    RestResponse<UserAccessVo> queryUserGroupAccess(@PathVariable("instanceId") Long instanceId, @PathVariable("userGroupId") Long userGroupId, @RequestParam("filter") String filter);

    /**
     * 用户资源权限查询
     *
     * @param instanceId 应用ID
     * @param userId     用户Id
     * @param example    查询条件json字符串
     * @param filter     过滤条件
     * @return Integer
     */
    @ApiOperation(value = "用户资源权限查询  返回角色对应的权限总数 15=1:查询+2:编辑+4:删除+8:创建, filter支持code、action、name纬度")
    @GetMapping(value = {"/{instanceId}/permissions/of/user/{userId}"}, produces = "application/json")
    @ApiImplicitParam(name = "filter", value = "权限属性(按位操作) 1:查询,2:编辑,4:删除,8:创建 多选则累加起来 对应code、action、name、routePath", dataType = "String", required = true)
    @Capability(capabilityCode = "user.access.query-permissions")
    RestResponse<Integer> queryPermissions(@PathVariable("instanceId") Long instanceId,
                                           @PathVariable("userId") Long userId,
                                           @RequestParam("example") String example,
                                           @RequestParam("filter") String filter);

    /**
     * 查询用户的角色列表(包含应用名称，不包括资源信息)
     *
     * @param userRoleReqDto 查询条件
     * @return
     */
    @ApiOperation(value = "查询用户的角色列表(包含应用名称，不包括资源信息)")
    @GetMapping(value = {"/user/role"}, produces = "application/json")
    @Capability(capabilityCode = "user.access.query-user-role")
    RestResponse<List<UserRoleRespDto>> queryUserRole(@SpringQueryMap UserRoleReqDto userRoleReqDto);

    /**
     * 根据角色code查询角色的用户列表
     *
     * @param code 角色code
     * @return
     */
    @ApiOperation(value = "根据角色code查询角色的用户列表")
    @GetMapping(value = {"/role/user"}, produces = "application/json")
    @Capability(capabilityCode = "user.access.query-role-user")
    RestResponse<RoleUserRespDto> queryRoleUser(@RequestParam("code") String code);

    /**
     * 根据code查询角色模板
     *
     * @param code 角色模板code
     * @return 角色模板数据
     */
    @GetMapping("/role-template/{code}")
    @ApiOperation(value = "根据code查询角色模板", notes = "根据code查询角色模板")
    @Capability(capabilityCode = "user.access.query-by-code")
    RestResponse<RoleTemplateRespDto> queryByCode(@PathVariable("code") String code);

    /**
     * 角色模板分页数据
     *
     * @param filter   角色模板查询条件
     * @param pageNum  当前页
     * @param pageSize 页大小
     * @return 角色模板分页数据
     */
    @GetMapping("/role-template/page")
    @ApiOperation(value = "角色模板分页数据", notes = "根据filter查询条件查询角色模板数据，filter=RoleTemplateQueryReqDto")
    @Capability(capabilityCode = "user.access.query-by-page")
    RestResponse<PageInfo<RoleTemplateRespDto>> queryByPage(@RequestParam("filter") String filter,
                                                            @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                            @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 查询当前应用关联的角色，只返回50条角色记录，已不推荐使用，请使用带分页参数的重载版本
     * @param instanceId 应用id
     * @param userId 用户id
     * @return 当前应用关联的角色
     * @deprecated  3.0.9.0  建议使用{@link IAccessQueryApi#queryAppsAndRolesByPage}
     */
    @Deprecated
    @RequestMapping(value = "/{instanceId}/relate", method = RequestMethod.GET)
    @ApiOperation(value = "查询应用关联的角色(查询用户详情用)", notes = "isRelateUser表示角色是否已关联当前用户, 0: 未关联, 1: 已关联," +
            "由于只能返回50条关联的角色，已不推荐使用，建议使用IAccessQueryApi.queryAppsAndRolesByPage")
    @Capability(capabilityCode = "user.access.query-apps-and-roles")
    RestResponse<ApplicationAndRelatedRolesRespDto> queryAppsAndRoles(@PathVariable("instanceId") Long instanceId,
                                                                      @RequestParam(value = "userId", required = false) Long userId);

    /**
     * 分页查询当前应用关联的角色
     * @param instanceId 应用id
     * @param userId 用户id
     * @param pageNum 当前页
     * @param pageSize 页大小
     * @return 当前应用关联的角色
     */
    @RequestMapping(value = "/{instanceId}/relate-page", method = RequestMethod.GET)
    @ApiOperation(value = "分页查询应用关联的角色(查询用户详情用)", notes = "isRelateUser表示角色是否已关联当前用户, 0: 未关联, 1: 已关联")
    @Capability(capabilityCode = "user.access.query-apps-and-roles-by-page")
    RestResponse<PageInfo<ApplicationAndRelatedRolesRespDto>> queryAppsAndRolesByPage(@PathVariable("instanceId") Long instanceId,
                                                                            @RequestParam(value = "userId", required = false) Long userId,
                                                                            @RequestParam(name = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                                            @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    @RequestMapping(value = "detail/{id}/access", method = RequestMethod.GET)
    @ApiOperation(value = "根据id查询角色详情及资源权限(查询角色详情用)", notes = "根据id查询角色详情及资源权限")
    @Capability(capabilityCode = "user.access.query-by-id")
    RestResponse<RoleDetailAndAccessRespDto> queryById(@PathVariable("id") Long id);

    /**
     * 根据编码查询角色信息，仅查询简单的角色信息
     *
     * @param code
     * @return RoleRespDto
     */
    @GetMapping(value = "/role/by-code")
    @ApiOperation(value = "根据编码查询角色信息，仅查询简单的角色信息", notes = "根据编码查询角色信息，仅查询简单的角色信息")
    @Capability(capabilityCode = "user.access.query-role-by-code")
    RestResponse<RoleRespDto> queryRoleByCode(@RequestParam("code") String code);
}
