/**
 * @(#)IApplicationQueryApi.java 1.0 2018/6/21
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api.query;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.eo.SqlFilter;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.AppInstanceDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.ApplicationDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.ApplicationQueryRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.response.UserAppInstanceDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.AppInstanceQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.AppInstanceResourceDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.BizSpaceEnableInstanceReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.application.AppInstanceUserQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.application.ApplicationBatchQueryDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.*;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.util.List;

/**
 * 用户应用信息查询服务API接口
 *
 * @author 南枫
 * @since 1.0.0
 */
@Api(tags = {"用户中心：应用信息查询服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-query-IApplicationQueryApi", path = "/v1/application", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IApplicationQueryApi {

    /**
     * 通过id查询应用模板信息
     *
     * @param appId  应用ID
     * @param filter tenantId, instanceId通过filter传递
     * @return ApplicationDto
     */
    @ApiOperation(value = "查询应用模板简单信息", notes = "查询应用：" +
            "1、不返回菜单和按钮信息" +
            "2、选择是否返回角色模板列表，withRoleTemplate 0 否 1 是 默认否 ，使用filter传值=ApplicationQueryReqDto；" +
            "filter 格式：{\"withRoleTemplate\":0}")
    @RequestMapping(value = "/{appId}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-by-id")
    RestResponse<ApplicationDto> queryById(@PathVariable("appId") Long appId, @RequestParam("filter") String filter);

    /**
     * 通过id查询应用模板详细信息
     *
     * @param code 应用模板编码
     * @param res  表示查询按钮和菜单信息
     * @return ApplicationDto
     */
    @ApiOperation(value = "查询应用模板详细信息", notes = "查询应用，根据传参res带出相应的资源信息，filter=暂未使用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "应用模板编码", paramType = "query"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", paramType = "query", allowMultiple = true)
    })
    @RequestMapping(value = "/app-template", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-by-code-and-resources")
    RestResponse<ApplicationQueryResDto> queryByCodeAndResources(@RequestParam("code") String code,
                                                                 @RequestParam(value = "res", required = false) List<String> res);


    /**
     * 通过id查询应用模板信息
     *
     * @param appId  应用模板ID
     * @param res    表示查询按钮和菜单信息
     * @param filter tenantId, instanceId通过filter传递
     * @return ApplicationDto
     * @deprecated 2.6.21.0 通过绑定资源方式添加的应用模板菜单按钮，应使用 {@link IApplicationQueryApi#queryByCodeAndResources(String, List)}
     */
    @Deprecated
    @ApiOperation(value = "查询应用模板详细信息", notes = "查询应用模板，根据传参res带出相应的资源信息，filter=暂未使用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "appId", value = "应用模板id", paramType = "path"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", dataType = "String[]", paramType = "path"),
    })
    @RequestMapping(value = "/{appId}/resources/{res}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-by-id-and-resources")
    RestResponse<ApplicationDto> queryByIdAndResources(@PathVariable("appId") Long appId, @PathVariable("res") String[] res, @RequestParam("filter") String filter);

    /**
     * 查询所有应用模板列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询条件json字符串
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @ApiOperation(value = "查询所有应用模板列表", notes = "查询所有应用模板，不包括菜单和按钮信息，filter和example合并=ApplicationEo，" +
            "可支持name（模糊），code，applicationType，ownerId，status查询；withRoleTemplate==0为不查询角色模板，==1为查询角色模板")
    @RequestMapping(value = "/list", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "example", value = "查询条件json字符串,例如 {name:\"App1\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "查询条件", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),
    })
    @Capability(capabilityCode = "user.application.query-by-example")
    RestResponse<PageInfo<ApplicationDto>> queryByExample(@RequestParam(name = "example", required = false) String example,
                                                          @RequestParam(value = "filter", required = false, defaultValue = "{}") String filter,
                                                          @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                          @RequestParam(value = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 根据应用id查询应用详情
     *
     * @param instanceId 应用ID
     * @param filter     tenantId, instanceId通过filter传递
     * @return AppInstanceDto
     */
    @ApiOperation(value = "根据应用id查询应用详情", notes = "根据应用id查询应用信息，不包括菜单和按钮资源信息，filter=暂未使用")
    @RequestMapping(value = "/appInstance/{instanceId}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-ins-by-id")
    RestResponse<AppInstanceDto> queryInsById(@PathVariable("instanceId") Long instanceId, @RequestParam(value = "filter", required = false, defaultValue = "{}") String filter);

    /**
     * 查找应用对应的应用模板的资源并应用的菜单和按钮(用于初始化应用资源)
     *
     * @param instanceId 应用id
     * @param res        只能为buttons,menus或者buttons或者menus
     */
    @ApiOperation(value = "查询应用初始化资源", notes = "查询应用初始化资源")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", paramType = "query", allowMultiple = true)
    })
    @RequestMapping(value = "/appInstance/init", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-app-instance-init-resource")
    RestResponse<InstanceInitResourceResDto> queryAppInstanceInitResource(@RequestParam("instanceId") @NotNull(message = "应用id不能为空") Long instanceId,
                                                                          @RequestParam("res") @Size(min = 1, message = "res只能为buttons,menus或者buttons或者menus") List<String> res);

    /**
     * 通过id查询应用信息
     *
     * @param instanceId 应用id
     * @param res        表示查询按钮和菜单信息
     * @param filter     tenantId, instanceId通过filter传递
     * @return ApplicationDto
     */
    @ApiOperation(value = "查询应用详细信息", notes = "查询应用，根据传参res带出相应的资源信息，filter=暂未使用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query"),
            @ApiImplicitParam(name = "menuId", value = "菜单id", paramType = "query"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "filter", value = "过滤条件：AppInstanceResourceDto，格式为json字符串", dataType = "string", paramType = "query")
    })
    @RequestMapping(value = "/appInstance/resources", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-ins-by-instance-id-and-resources")
    @Deprecated
    RestResponse<AppInstanceQueryResDto> queryInsByInstanceIdAndResources(@RequestParam("instanceId") Long instanceId,
                                                                          @RequestParam(value = "menuId", required = false) Long menuId,
                                                                          @RequestParam("res") List<String> res,
                                                                          @RequestParam(value = "filter", required = false) String filter);

    /**
     * 查询应用资源信息
     * @param appInstanceResourceDto    应用实例资源查询dto
     * @return
     */
    @ApiOperation(value = "查询应用资源信息", notes = "查询应用资源信息")
    @RequestMapping(value = "/app-instance/resources", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-instance-resources")
    RestResponse<AppInstanceQueryResDto> queryInstanceResources(@SpringQueryMap AppInstanceResourceDto appInstanceResourceDto);


    /**
     * 查询应用下用户分配的菜单和按钮信息，BOC应用专用。其他应用查询建议使用用户角色权限查询/v1/access/{instanceId}/user/{userId}接口
     *
     * @param instanceId 应用id
     * @param userId     用户id
     * @param res        表示查询按钮和菜单信息
     * @param filter     查询条件
     * @return AppInstanceQueryResDto
     * @see #queryUserResources(AppInstanceUserQueryReqDto)
     */
    @Deprecated
    @ApiOperation(value = "查询应用下用户分配的菜单和按钮信息，BOC应用专用。其他应用查询建议使用用户角色权限查询/v1/access/{instanceId}/user/{userId}接口", notes = "查询应用下用户分配的菜单和按钮信息，根据传参res带出相应的资源信息，filter=暂未使用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "query"),
            @ApiImplicitParam(name = "userId", value = "用户id", paramType = "query"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", paramType = "query", allowMultiple = true),
            @ApiImplicitParam(name = "filter", value = "filter未启用", dataType = "string", paramType = "query")
    })
    @RequestMapping(value = "/appInstance/user/resources", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-ins-by-instance-id-and-user-id-and-resources")
    RestResponse<AppInstanceQueryResDto> queryInsByInstanceIdAndUserIdAndResources(@RequestParam("instanceId") Long instanceId,
                                                                                   @RequestParam("userId") Long userId,
                                                                                   @RequestParam("res") List<String> res,
                                                                                   @RequestParam(value = "filter", defaultValue = "{}") String filter);

    /**
     * 查询应用下用户分配的菜单和按钮信息(instanceId由应用识别)
     */
    @ApiOperation(value = "查询应用下用户分配的菜单和按钮信息", notes = "查询应用下用户分配的菜单和按钮信息，根据传参res带出相应的资源信息")
    @RequestMapping(value = "/app-instance/user/resources", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-user-resources")
    RestResponse<AppInstanceQueryResDto> queryUserResources(@SpringQueryMap AppInstanceUserQueryReqDto req);

    /**
     * 根据应用id查询应用详情,同时将应用下面的资源和菜单按钮查询到应用下返回给客户端
     *
     * @param instanceId 应用ID
     * @param res        表示查询按钮和菜单信息
     * @param filter     tenantId, instanceId通过filter传递
     * @return AppInstanceDto
     * @deprecated 2.6.21.0 通过绑定资源方式添加的应用模板菜单按钮，应使用 {@link IApplicationQueryApi#queryInsByInstanceIdAndResources}
     */
    @Deprecated
    @ApiOperation(value = "根据应用查询应用详情", notes = "appInstanceId应用id，res 资源信息可以包括menus,buttons，filter=暂未使用")
    @RequestMapping(value = "/appInstance/{instanceId}/resources/{res}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "instanceId", value = "应用id", paramType = "path"),
            @ApiImplicitParam(name = "res", value = "buttons,menus或者buttons或者menus，表示查询按钮和菜单信息", dataType = "String[]", paramType = "path"),
            @ApiImplicitParam(name = "filter", value = "filter未启用", dataType = "string", paramType = "query")
    })
    @Capability(capabilityCode = "user.application.query-ins-by-id-and-resources")
    RestResponse<AppInstanceDto> queryInsByIdAndResources(@PathVariable("instanceId") Long instanceId, @PathVariable("res") String[] res,
                                                          @RequestParam(value = "filter", defaultValue = "{}") String filter);

    /**
     * 根据应用模板id查询应用，一个应用模板包含多个应用，根据应用模板id查询多个实例
     *
     * @param appId  应用模板Id
     * @param filter tenantId, instanceId通过filter传递
     * @return AppInstanceDtoList
     */
    @ApiOperation(value = "根据应用模板id查询应用", notes = "根据应用模板id查询该应用全部的应用，filter=暂未使用")
    @RequestMapping(value = "/{appId}/appInstances", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-app-instance-list")
    RestResponse<List<AppInstanceDto>> queryAppInstanceList(@PathVariable("appId") String appId, @RequestParam("filter") String filter);


    @ApiOperation(value = "统计应用数量", notes = "统计应用数量")
    @RequestMapping(value = "/app-instance/count", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.count-app-instances")
    RestResponse<Integer> countAppInstances();


    /**
     * 根据用户查询应用，一个用户拥有多个应用，根据用户id查询多个实例
     *
     * @param owner  用户ID
     * @param filter tenantId, instanceId通过filter传递
     * @return AppInstanceDtoList
     */
    @ApiOperation(value = "根据用户id查询应用", notes = "根据用户id查询该用户所拥有的应用，实例列表不包括菜单和按钮信息，filter=暂未使用")
    @RequestMapping(value = "/appInstances/of/{owner}", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-app-instance-list")
    RestResponse<List<AppInstanceDto>> queryAppInstanceList(@PathVariable("owner") Long owner, @RequestParam("filter") String filter);


    /**
     * 查询应用列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询条件json字符串
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return AppInstanceDto
     * @see #queryInstances(AppInstanceQueryReqDto, Integer, Integer)
     */
    @ApiOperation(value = "查询所有应用信息", notes = "查询所有应用信息，不包括菜单和按钮信息，example和filter合并=AppInstanceEo，" +
            "支持name（模糊），hostName（模糊），purchaseType，owner，instanceId，status，appInsName纬度查询")
    @RequestMapping(value = "/appInstance/list", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "example", value = "查询条件json字符串,例如 {name:\"App1\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "查询条件", dataType = "string", paramType = "query", dataTypeClass = SqlFilter.class),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "integer", paramType = "query"),
    })
    @Deprecated
    @Capability(capabilityCode = "user.application.query-instance-by-example")
    RestResponse<PageInfo<AppInstanceDto>> queryInstanceByExample(@RequestParam(name = "example", required = false) String example,
                                                                  @RequestParam(value = "filter", defaultValue = "{}") String filter,
                                                                  @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                  @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询应用概况(code,name,status) ,用于前端下拉列表
     *
     * @param queryReq 查询条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return AppInstanceDto
     */
    @ApiOperation(value = "查询所有应用概况")
    @RequestMapping(value = "/app-instance/summary", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-instance-summaries")
    RestResponse<List<AppInstanceSummaryResDto>> queryInstanceSummaries(@SpringQueryMap AppInstanceQueryReqDto queryReq,
                                                                        @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                        @RequestParam(value = "pageSize", defaultValue = "1000") Integer pageSize);

    /**
     * 查询应用列表
     *
     * @param queryReq 查询条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return AppInstanceDto
     */
    @ApiOperation(value = "查询所有应用信息", notes = "查询所有应用信息，不包括菜单和按钮信息")
    @RequestMapping(value = "/app-instance", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-instances")
    RestResponse<PageInfo<AppInstanceDto>> queryInstances(@SpringQueryMap AppInstanceQueryReqDto queryReq,
                                                          @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                          @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 根据域名或者appInstaceId查询应用
     *
     * @param key    key
     * @param filter tenantId, instanceId通过filter传递
     * @return AppInstanceDto
     * @see #findByHostname(String)  替代
     */
    @Deprecated
    @ApiOperation(value = "根据域名或者appInstaceId查询应用", notes = "根据域名或者appInstaceId查询应用，filter=暂未使用")
    @RequestMapping(value = "{key}/appInstance", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-by-host-name")
    RestResponse<AppInstanceDto> queryByHostName(@PathVariable("key") String key, @RequestParam("filter") String filter);


    /**
     * 根据域名或者appInstaceId查询应用
     *
     * @param hostname 域名
     * @return AppInstanceDto
     */
    @ApiOperation(value = "根据域名或者appInstaceId查询应用", notes = "根据域名或url")
    @RequestMapping(value = "/instance", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.find-by-hostname")
    RestResponse<AppInstanceDto> findByHostname(@RequestParam("hostname") String hostname);

    /**
     * 一次性根据参数查询应用
     *
     * @param applicationBatchQueryDto 域名
     * @return AppInstanceDto
     */
    @ApiOperation(value = "一次性根据参数查询应用", notes = "根据域名或url")
    @RequestMapping(value = "/instance/one-time", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    RestResponse<AppInstanceDto> findByHostnameInOneTime(@SpringQueryMap ApplicationBatchQueryDto applicationBatchQueryDto);

    /**
     * 根据userId查询用户授权的应用列表
     *
     * @param userId 用户id
     * @return 应用id列表
     * @see #queryInstanceByTenantId
     */
    @Deprecated
    @ApiOperation(value = "根据userId查询用户授权的应用列表", notes = "根据userId查询用户授权的应用列表")
    @RequestMapping(value = "{userId}/appInstance/list", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-user-auth-instance-list")
    RestResponse<List<Long>> queryUserAuthInstanceList(@PathVariable("userId") Long userId);

    /**
     * 返回该租户下业务空间的有效应用列表id，该接口提供给业务空间的有效应用数目展示
     *
     * @param bizSpaceEnableInstanceReqDto 租户id 以及业务空间code列表
     * @return BizSpaceEnableInstanceRespDto
     */
    @ApiOperation(value = "返回 业务空间下 instanceId 列表 中启用的数量，该接口提供给业务空间的有效应用数目展示",
            notes = "返回业务空间下  instanceId 列表 中启用的数量，该接口提供给业务空间的有效应用数目展示")
    @RequestMapping(value = "/biz-space/appInstance/enable", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-enable-instance-num")
    RestResponse<List<BizSpaceEnableInstanceRespDto>> queryEnableInstanceNum(@Valid @SpringQueryMap BizSpaceEnableInstanceReqDto bizSpaceEnableInstanceReqDto);


    /**
     * 根据租户ID查询应用列表，同时返回用户是否拥有应用的访问权限(该接口会过滤掉MPC应用)
     * 仅mpc/boc前端可调用本接口
     *
     * @param tenantId 租户ID
     * @param userId   用户ID
     * @return AppInstanceDto List
     */
    @ApiOperation(value = "根据租户ID查询应用列表", notes = "根据租户ID查询应用列表，同时返回用户是否拥有应用的访问权限(该接口会过滤掉MPC应用), 仅mpc/boc前端可调用本接口")
    @RequestMapping(value = "/tenant/{tenantId}/instances", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-instance-by-tenant-id")
    RestResponse<List<UserAppInstanceDto>> queryInstanceByTenantId(@PathVariable("tenantId") Long tenantId,
                                                                   @RequestParam(value = "userId", required = false) Long userId);

    /**
     * 查询租户下的应用列表
     * ps: 如果不是MPC应用,则返回结果中不包含MPC
     *
     * @param tenantId 租户id
     * @return 应用列表
     */
    @ApiOperation(value = "查询租户下的应用列表", notes = "租户概览页使用, 如果发起请求的系统不是MPC, 则去掉MPC的信息")
    @RequestMapping(value = "/tenant/{tenantId}/app", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-app-by-tenant-id")
    RestResponse<List<ApplicationQueryRespDto>> queryAppByTenantId(@PathVariable("tenantId") Long tenantId);

    /**
     * 根据租户ID查询应用
     * User
     *
     * @param tenantId 租户ID
     * @param filter   tenantId, instanceId通过filter传递
     * @return AppInstanceDto List
     */
    @ApiOperation(value = "根据租户ID查询应用", notes = "根据租户ID查询应用，filter=暂未使用")
    @RequestMapping(value = "/tenant/{tenantId}/appInstances", produces = {"application/json;charset=UTF-8"}, method = RequestMethod.GET)
    @Capability(capabilityCode = "user.application.query-app-instance-by-tenant-id")
    RestResponse<List<AppInstanceDto>> queryAppInstanceByTenantId(@PathVariable("tenantId") Long tenantId, @RequestParam("filter") String filter);
}
