/**
 * @(#) IAccessApi.java 1.0 2018-06-20
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api;

import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.RoleAccessDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.RoleDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.request.RoleCreateReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.boc.request.RoleModifyDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 用户授权服务API接口
 *
 * @author 黑马
 * @since 1.0.0
 */
@Api(tags = {"用户中心：授权服务"})
@FeignClient(contextId = "com-dtyunxi-yundt-cube-center-user-api-IAccessApi", path = "/v1/access", name = "${dtyunxi.yundt.cube_base-center-user_api.name:base-center-user}", url = "${dtyunxi.yundt.cube_base-center-user_api:}")
public interface IAccessApi {

    /**
     * 批量创建角色
     *
     * @param instanceId    应用实例Id
     * @param createReqDtos 角色List
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/role"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "批量创建角色")
    @Capability(capabilityCode = "user.access.create-role")
    RestResponse<Void> createRole(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<RoleAccessCreateReqDto> createReqDtos);

    /**
     * 创建单个角色
     *
     * @param instanceId   应用实例Id
     * @param createReqDto 角色新增Dto
     * @return RoleDto 角色信息
     */
    @RequestMapping(value = {"/{instanceId}/role/single"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "创建单个角色")
    @Capability(capabilityCode = "user.access.create-role")
    RestResponse<RoleDto> createRole(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody RoleAccessCreateReqDto createReqDto);

    /**
     * 根据编码更新角色
     *
     * @param code             角色编码
     * @param roleModifyReqDto 修改角色信息
     * @return void
     */
    @RequestMapping(value = {"/role/code/{code}"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "根据编码更新角色")
    @Capability(capabilityCode = "user.access.update-role-by-code")
    RestResponse<Void> updateRoleByCode(@PathVariable("code") String code, @Valid @RequestBody RoleModifyReqDto roleModifyReqDto);

    /**
     * 更新角色
     *
     * @param roleId       角色Id
     * @param modifyReqDto 角色更新dto
     * @return void
     */
    @RequestMapping(value = {"/role/{roleId}"}, produces = "application/json", method = RequestMethod.PUT)
    @ApiOperation(value = "更新角色")
    @Capability(capabilityCode = "user.access.update-role")
    RestResponse<Void> updateRole(@PathVariable("roleId") Long roleId, @Valid @RequestBody RoleAccessModifyReqDto modifyReqDto);

    /**
     * 批量删除角色
     *
     * @param roleIdList 角色IdList
     * @return void
     */
    @RequestMapping(value = {"/role"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "批量删除角色，如果用户角色或者用户组角色还有对应角色的记录，则删除失败")
    @Capability(capabilityCode = "user.access.delete-role")
    RestResponse<Void> deleteRole(@RequestBody List<Long> roleIdList);

    /**
     * 批量删除角色
     *
     * @param roleRemoveReqDto 角色codeList
     * @return void
     */
    @RequestMapping(value = {"/role/code"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "根据code列表批量删除角色，如果用户角色或者用户组角色还有对应角色的记录，则删除失败")
    @Capability(capabilityCode = "user.access.delete-role-by-code")
    RestResponse<Void> deleteRoleByCode(@SpringQueryMap RoleRemoveReqDto roleRemoveReqDto);

    /**
     * 角色关联资源、操作权限，增量授权
     *
     * @param instanceId     应用实例Id
     * @param roleAccessList 角色权限List
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/role/permissions"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "角色关联资源、操作权限（增量授权）", notes = "roleAccessList参数为增量的资源和权限，调用该接口将在不清除原关联的资源和操作权限基础上叠加数据")
    @Capability(capabilityCode = "user.access.give-role-resources")
    RestResponse<Void> giveRoleResources(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<RoleAccessDto> roleAccessList);


    /**
     * 角色关联资源、操作权限,全部授权
     *
     * @param instanceId     应用实例Id
     * @param roleAccessList 角色权限List
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/role/permissions/all"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "角色关联资源、操作权限(全部授权)", notes = "roleAccessList参数为全部的资源和权限，调用该接口将在清除原关联的资源和操作权限基础上重新关联资源和权限")
    @Capability(capabilityCode = "user.access.add-role-resources")
    RestResponse<Void> addRoleResources(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<RoleAccessDto> roleAccessList);

    /**
     * 角色解除资源、操作权限,删除关联关系
     *
     * @param roleResourceRemoveReqDtos 角色资源权限解除List
     * @return void
     */
    @RequestMapping(value = {"/role/resources/access"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "角色解除资源、操作权限(删除关联关系)", notes = "角色解除资源、操作权限(删除关联关系)")
    @Capability(capabilityCode = "user.access.remove-role-resources-relation")
    RestResponse<Void> removeRoleResourcesRelation(@RequestBody List<RoleResourceRemoveReqDto> roleResourceRemoveReqDtos);

    /**
     * 角色解除资源、操作权限(操作权限变更，不删除关联关系)
     *
     * @param instanceId     应用实例Id
     * @param roleAccessList 角色权限List
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/role/permissions"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "角色解除资源、操作权限(操作权限变更，不删除关联关系)")
    @Capability(capabilityCode = "user.access.remove-role-resources")
    RestResponse<Void> removeRoleResources(@PathVariable("instanceId") Long instanceId, @Valid @RequestBody List<RoleAccessDto> roleAccessList);

    /**
     * 给用户赋予角色(追加赋予)
     *
     * @param instanceId 应用实例Id
     * @param userId     用户Id
     * @param roleIdList 角色IdList
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/user/{userId}/roles"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "给用户赋予角色(追加赋予)", notes = "该接口不会清除原绑定角色，每次都追加赋予角色")
    @Capability(capabilityCode = "user.access.give-user-roles")
    RestResponse<Void> giveUserRoles(@PathVariable("instanceId") Long instanceId,
                                     @PathVariable("userId") Long userId,
                                     @RequestBody List<Long> roleIdList);

    /**
     * 给用户赋予角色（重新赋予）
     *
     * @param instanceId 应用实例Id
     * @param userId     用户Id
     * @param roleIdList 角色IdList
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/user/{userId}/roles/all"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "给用户赋予角色（重新赋予）", notes = "该接口将清除原绑定角色，重新赋予角色")
    @Capability(capabilityCode = "user.access.add-user-roles")
    RestResponse<Void> addUserRoles(@PathVariable("instanceId") Long instanceId,
                                    @PathVariable("userId") Long userId,
                                    @RequestBody List<Long> roleIdList);


    /**
     * 根据code给用户赋予角色（重新赋予）
     *
     * @param userId       用户Id
     * @param roleCodeList 角色codeList
     * @return void
     */
    @RequestMapping(value = {"/user/{userId}/roles/code"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "给用户赋予角色（重新赋予）", notes = "该接口将清除原绑定角色，重新赋予角色")
    @Capability(capabilityCode = "user.access.add-user-roles-by-code")
    RestResponse<Void> addUserRolesByCode(@PathVariable("userId") Long userId,
                                          @RequestBody List<String> roleCodeList);


    /**
     * 给用户组授予角色
     *
     * @param instanceId  应用实例Id
     * @param userGroupId 用户组Id
     * @param roleIdList  角色IdList
     * @return void
     */
    @RequestMapping(value = {"/{instanceId}/user-group/{userGroupId}/roles"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "给用户组授予角色")
    @Capability(capabilityCode = "user.access.give-user-group-roles")
    RestResponse<Void> giveUserGroupRoles(@PathVariable("instanceId") Long instanceId,
                                          @PathVariable("userGroupId") Long userGroupId,
                                          @RequestBody List<Long> roleIdList);

    /**
     * 根据code给用户组赋予角色（重新赋予）
     *
     * @param userGroupId  用户组Id
     * @param roleCodeList 角色codeList
     * @return void
     */
    @RequestMapping(value = {"/user-group/{userGroupId}/roles/code"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "根据code给用户组赋予角色（重新赋予）", notes = "该接口将清除原绑定角色，重新赋予角色")
    @Capability(capabilityCode = "user.access.add-user-group-roles-by-code")
    RestResponse<Void> addUserGroupRolesByCode(@PathVariable("userGroupId") Long userGroupId,
                                               @RequestBody List<String> roleCodeList);

    /**
     * 解除用户角色（角色code）
     *
     * @param userId    用户ID
     * @param roleCodes 角色code
     * @return 操作结果
     */
    @RequestMapping(value = {"/{userId}/roles/code"}, produces = "application/json", method = RequestMethod.DELETE)
    @ApiOperation(value = "解除用户角色(角色code）", notes = "解除用户角色，userId：用户ID \t\n roleCodes：角色code列表")
    @Capability(capabilityCode = "user.access.remove-user-roles")
    RestResponse<Void> removeUserRoles(@PathVariable("userId") Long userId, @RequestParam("roleCodes") String roleCodes);

    /**
     * 角色模板生成角色,同时角色模板与权限模板的关系 对应生成 角色与权限的关系
     *
     * @param roleGenerateReqDtos 角色模板生成角色请求dto列表
     * @return
     */
    @PostMapping("/conversion/role/")
    @ApiOperation(value = "角色模板生成角色", notes = "角色模板生成角色,同时角色模板与权限模板的关系 对应生成 角色与权限的关系")
    @Capability(capabilityCode = "user.access.generate-role")
    RestResponse<Void> generateRole(@RequestBody List<RoleGenerateReqDto> roleGenerateReqDtos);

    /**
     * 角色模板关联资源、操作权限 ,对应的是权限模板
     *
     * @param roleAccessTemCreateReqDtos
     * @return
     */
    @PostMapping("/role-template/resource")
    @ApiOperation(value = "角色模板关联资源、操作权限，对应的是权限模板", notes = "角色模板关联资源、操作权限，对应的是权限模板")
    @Capability(capabilityCode = "user.access.give-role-template-resource")
    RestResponse<Void> giveRoleTemplateResource(@RequestBody List<RoleAccessTemCreateReqDto> roleAccessTemCreateReqDtos);

    /**
     * 角色模板解除资源、操作权限 ,对应的是权限模板
     *
     * @param roleAccessTemRemoveReqDtos
     * @return
     */
    @DeleteMapping("/role-template/resource")
    @ApiOperation(value = "角色模板解除资源、操作权限 ,对应的是权限模板", notes = "角色模板解除资源、操作权限 ,对应的是权限模板")
    @Capability(capabilityCode = "user.access.remove-role-template-resource")
    RestResponse<Void> removeRoleTemplateResource(@RequestBody List<RoleAccessTemRemoveReqDto> roleAccessTemRemoveReqDtos);


    /**
     * 新增角色模板
     *
     * @param roleTemplateCreateReqDto 角色模板请求对象
     * @return 处理结果
     */
    @PostMapping("/role-template")
    @ApiOperation(value = "新增角色模板", notes = "新增角色模板")
    @Capability(capabilityCode = "user.access.add-role-template")
    RestResponse<Long> addRoleTemplate(@RequestBody RoleTemplateCreateReqDto roleTemplateCreateReqDto);

    /**
     * 修改角色模板
     *
     * @param code                     角色模板编码
     * @param roleTemplateModifyReqDto 角色模板修改请求对象
     * @return 处理结果
     */
    @PutMapping("/role-template/{code}")
    @ApiOperation(value = "修改角色模板", notes = "修改角色模板")
    @Capability(capabilityCode = "user.access.modify-role-template")
    RestResponse<Void> modifyRoleTemplate(@PathVariable("code") String code, @RequestBody RoleTemplateModifyReqDto roleTemplateModifyReqDto);

    /**
     * 删除角色模板
     *
     * @param codes 角色模板删除数据ID
     * @return 处理结果
     */
    @DeleteMapping("/role-template/{codes}")
    @ApiOperation(value = "删除角色模板", notes = "删除角色模板")
    @Capability(capabilityCode = "user.access.remove-role-template")
    RestResponse<Void> removeRoleTemplate(@PathVariable("codes") List<String> codes);

    /**
     * 根据角色code与用户ID列表建立关联关系
     *
     * @param roleUserRelationReqDto 角色关联用户ID列表请求dto
     * @return void
     */
    @RequestMapping(value = {"/role/user/relate"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "根据角色code与用户ID列表建立关联关系", notes = "根据角色code与用户ID列表建立关联关系,覆盖性关联")
    @Capability(capabilityCode = "user.access.set-role-user-relation")
    RestResponse<Void> setRoleUserRelation(@RequestBody RoleUserRelationReqDto roleUserRelationReqDto);

    /**
     * 角色批量关联用户
     *
     * @param instanceId 应用实例ID
     * @param roleId     角色ID
     * @param userIdList 用户ID列表
     * @return 操作结果
     */
    @RequestMapping(value = {"/{instanceId}/role/{roleId}/users"}, produces = "application/json", method = RequestMethod.POST)
    @ApiOperation(value = "角色批量关联用户", notes = "该接口不会清除用户原绑定角色，每次都追加赋予角色")
    RestResponse grantUserRoles(@NotNull(message = "实例ID不允许为空") @PathVariable("instanceId") Long instanceId,
                                @NotNull(message = "角色ID不允许为空") @PathVariable("roleId") Long roleId,
                                @RequestBody List<Long> userIdList);

    /**
     * 新增角色并分配资源
     *
     * @param roleCreateReqDto 创建角色dto
     * @param instanceId       应用id
     * @return void
     */
    @RequestMapping(value = "/{instanceId}", method = RequestMethod.POST)
    @ApiOperation(value = "新增角色并分配资源", notes = "新增角色并分配资源, RoleAccessDto角色id无需填写")
    @Capability(capabilityCode = "user.access.create-role-and-give-resource")
    RestResponse<Long> createRoleAndGiveResource(@RequestBody RoleCreateReqDto roleCreateReqDto, @PathVariable("instanceId") Long instanceId);

    @RequestMapping(value = "/{roleId}", method = RequestMethod.PUT)
    @ApiOperation(value = "编辑角色并分配资源", notes = "编辑角色并分配资源")
    @Capability(capabilityCode = "user.access.modify-role-and-give-resource")
    RestResponse<Void> modifyRoleAndGiveResource(@PathVariable("roleId") Long roleId, @RequestBody RoleModifyDto roleModifyDto);


}
