/**
 * @(#)IOrganizationQueryApi.java 1.0 2018/6/21
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.user.api.query;

import com.dtyunxi.annotation.DataLimitCalcFunc;
import com.dtyunxi.cube.enhance.generator.annotation.Capability;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.constant.RegexConstants;
import com.dtyunxi.yundt.cube.center.user.api.dto.EmployeeDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.OrgStructureTreeRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.OrganizationDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.UserOrgQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.UserOrgRelationQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.OrgAndOrgInfoRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserRespDto;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.Pattern;
import java.util.List;

/**
 * 用户组织机构查询服务API接口
 *
 * @author 南枫
 * @since 1.0.0
 */
@Api(tags = {"用户中心：组织机构查询服务(v1)"})
@FeignClient(
        contextId = "com-dtyunxi-yundt-cube-center-user-api-query-IOrganizationQueryApi",
        name = "${dtyunxi.yundt.cube_organization-center-user_api.name:organization-center-user}",
        url = "${dtyunxi.yundt.cube_organization-center-user_api:}",
        path = "/v1/organization"
)
public interface IOrganizationQueryApi {

    /**
     * 根据应用实例获取整个组织树信息 通过实例获取组织信息
     *
     * @param instanceId 应用实例ID
     * @return OrganizationDto
     */
    @RequestMapping(method = RequestMethod.GET)
    @ApiOperation(value = "查询指定应用实例下的组织机构", notes = "请求参数为： Application-Key 应用实例id 查询条件json字符串没有实际使用, 传{}")
    @Capability(capabilityCode = "user.organization.query-one")
    RestResponse<OrganizationDto> queryOne(@RequestParam("instanceId") Long instanceId, @RequestParam("filter") String filter);

    /**
     * 根据id或者姓名或者员工编号查询员工列表信息，返回员工基本信息和所在组织列表，没有扩展信息 查询员工列表
     *
     * @param organizationId 组织id
     * @param example        查询条件json字符串
     * @param filter         查询条件json字符串 使用 com.dtyunxi.yundt.cube.center.user.api.dto.EmployeeDto
     * @param pageNum        起始页
     * @param pageSize       每一页的记录数
     * @return PageInfo
     * @see EmployeeQueryReqDto
     */
    @RequestMapping(value = "/employees", method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "organizationId", value = "组织id", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "example", value = "(已废弃,新接入使用建议统一使用filter作为查询条件)查询条件json字符串,例如 {name:\"张三\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "查询条件json字符串,对应的实体类为: com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeQueryReqDto ." +
                    "例如 {\n" +
                    "  \"organizationId\": //对应employee.organization_id,\n" +
                    "  \"employeeNo\": \"demoData\",\n" +
                    "  \"station\": \"demoData\",\n" +
                    "  \"name\": \"demoData\",\n" +
                    "  \"email\": \"demoData\",\n" +
                    "  \"phoneNum\": \"demoData\",\n" +
                    "  \"userId\": 1,\n" +
                    "  \"tenantId\": 1,\n" +
                    "  \"instanceId\": 1,\n" +
                    "  \"returnWithUser\": true,//是否返回用户信息\n" +
                    "  \"returnWithOrgIds\": true //是否返回orgIdSet\n" +
                    "}",
                    dataType = "com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeQueryReqDto",
                    paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query")})
    @ApiOperation(value = "根据id或者姓名或者员工编号查询员工列表信息", notes = "返回员工基本信息和所在组织列表，没有扩展信息 查询员工列表, 单次查询限额1000条")
    @Capability(capabilityCode = "user.organization.query-page")
    RestResponse<PageInfo<EmployeeDto>> queryPage(@RequestParam(name = "organizationId", required = false) Long organizationId,
                                                  @RequestParam(name = "example", required = false) String example,
                                                  @RequestParam("filter") String filter,
                                                  @RequestParam(value = "pageNum", required = false, defaultValue = "1") Integer pageNum,
                                                  @RequestParam(value = "pageSize", required = false, defaultValue = "10") Integer pageSize);

    /**
     * 【改造的需求】根据id或者姓名或者员工编号查询员工列表信息，返回员工基本信息和所在组织列表，没有扩展信息 查询员工列表:新增userType列表筛选
     *
     * @param organizationId 组织id
     * @param filter         查询条件json字符串
     * @param userTypes      用户类型数组
     * @param pageNum        起始页
     * @param pageSize       每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = "/employee/list", method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "organizationId", value = "组织id", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "查询条件json字符串,例如 {\"name\":\"王成\", \"instanceId\":1, \"tenantId\":1}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "userTypes", value = "用户类型数组(逗号隔开)", dataType = "String[]", paramType = "query", required = false),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    @ApiOperation(value = "【改造的需求】根据id或者姓名或者员工编号查询员工列表信息:新增userType列表筛选", notes = "返回员工基本信息和所在组织列表，没有扩展信息 查询员工列表")
    @Capability(capabilityCode = "user.organization.query-page")
    RestResponse<PageInfo<EmployeeDto>> queryPage(@RequestParam(name = "organizationId", required = false) Long organizationId,
                                                  @RequestParam("filter") String filter,
                                                  @RequestParam(name = "userTypes", required = false) String[] userTypes,
                                                  @RequestParam("pageNum") Integer pageNum,
                                                  @RequestParam("pageSize") Integer pageSize);

    /**
     * 【改造的需求】IDEB-1206通过客户名、手机、邮箱模糊分页查询user表active_state=1的数据
     *
     * @param filter   查询条件json字符串
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @RequestMapping(value = "/employees/user/active", method = RequestMethod.GET)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "查询条件json字符串,例如 {name:\"王成\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    @ApiOperation(value = "分页查询已激活员工的数据", notes = "分页查询已激活员工的数据" +
            "单次查询限额1000条")
    @Capability(capabilityCode = "user.organization.query-page-by-active-user")
    RestResponse<PageInfo<EmployeeDto>> queryPageByActiveUser(@RequestParam("filter") String filter,
                                                              @RequestParam("pageNum") Integer pageNum,
                                                              @RequestParam("pageSize") Integer pageSize);

    /**
     * 根据id查询员工详细信息，返回员工基本信息和所在组织列表，和扩展信息 根据员工id查询员工详情
     *
     * @param id 员工id
     * @return EmployeeDto
     */
    @RequestMapping(value = "/employee/{id}", method = RequestMethod.GET)
    @ApiOperation(value = "查询员工详情", notes = "请求参数为： id 组织id filter: 该查询参数无效")
    @Capability(capabilityCode = "user.organization.employee-detail")
    RestResponse<EmployeeDto> employeeDetail(@PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 根据用户id查询员工列表
     *
     * @param userId 用户id
     * @return EmployeeDtoList
     */
    @RequestMapping(value = "/employee/user/{userId}", method = RequestMethod.GET)
    @ApiOperation(value = "根据用户id查询员工列表", notes = "请求参数为： userId 用户id filter: 该查询参数无效")
    @Capability(capabilityCode = "user.organization.query-employee-by-user-id")
    RestResponse<List<EmployeeDto>> queryEmployeeByUserId(@PathVariable("userId") Long userId, @RequestParam("filter") String filter);

    /**
     * 根据用户ids查询员工列表，id个数不能超过1000
     *
     * @param ids    多个用户id
     * @param filter tenantId, instanceId通过filter传递
     * @return EmployeeDtoList
     */
    @GetMapping(value = {"/by/{ids}"}, produces = "application/json")
    @ApiOperation(value = "根据多个用户id查询员工列表", notes = "根据多个用户id查询员工列表，ids多个id逗号连接，id个数不能超过1000")
    @Capability(capabilityCode = "user.organization.query-employee-by-user-id-list")
    RestResponse<List<EmployeeDto>> queryEmployeeByUserIdList(@PathVariable("ids") String ids, @RequestParam("filter") String filter);

    /**
     * 根据组织机构的id查询返回组织机构信息
     *
     * @param id 组织机构ID
     * @return OrganizationDto
     */
    @Deprecated
    @RequestMapping(value = "/{id}", method = RequestMethod.GET)
    @ApiOperation(value = "查询组织机构信息（废弃）", notes = "根据组织机构的ID查询返回组织的信息  filter: 该查询参数无效;" +
            "废弃：由【com.dtyunxi.yundt.cube.center.user.api.query.IOrganizationQueryExtApi.queryById】替换，" +
            "接口不兼容，type记录区分业务单元、虚拟单元，历史类型：【公司】【部门】由entity_prop_code记录，使用新接口必须结合功能迁移使用接口；")
    @ApiImplicitParam(name = "id", value = "组织机构Id", dataType = "long", paramType = "path", required = true)
    @Capability(capabilityCode = "user.organization.query-by-id")
    RestResponse<OrganizationDto> queryById(@PathVariable("id") Long id, @RequestParam("filter") String filter);

    /**
     * 查询组织机构列表
     *
     * @param example  查询条件json字符串
     * @param filter   查询条件json字符串
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @param sort     排序字段
     * @return PageInfo
     */
    @Deprecated
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    @ApiOperation(value = "查询组织机构列表（废弃）",
            notes = "分页查询组织列表;废弃，由【com.dtyunxi.yundt.cube.center.user.api.query.IOrganizationQueryExtApi.queryPage  查询组织业务单元虚拟单元】与【com.dtyunxi.yundt.cube.center.user.api.query.IOrgGroupQueryApi.queryByPage  查询根组织信息】" +
                    "结合使用替换，接口不兼容，type记录区分业务单元、虚拟单元，历史类型：【公司】【部门】由entity_prop_code记录，使用新接口必须结合功能迁移使用接口；" +
                    "该接口兼容旧逻辑，把entity_prop_code为公司的转为type=0;把entity_prop_code为部门的转为type=1")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "example", value = "查询条件json字符串,例如 {name:\"app1\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "filter", value = "(新接入使用建议统一使用filter作为查询条件)查询条件json字符串,例如 {name:\"人事部\"}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "sort", value = "排序字段，格式如:age_desc,name_asc", dataType = "string", paramType = "query"),})
    @Capability(capabilityCode = "user.organization.query-by-example")
    RestResponse<PageInfo<OrganizationDto>> queryByExample(@RequestParam(name = "example", required = false) String example,
                                                           @RequestParam("filter") String filter,
                                                           @RequestParam("pageNum") Integer pageNum,
                                                           @RequestParam("pageSize") Integer pageSize,
                                                           @Pattern(regexp = RegexConstants.REGEX_SORT_STR, message = "sort 输入的格式不正确,格式为：属性名_asc|desc") @RequestParam(name = "sort", required = false) String sort);

    /**
     * 根据组织机构的ID查询返回组织的信息,指定items时可指定的附加信息
     *
     * @param id    组织机构Id
     * @param items 组织机构的附加信息
     * @return OrganizationDto
     */
    @RequestMapping(value = "/{id}/sub/{items}", method = RequestMethod.GET)
    @ApiOperation(value = "查询组织机构的详细信息", notes = "根据组织机构的ID查询返回组织的信息及其员工和下级机构,输入items可以查询机构下面的员工或者下级机构，或查多个时可用逗号隔开,如employee,org" +
            "filter格式参考:{\"instanceId\":1,\"tenantId\":1,\"name\":\"aa\"}")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "组织机构Id", dataType = "long", paramType = "path", required = true),
            @ApiImplicitParam(name = "items", value = "组织机构的附加信息(如“employee,org”)", dataType = "string", paramType = "path", required = true),
            @ApiImplicitParam(name = "filter", value = "filter", dataType = "string", paramType = "query")
    })
    @Capability(capabilityCode = "user.organization.query-org-and-items-by-id")
    RestResponse<OrganizationDto> queryOrgAndItemsById(@PathVariable("id") Long id, @PathVariable("items") String[] items, @RequestParam("filter") String filter);

    /**
     * 根据租户id查询根组织信息
     *
     * @param tenantId 租户Id
     * @param filter
     * @return
     */
    @RequestMapping(value = "/root/{tenantId}", method = RequestMethod.GET)
    @ApiOperation(value = "根据租户id查询根组织信息", notes = "根据租户Id查询返回租户对应的根组织的信息 filter: 该查询参数无效")
    @ApiImplicitParam(name = "tenantId", value = "租户Id", dataType = "long", paramType = "path", required = true)
    @Capability(capabilityCode = "user.organization.query-root-by-tenant-id")
    RestResponse<List<OrganizationDto>> queryRootByTenantId(@PathVariable("tenantId") Long tenantId, @RequestParam("filter") String filter);

    /**
     * 通过组织名称反查该组织所有父节点
     *
     * @param name     组织名称
     * @param tenantId 租户id
     * @return RestResponse<List < List < OrgStructureTreeRespDto>>>
     */
    @GetMapping(value = "/query/structure/tree")
    @ApiOperation(value = "通过组织名称反查该组织所有父节点", notes = "通过组织名称反查该组织所有父节点")
    @Capability(capabilityCode = "user.organization.query-structure-tree-by-org-name")
    RestResponse<List<List<OrgStructureTreeRespDto>>> queryStructureTreeByOrgName(@RequestParam("name") String name, @RequestParam("tenantId") Long tenantId);

    /**
     * 通过组织id反查该组织所有父节点
     *
     * @param orgId 组织id
     * @return RestResponse<List < OrgStructureTreeRespDto>>
     */
    @GetMapping(value = "/query/org/{orgId}/structure/tree")
    @ApiOperation(value = "通过组织id反查该组织所有父节点", notes = "通过组织id反查该组织所有父节点")
    @ApiImplicitParam(name = "orgId", value = "组织id", dataType = "Long", paramType = "path", required = true)
    @Capability(capabilityCode = "user.organization.query-structure-tree-by-org-id")
    RestResponse<List<OrgStructureTreeRespDto>> queryStructureTreeByOrgId(@PathVariable("orgId") Long orgId);

    /**
     * 根据组织id查询组织及企业详情
     *
     * @param orgId 组织id
     * @return 组织及企业详情
     */
    @RequestMapping(value = "/{orgId}/org-Info", method = RequestMethod.GET)
    @ApiOperation(value = "根据组织id查询组织及企业详情", notes = "根据组织id查询组织及企业详情")
    @Capability(capabilityCode = "user.organization.query-org-and-org-info")
    RestResponse<OrgAndOrgInfoRespDto> queryOrgAndOrgInfo(@PathVariable("orgId") Long orgId);

    /**
     * 查询未关联当前组织的员工列表
     *
     * @param orgId        组织id
     * @param employeeName 员工名称
     * @param employeeNo   员工编号
     * @param phoneNum     手机号
     * @param pageNum      当前页
     * @param pageSize     每页数据量
     * @return 员工列表
     */
    @RequestMapping(value = "employee-list", method = RequestMethod.GET)
    @ApiOperation(value = "查询未关联当前组织的员工列表", notes = "查询未关联当前组织的员工列表")
    @Capability(capabilityCode = "user.organization.query-not-relate-employees-by-org-id")
    RestResponse<PageInfo<EmployeeDto>> queryNotRelateEmployeesByOrgId(@RequestParam(value = "orgId") Long orgId,
                                                                       @RequestParam(value = "employeeName", required = false) String employeeName,
                                                                       @RequestParam(value = "employeeNo", required = false) String employeeNo,
                                                                       @RequestParam(value = "phoneNum", required = false) String phoneNum,
                                                                       @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                       @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询当前组织关联的员工列表
     *
     * @param orgId        组织id
     * @param employeeName 员工名称
     * @param employeeNo   员工编号
     * @param phoneNum     手机号
     * @param pageNum      当前页
     * @param pageSize     每页数据量
     * @return 员工列表
     */
    @RequestMapping(value = "org/{orgId}/employee-list", method = RequestMethod.GET)
    @ApiOperation(value = "查询当前组织关联的员工列表", notes = "查询当前组织关联的员工列表")
    @Capability(capabilityCode = "user.organization.query-relate-employees-by-org-id")
    RestResponse<PageInfo<EmployeeDto>> queryRelateEmployeesByOrgId(@PathVariable("orgId") Long orgId,
                                                                    @RequestParam(value = "employeeName", required = false) String employeeName,
                                                                    @RequestParam(value = "employeeNo", required = false) String employeeNo,
                                                                    @RequestParam(value = "phoneNum", required = false) String phoneNum,
                                                                    @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                    @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询未关联员工的用户列表
     *
     * @param userName 用户名
     * @param pageNum  当前页
     * @param pageSize 每页数据量
     * @return 用户列表
     */
    @RequestMapping(value = "user/list", method = RequestMethod.GET)
    @ApiOperation(value = "查询未关联员工的用户列表", notes = "查询未关联员工的用户列表")
    @Capability(capabilityCode = "user.organization.query-not-relate-employee-users")
    RestResponse<PageInfo<UserRespDto>> queryNotRelateEmployeeUsers(@RequestParam(value = "userName", required = false) String userName,
                                                                    @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                    @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询用户已关联的组织
     *
     * @param userId 用户id
     * @return 组织信息
     */
    @GetMapping("user/{userId}/relate")
    @ApiOperation(value = "查询用户已关联的组织", notes = "查询用户已关联的组织")
    @Capability(capabilityCode = "user.organization.query-user-org-relation")
    RestResponse<PageInfo<OrganizationDto>> queryUserOrgRelation(@PathVariable("userId") Long userId,
                                                                 @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                                 @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询用户所在（通过员工-组织关联）的组织列表
     *
     * @return 用户关联的组织列表
     */
    @ApiOperation(value = "查询用户所在（通过员工-组织关联）的组织列表", notes = "查询用户所在（通过员工-组织关联）的组织列表")
    @GetMapping("user/org-list")
    @Capability(capabilityCode = "user.organization.query-user-org-list")
    RestResponse<List<OrganizationDto>> queryUserOrgList();

    /**
     * 查询组织已关联的用户
     *
     * @param userOrgRelationQueryReqDto 查询请求dto
     * @param pageNum                    当前页
     * @param pageSize                   每页数据量
     * @return 用户列表
     */
    @GetMapping("user/relate")
    @ApiOperation(value = "查询组织已关联的用户", notes = "查询组织已关联的用户")
    @Capability(capabilityCode = "user.organization.query-user-org-relation")
    RestResponse<PageInfo<UserRespDto>> queryUserOrgRelation(@SpringQueryMap UserOrgRelationQueryReqDto userOrgRelationQueryReqDto,
                                                             @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                             @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 查询组织未关联的用户
     *
     * @param userOrgRelationQueryReqDto 查询请求dto
     * @param pageNum                    当前页
     * @param pageSize                   每页数据量
     * @return 用户列表
     */
    @GetMapping("user/no-relate")
    @ApiOperation(value = "查询组织未关联的用户", notes = "查询组织未关联的用户")
    @Capability(capabilityCode = "user.organization.query-user-org-no-relation")
    RestResponse<PageInfo<UserRespDto>> queryUserOrgNoRelation(@SpringQueryMap UserOrgRelationQueryReqDto userOrgRelationQueryReqDto,
                                                               @RequestParam(value = "pageNum", defaultValue = "1") Integer pageNum,
                                                               @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize);

    /**
     * 获取当前登录用户可查看的组织节点(仅针对登录用户)
     *
     * @return 组织树
     */
    @GetMapping("/by-cur-user")
    @ApiOperation(value = "获取当前登录用户可查看的组织节点(仅针对登录用户)，该接口需与数据权限函数一起使用，默认返回所有组织列表")
    @Capability(capabilityCode = "user.organization.query-org-by-curUser")
    RestResponse<List<OrganizationDto>> queryOrgByCurUser(@SpringQueryMap UserOrgQueryReqDto reqDto);

    /**
     * 获取当前组织的下级组织列表，包括当前组织
     *
     * @param orgId 组织id
     * @return 组织列表
     */
    @GetMapping("/all/subordinate/{orgId}")
    @ApiOperation(value = "获取当前组织的下级组织列表，包括当前组织")
    @Capability(capabilityCode = "user.organization.query-Subordinate-org-by-parentId")
    RestResponse<List<OrganizationDto>> querySubordinateOrgByParentId(@PathVariable("orgId") Long orgId);

    /**
     * 查询用户直属上级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryDirectSuperiorOrgIds", desc = "用户中心：用户直属上级组织")
    @GetMapping("/direct/superior")
    @ApiOperation(value = "查询用户直属上级组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-superior-orgids")
    RestResponse<List<Long>> queryDirectSuperiorOrgIds();

    /**
     * 查询用户所有上级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryAllSuperiorOrgIdList", desc = "用户中心：用户所有上级组织")
    @GetMapping("/all/superior")
    @ApiOperation(value = "用户所有上级组织id列表")
    @Capability(capabilityCode = "user.organization.query-all-superior-orgIdList")
    RestResponse<List<Long>> queryAllSuperiorOrgIdList();

    /**
     * 查询用户直属下级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryDirectSubordinateOrgIds", desc = "用户中心：用户直属下级组织")
    @GetMapping("/direct/subordinate")
    @ApiOperation(value = "查询用户直属下级组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgIds")
    RestResponse<List<Long>> queryDirectSubordinateOrgIds();

    /**
     * 查询用户全部下级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryAllSubordinateOrgIds", desc = "用户中心：用户全部下级组织")
    @GetMapping("/all/subordinate")
    @ApiOperation(value = "查询用户全部下级组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgIds")
    RestResponse<List<Long>> queryAllSubordinateOrgIds();

    /**
     * 查询用户兄弟组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryBrotherOrgIds", desc = "用户中心：用户兄弟组织")
    @GetMapping("/brother")
    @ApiOperation(value = "查询用户兄弟组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgIds")
    RestResponse<List<Long>> queryBrotherOrgIds();

    /**
     * 查询用户兄弟组织的直属下级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryBrotherDirectSubordinateOrgIds",
            desc = "用户中心：用户兄弟组织的直属下级组织")
    @GetMapping("/brother/direct/subordinate")
    @ApiOperation(value = "查询用户兄弟组织的直属下级组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgIds")
    RestResponse<List<Long>> queryBrotherDirectSubordinateOrgIds();

    /**
     * 查询用户兄弟组织的所有下级组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryBrotherAllSubordinateOrgIds",
            desc = "用户中心：用户兄弟组织的所有下级组织")
    @GetMapping("/brother/all/subordinate")
    @ApiOperation(value = "查询用户兄弟组织的所有下级组织id列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgIds")
    RestResponse<List<Long>> queryBrotherAllSubordinateOrgIds();

    /**
     * 查询用户所在组织id列表
     *
     * @return 组织id列表
     */
    @DataLimitCalcFunc(beanName = "", funcName = "queryCurrentUserOrgIds", desc = "用户中心：用户所在组织")
    @GetMapping("/current/org")
    @ApiOperation(value = "查询用户所在组织Id")
    @Capability(capabilityCode = "user.organization.query-current-user-orgIds")
    RestResponse<List<Long>> queryCurrentUserOrgIds();

    /**
     * 查询用户所属组织列表
     *
     * @param userId 用户id
     * @return 查询用户所属组织列表
     */
    @GetMapping("/{userId}/org")
    @ApiOperation(value = "查询用户所属组织列表")
    @Capability(capabilityCode = "user.organization.query-belong-org-by-user-id")
    RestResponse<List<OrganizationDto>> queryBelongOrgByUserId(@PathVariable("userId") Long userId);

    /**
     * 查询用户直属上级组织列表
     *
     * @return 用户直属上级组织列表
     */
    @GetMapping("/direct/superior-list")
    @ApiOperation(value = "查询用户直属上级组织列表")
    @Capability(capabilityCode = "user.organization.query-direct-superior-orgIds")
    RestResponse<List<OrganizationDto>> queryDirectSuperiorOrgList();

    /**
     * 查询用户直属下级组织列表
     *
     * @return 用户直属下级组织列表
     */
    @GetMapping("/direct/subordinate-list")
    @ApiOperation(value = "查询用户直属下级组织列表")
    @Capability(capabilityCode = "user.organization.query-direct-subordinate-orgList")
    RestResponse<List<OrganizationDto>> queryDirectSubordinateOrgList();

    /**
     * 查询用户全部下级组织列表
     *
     * @return 用户全部下级组织列表
     */
    @GetMapping("/all/subordinate-list")
    @ApiOperation(value = "查询用户全部下级组织列表")
    @Capability(capabilityCode = "user.organization.query-all-subordinate-orgList")
    RestResponse<List<OrganizationDto>> queryAllSubordinateOrgList();

    /**
     * 查询用户全部下级组织，并以树状结构返回
     *
     * @return 查询用户全部下级组织
     */
    @ApiOperation(value = "查询用户全部下级组织，并以树状结构返回")
    @GetMapping("/subordinate/all/tree")
    @Capability(capabilityCode = "user.organization.query-all-subordinate-orgTree")
    RestResponse<List<OrganizationDto>> queryAllSubordinateOrgTree();

    /**
     * 查询用户兄弟组织列表
     *
     * @return 用户兄弟组织列表
     */
    @GetMapping("/brother-list")
    @ApiOperation(value = "查询用户兄弟组织列表")
    @Capability(capabilityCode = "user.organization.query-brother-subordinate-orgList")
    RestResponse<List<OrganizationDto>> queryBrotherOrgList();

    /**
     * 根据应用实例获取整个组织树信息 通过实例获取组织信息
     *
     * @param instanceId 应用实例ID
     * @return OrganizationDto
     */
    @GetMapping("/queryOneByOrgCode")
    @ApiOperation(value = "查询指定应用实例下的组织机构", notes = "请求参数为： Application-Key 应用实例id orgCode为组织编码")
    RestResponse<OrganizationDto> queryOneByOrgCode(@RequestParam(value = "instanceId",required = false) Long instanceId, @RequestParam("orgCode") String orgCode);

    /**
     * 根据应用实例获取整个组织树信息 通过实例获取组织信息
     *
     * @param instanceId 应用实例ID
     * @return OrganizationDto
     */
    @PostMapping("/queryListByOrgCode")
    @ApiOperation(value = "查询指定应用实例下的组织机构", notes = "请求参数为： Application-Key 应用实例id orgCodeList为组织编码集合")
    RestResponse<List<OrganizationDto>> queryListByOrgCode(@RequestParam(value = "instanceId",required = false) Long instanceId, @RequestParam("orgCodeList") List<String> orgCodeList);

    /**
     * 根据组织id查询组织及企业详情
     *
     * @param orgIds 组织id
     * @return 组织及企业详情
     */
    @PostMapping(value = "/org-Info/byIds")
    @ApiOperation(value = "根据组织id集合查询组织及企业详情", notes = "根据组织id集合查询组织及企业详情")
    @Capability(capabilityCode = "user.organization.query-org-and-org-info-byIds")
    RestResponse<List<OrgAndOrgInfoRespDto>> queryOrgAndOrgInfoByIds(@RequestBody List<Long> orgIds);



}
