/**
 * @(#)IShopApi.java 1.0 2018/9/3
 * <p>
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.dtyunxi.yundt.cube.center.shop.api;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.vo.BaseVo;
import com.dtyunxi.yundt.cube.center.shop.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.shop.api.fallback.ShopFallBack;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.List;

/**
 * 操作店铺Api接口
 *
 * @author 风信
 * @since 1.0.0
 */
@FeignClient(contextId = "src-main-com-dtyunxi-yundt-cube-center-shop-api-IShopApi", name = "${yundt.cube.center.shop.api.name:yundt-cube-center-shop}", path = "/v2/shop", url = "${yundt.cube.center.shop.api:}"
,fallback = ShopFallBack.class)
public interface IShopApi {

    /**
     * 新增店铺（开通店铺）
     *
     * @param shopDto   开通店铺信息（包括运营范围、经营区域、仓库信息）
     * @param needAudit 是否需要审核
     * @return 店铺ID
     */
    @PostMapping(value = "", produces = "application/json")
    @ApiOperation(value = "新增店铺", notes = "店铺名称和店铺编码，在数据库租户+实例下进行唯一性校验\n"
            + "店铺经营区域、经营范围、仓库、地址，选填")
    RestResponse<Long> addShop(@Valid @RequestBody ShopDto shopDto,
                               @RequestParam("audit") boolean needAudit);

    /**
     * 批量新增店铺
     *
     * @param shopDtos  店铺信息
     * @param needAudit 是否需要审核
     * @return void
     */
    @PostMapping(value = "/batch")
    @ApiOperation(value = "批量新增店铺", notes = "底层调用IShopApi#addShop()，要么全部成功，要么全部失败")
    RestResponse<Void> addShops(@Valid @RequestBody List<ShopReqDto> shopDtos,
                                @RequestParam("audit") boolean needAudit);


    /**
     * 根据主键id修改店铺
     *
     * @param shopDto   店铺信息（包括运营范围、经营区域、仓库信息）
     * @param needAudit 是否需要审核
     * @return void
     */
    @PutMapping(value = "", produces = "application/json")
    @ApiOperation(value = "根据主键id修改店铺", notes = "店铺名称和编码修改时，数据库租户+实例下进行唯一性校验\n"
            + "更新店铺基本信息、经营范围、经营区域、仓库、地址")
    RestResponse<Void> modifyShop(@RequestBody ShopDto shopDto,
                                  @RequestParam("audit") boolean needAudit);

    /**
     * 店铺信息编辑（只是单纯编辑店铺信息）
     *
     * @param id
     * @param shopReqDto
     * @return
     */
    @PutMapping(value = "/modifyShopOnly/{id}", produces = "application/json")
    @ApiOperation(value = "店铺信息编辑（只是单纯编辑店铺信息）", notes = "店铺信息编辑（只是单纯编辑店铺信息）")
    RestResponse<Void> modifyShopOnly(@PathVariable("id") Long id, @RequestBody ShopReqDto shopReqDto);

    /**
     * 根据主键id删除店铺（逻辑删除）
     *
     * @param id 主键ID
     * @return void
     */
    @DeleteMapping(value = "/{id}", produces = "application/json")
    @ApiOperation(value = "根据主键id删除店铺（逻辑删除）", notes = "删除店铺基本信息以及关联的经营范围、经营区域、仓库、地址")
    RestResponse<Void> removeShop(@PathVariable("id") Long id);

    /**
     * 新增店铺经营区域
     *
     * @param shopAreaDto 店铺区域信息Dto
     * @return 经营区域ID
     */
    @PostMapping(value = "/area", produces = "application/json")
    @ApiOperation(value = "新增店铺经营区域", notes = "新增店铺经营区域")
    RestResponse<Long> addShopArea(@Valid @RequestBody ShopAreaDto shopAreaDto);

    /**
     * 根据店铺区域id修改店铺经营区域
     *
     * @param shopAreaDto 店铺区域信息Dto
     * @return void
     */
    @PutMapping(value = "/area", produces = "application/json")
    @ApiOperation(value = "根据店铺区域id修改店铺经营区域", notes = "根据店铺区域id修改店铺经营区域")
    RestResponse<Void> modifyShopArea(@Valid @RequestBody ShopAreaDto shopAreaDto);

    /**
     * 根据店铺区域id删除店铺的经营区域
     *
     * @param areaId 店铺区域ID
     * @return void
     */
    @DeleteMapping(value = "/area/{areaId}", produces = "application/json")
    @ApiOperation(value = "根据店铺区域id删除店铺的经营区域", notes = "根据店铺区域id删除店铺的经营区域")
    RestResponse<Void> removeShopAreaById(@PathVariable("areaId") Long areaId);

    /**
     * 增加店铺运营范围
     *
     * @param shopBusinessScopeDto 店铺运营范围信息
     * @return 运营范围ID
     */
    @PostMapping(value = "/business-scope", produces = "application/json")
    @ApiOperation(value = "增加店铺运营范围", notes = "增加店铺运营范围")
    RestResponse<Long> addShopBusinessScope(@Valid @RequestBody ShopBusinessScopeDto shopBusinessScopeDto);

    /**
     * 修改店铺运营范围
     *
     * @param shopBusinessScopeDto 店铺运营范围信息
     * @return void
     */
    @PutMapping(value = "/business-scope", produces = "application/json")
    @ApiOperation(value = "修改店铺运营范围", notes = "修改店铺运营范围")
    RestResponse<Void> modifyShopBusinessScope(@Valid @RequestBody ShopBusinessScopeDto shopBusinessScopeDto);

    /**
     * 删除店铺业务
     *
     * @param shopId     店铺ID
     * @param businessId 业务ID
     * @return void
     */
    @DeleteMapping(value = "/{shopId}/business-scope/{businessId}", produces = "application/json")
    @ApiOperation(value = "删除店铺业务", notes = "删除店铺业务")
    RestResponse<Void> removeShopBusinessScopeById(@PathVariable("shopId") Long shopId,
                                                   @PathVariable("businessId") Long businessId);

    /**
     * 绑定店铺对应仓库
     *
     * @param shopWarehouseDto 仓库信息
     * @return 仓库ID
     */
    @PostMapping(value = "/warehouse", produces = "application/json")
    @ApiOperation(value = "绑定店铺对应仓库", notes = "绑定店铺对应仓库")
    RestResponse<Long> addWarehouse(@Valid @RequestBody ShopWarehouseDto shopWarehouseDto);

    /**
     * 解绑店铺对应仓库
     *
     * @param shopId      店铺ID
     * @param warehouseId 仓库ID
     * @return void
     */
    @DeleteMapping(value = "/{shopId}/warehouse/{warehouseId}", produces = "application/json")
    @ApiOperation(value = "解绑店铺对应仓库", notes = "解绑店铺对应仓库")
    RestResponse<Void> removeWarehouse(@PathVariable("shopId") Long shopId,
                                       @PathVariable("warehouseId") Long warehouseId);

    /**
     * 定义店铺自有的会员等级
     *
     * @param shopId 归属店铺
     * @param dto    等级定义内容
     * @return 操作是否成功
     */
    @PutMapping("{shopId}/member/level")
    @ApiOperation(value = "定义店铺自有的会员等级", notes = "1、以shop_member${shopId}作为会员体系名称，调用会员中心创建会员体系\n"
            + "2、调用会员中心，创建会员等级定义")
    RestResponse<Boolean> defineShopMemberLevel(@PathVariable("shopId") long shopId,
                                                @RequestBody List<ShopMemberLevelDefineReqDto> dto);

    /**
     * 成为店铺会员
     *
     * @param shopId 店铺ID
     * @param dto    保存对象实体 必填项实例ID，用户ID，用户名称。选填项 创建时间，激活时间
     * @return 操作是否成功
     */
    @PutMapping("{shopId}/member")
    @ApiOperation(value = "成为店铺会员", notes = "1、如果会员体系不存在，先创建会员体系\n"
            + "2、如果参数没有提供会员id，先创建会员；然后绑定用户ID和会员")
    RestResponse<Boolean> addShopMember(@PathVariable("shopId") long shopId, @RequestBody ShopMemberReqDto dto);

    /**
     * 注销店铺会员
     *
     * @param shopId   店铺ID
     * @param memberId 用户ID
     * @return 操作是否成功
     */
    @DeleteMapping("{shopId}/member/{memberId}")
    @ApiOperation(value = "注销店铺会员", notes = "注销店铺会员")
    RestResponse<Boolean> delShopMember(@PathVariable("shopId") long shopId, @PathVariable("memberId") long memberId);

    /**
     * 绑定店铺用户关联关系
     *
     * @param addShopUserReqDto 新增店铺用户关联关系请求DTO
     * @return 店铺用户关联表的主键ID
     */
    @PostMapping("/user")
    @ApiOperation(value = "绑定店铺用户关联关系", notes = "如果之前已关联，则返回：6000 店铺用户已经存在关联")
    RestResponse<Long> addShopUser(@RequestBody AddShopUserReqDto addShopUserReqDto);

    /**
     * 根据所传的待审核信息审核店铺
     *
     * @param status      审核状态(PENDING：待审核；PASS：审核通过；REFUSE：审核拒绝)
     * @param result      审核结果
     * @param auditPerson 审核人
     * @param tenantId    租户ID
     * @param auditId     待审核信息的ID
     * @return void
     */
    @PostMapping(value = "/audit/{auditId}", produces = "application/json")
    @ApiOperation(value = "根据所传待审核信息审核店铺", notes = "1、审核通过，更新审核表状态、店铺状态、地址、经营区域、经营范围、仓库\n"
            + "2、待审核或者审核拒绝，只更新审核表的状态")
    RestResponse<Void> auditShop(
            @PathVariable("auditId") @Min(1) long auditId,
            @RequestParam("tenantId") @Min(1) long tenantId,
            @RequestParam("status") String status,
            @RequestParam("result") String result,
            @RequestParam("auditPerson") String auditPerson
    );

    /**
     * 定义平台级店铺等级
     *
     * @param tenantId   租户ID
     * @param instanceId 应用ID
     * @param dto        等级定义内容
     * @return void
     */
    @PostMapping("/level")
    @ApiOperation(value = "定义平台级店铺等级", notes = "1、以tenant_seller作为会员体系名称，调用会员中心创建会员体系\n"
            + "2、调用会员中心，创建会员等级定义")
    RestResponse<Boolean> defineShopLevel(@RequestParam(value = "tenantId") @Min(1) long tenantId,
                                          @RequestParam("instanceId") @Min(1) long instanceId,
                                          @RequestBody @Valid List<ShopMemberLevelDefineReqDto> dto);

    /**
     * 检查地址是否在配送范围内
     *
     * @param locationX 经度
     * @param locationY 纬度
     * @param shopIds   店铺id,多个以逗号隔开
     * @return RestResponse<java.lang.Integer>
     */
    @GetMapping("/isCheckAddress")
    @ApiOperation("检查地址是否在配送范围内(返回 1,表示是o20且不在配送范围内，2表示是o20且在配送范围内)")
    RestResponse<Integer> isCheckAddress(
            @RequestParam("locationX") String locationX,
            @RequestParam("locationY") String locationY,
            @RequestParam("shopIds") String shopIds
    );

    /**
     * 根据坐标按照距离由近到远排序查询es店铺列表
     *
     * @param lat 纬度
     * @param lon 经度
     * @return RestResponse<shopEsDto>
     */
    @GetMapping("/queryRangeShopListbyGeo")
    @ApiOperation(value = "根据坐标按照距离由近到远排序查询es店铺列表", notes = "isValidRegion校验是否在配送范围内，默认false")
    RestResponse<List<ShopEsDto>> queryRangeShopListByGeo(@RequestParam("lat") String lat,
                                                          @RequestParam("lon") String lon,
                                                          @RequestParam(value = "isValidRegion", required = false) Boolean isValidRegion);

    /**
     * 校验店铺名称是否存在
     *
     * @return void
     */
    @GetMapping("/checkShopName")
    @ApiOperation("校验店铺名称是否存在")
    RestResponse<Boolean> checkShopName(@RequestParam("shopName") String shopName);

    /**
     * 校验店铺编码是否存在
     *
     * @return void
     */
    @GetMapping("/checkShopCode")
    @ApiOperation("校验店铺编码是否存在")
    RestResponse<Boolean> checkShopCode(@RequestParam("shopCode") String shopCode);

    @PutMapping(value = "/modifyShopById/{id}", produces = "application/json")
    @ApiOperation(value = "店铺信息编辑（修改身份证信息等）", notes = "店铺信息编辑（修改身份证信息等）")
    RestResponse<Void> modifyShopById(@PathVariable("id") Long id, @RequestBody ShopReqDto shopReqDto);

}
