/*
 * @(#)ISellerQueryApi.java 1.0 2018/9/5
 *
 * Copyright (c) 2018, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.shop.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.shop.api.dto.request.SellerAreaDto;
import com.dtyunxi.yundt.cube.center.shop.api.dto.request.SellerQueryReqDto;
import com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerBusinessScopeRespDto;
import com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto;
import com.dtyunxi.yundt.cube.center.shop.api.fallback.SellerFallBack;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 商户接口查询Api
 *
 * @author 华生
 * @since 1.0.0
 */

@FeignClient(contextId="src-main-com-dtyunxi-yundt-cube-center-shop-api-query-ISellerQueryApi", name = "${yundt.cube.center.shop.api.name:yundt-cube-center-shop}", path = "/v2/seller", url = "${yundt.cube.center.shop.api:}",fallback = SellerFallBack.class)
public interface ISellerQueryApi {


    /**
     * 根据商户ID，业务ID，业务类型，查询商户经营范围信息
     *
     * @param sellerId   商户ID
     * @param businessId 业务Id
     * @return {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerBusinessScopeRespDto}
     */
    @GetMapping(value = {"/query/{sellerId}/scope/{businessId}"}, produces = "application/json")
    @ApiOperation(value = "查询商户经营范围信息", notes = "查询商户经营范围信息 \t\n sellerId:商户ID \t\n businessId:业务ID")
    RestResponse<SellerBusinessScopeRespDto> queryBusinessScopeBySellerIdAndBusinessId(
            @PathVariable("sellerId") Long sellerId, @PathVariable("businessId") Long businessId);

    /**
     * 根据商户ID查询商户信息
     *
     * @param id 商户ID
     * @return {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}
     */
    @GetMapping("/{id}")
    @ApiOperation(value = "查询商户信息", notes = "查询商户信息 \t\n id:商户ID \t\n 只返回商户相关信息")
    RestResponse<SellerRespDto> queryById(@NotNull(message = "商户ID不允许为空") @PathVariable("id") Long id);

    /**
     * 根据实例ID查询商户列表
     *
     * @param instanceId 实例ID
     * @return List&lt;{@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}&gt;
     */
    @GetMapping("/instance/{instanceId}")
    @ApiOperation(value = "根据instanceId查询商户列表信息", notes = "根据instanceId查询商户列表信息 \t\n instanceId:实例ID \t\n 只返回商户相关信息")
    RestResponse<List<SellerRespDto>> queryByInstanceId(@PathVariable("instanceId") Long instanceId);

    /**
     * 根据租户ID查询商户列表
     *
     * @param tenantId 租户ID
     * @return List&lt;{@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}&gt;
     */
    @GetMapping("/tenant/{tenantId}")
    @ApiOperation(value = "根据tenantId查询商户列表信息", notes = "根据tenantId查询商户列表信息 \t\n tenantId:租户ID \t\n 只返回商户相关信息")
    RestResponse<List<SellerRespDto>> queryByTenantId(@PathVariable("tenantId") Long tenantId);

    /**
     * 根据商户ID查询商户经营范围列表
     *
     * @param sellerId 商户ID
     * @return List&lt;{@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerBusinessScopeRespDto}&gt;
     */
    @GetMapping("/{sellerId}/business-scope")
    @ApiOperation(value = "根据商户ID查询商户经营范围列表", notes = "根据商户ID查询商户经营范围列表 \t\n sellerId:商户ID")
    RestResponse<List<SellerBusinessScopeRespDto>> queryBySellerId(@PathVariable("sellerId") Long sellerId);

    /**
     * 根据条件分页查询商户基本信息列表
     *
     * @param sellerQueryReqDto 查询条件DTO
     * @return 带分页信息的列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}
     */
    @GetMapping("/page")
    @ApiOperation(value = "根据条件分页查询商户基本信息列表", notes = "根据条件分页查询商户基本信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<PageInfo<SellerRespDto>> queryPageSeller(@ModelAttribute SellerQueryReqDto sellerQueryReqDto);


    /**
     * 根据条件（经营范围、商户名称）分页查询商户基本信息列表
     *
     * @param sellerQueryReqDto 查询条件DTO
     * @return 带分页信息的列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}
     */
    @GetMapping("/page/byScope")
    @ApiOperation(value = "根据条件（经营范围、商户名称）分页查询商户基本信息列表", notes = "根据条件分页查询商户基本信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<PageInfo<SellerRespDto>> queryPageSellerByBusinessScope(@ModelAttribute SellerQueryReqDto sellerQueryReqDto);

    /**
     * 根据条件查询商户信息列表
     *
     * @param sellerQueryReqDto 查询条件DTO
     * @return List&lt;{@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}&gt;
     */
    @GetMapping("/list")
    @ApiOperation(value = "根据条件查询商户信息列表", notes = "根据条件查询商户信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<List<SellerRespDto>> queryList(@ModelAttribute SellerQueryReqDto sellerQueryReqDto);

    @GetMapping("/queryByCode")
    @ApiOperation(value = "根据条件查询商户信息列表", notes = "根据条件查询商户信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<List<SellerRespDto>> queryByCode(@RequestParam(name = "code") String code);

    @PostMapping("/queryListByFeign")
    @ApiOperation(value = "根据条件查询商户信息列表", notes = "根据条件查询商户信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<List<SellerRespDto>> queryListByFeign(@RequestBody SellerQueryReqDto sellerQueryReqDto);

    /**
     * 根据商户id查询经营区域
     *
     * @author wu.sheng
     * @date 2020/2/17 14:49
     * @return
     */
    @GetMapping("/{sellerId}/area")
    @ApiOperation(value = "根据商户ID查询商户经营区域列表", notes = "根据商户ID查询商户经营范围列表 \t\n sellerId:商户ID")
    RestResponse<List<SellerAreaDto>> queryAreaBySellerId(@PathVariable("sellerId") Long sellerId);


    /**
     * 根据条件搜索商户经营区域
     *
     * @author wu.sheng
     * @date 2020/2/18 14:18
     * @return
     */
    @GetMapping("/area/list")
    @ApiOperation(value = "根据查询商户经营区域列表")
    RestResponse<List<SellerAreaDto>> queryAreaList(@ModelAttribute SellerAreaDto sellerAreaDto);

    /**
     * 根据条件查询商户信息列表
     *
     * @param sellerQueryReqDto 查询条件DTO
     * @return List&lt;{@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto}&gt;
     */

    @PostMapping("/list")
    @ApiOperation(value = "根据条件查询商户信息列表", notes = "根据条件查询商户信息列表 \t\n sellerQueryReqDto:查询条件DTO")
    RestResponse<List<SellerRespDto>> queryListOnPost(@RequestBody SellerQueryReqDto sellerQueryReqDto);
}
