/**
 * @(#)IShopQueryApi.java 2.0.0 2018-09-04
 * Copyright (c) 2017, YUNXI. All rights reserved.
 * YUNXI PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package com.dtyunxi.yundt.cube.center.shop.api.query;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.shop.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.shop.api.dto.response.FreightCountRespDto;
import com.dtyunxi.yundt.cube.center.shop.api.dto.response.ShopBaseDto;
import com.dtyunxi.yundt.cube.center.shop.api.fallback.ShopFallBack;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.ApiOperation;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Set;

/**
 * 店铺查询接口
 *
 * @author luo.lei
 * @since 2.0.0
 */
@FeignClient(contextId = "src-main-com-dtyunxi-yundt-cube-center-shop-api-query-IShopQueryApi", name = "${yundt.cube.center.shop.api.name:yundt-cube-center-shop}", path = "/v2/shop", url = "${yundt.cube.center.shop.api:}", fallback = ShopFallBack.class)
public interface IShopQueryApi {
    /**
     * 简易查询，查询商户下所有店铺
     *
     * @param sellerId 商户ID
     * @param pageSize 分页参数-页大小 不得小于1
     * @param page     分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     * 3.18.1版本该接口已过期。{@linkplain IShopQueryApi#queryBaseShopPage}
     */
    @GetMapping("/seller/{sellerId}")
    @Deprecated
    RestResponse<PageInfo<ShopDto>> queryBySellerId(@PathVariable("sellerId") long sellerId,
                                                    @RequestParam(name = "pageSize", defaultValue = "10") int pageSize,
                                                    @RequestParam(name = "page", defaultValue = "1") int page);

    /**
     * 根据店铺id集合和地址区域查询对应的店铺经营范围集合
     * @param reqDto
     * @return
     */
    @PostMapping("/area/queryShopAreaByShopIds")
    RestResponse<List<ShopAreaDto>> queryShopAreaByShopIds(@RequestBody AddressAreaReqDto reqDto);

    /**
     * 简易查询，查询租户下所有店铺
     *
     * @param tenantId 租户ID
     * @param pageSize 分页参数-页大小 不得小于1
     * @param page     分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     * 3.18.1版本该接口已过期。{@linkplain IShopQueryApi#queryBaseShopPage}
     */
    @GetMapping("/tenant")
    @Deprecated
    RestResponse<PageInfo<ShopDto>> queryByTenantId(@RequestParam("tenantId") long tenantId,
                                                    @RequestParam(name = "pageSize", defaultValue = "10") int pageSize,
                                                    @RequestParam(name = "page", defaultValue = "1") int page);


    /**
     * 根据查询条件获取店铺列表
     *
     * @param queryCondition 查询条件
     * @param pageSize       分页参数-页大小 不得小于1
     * @param page           分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     */
    @GetMapping("/conditions")
    RestResponse<PageInfo<ShopDto>> queryByConditions(@ModelAttribute ShopQueryDto queryCondition,
                                                      @RequestParam(name = "pageSize", defaultValue = "10") int pageSize,
                                                      @RequestParam(name = "page", defaultValue = "1") int page);

    /**
     * 根据ID查询店铺信息
     *
     * @param shopId 店铺ID
     * @param enums  返回的附加属性 默认只返回基础属性部分，可选参数 {@link com.dtyunxi.yundt.cube.center.shop.api.enums.ShopQueryEnum}
     * @return 店铺信息 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     */
    @GetMapping("/{shopId}")
    RestResponse<ShopDto> queryById(@PathVariable("shopId") long shopId, @RequestParam(value = "enums", required = false) String... enums);

    /**
     * 根据code查询店铺信息
     *
     * @param code 店铺code
     * @return 店铺信息 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     */
    @GetMapping("/code/{code}")
    RestResponse<ShopDto> queryByCode(@PathVariable("code") String code);
    /**
     * 查询店铺基本信息
     *
     * @param shopId
     * @return
     */
    @GetMapping("/base/{shopId}")
    RestResponse<ShopBaseDto> queryBaseById(@PathVariable("shopId") Long shopId);
    /**
     * 查询店铺基本信息(不走缓存)
     *
     * @param shopId
     * @return
     */
    @GetMapping("/base/das/{shopId}")
    RestResponse<ShopBaseDto> queryBaseDasById(@PathVariable("shopId") Long shopId);


    /**
     * 批量查询店铺基本信息
     *
     * @param shopIds
     * @return
     */
    @GetMapping("/base/ids")
    RestResponse<List<ShopBaseDto>> queryBaseListByIds(@NotNull @RequestParam("shopIds") Set<Long> shopIds);


    /**
     * 根据ID列表查询店铺列表信息
     *
     * @param shopIds 店铺ID列表
     * @return 店铺详情列表
     */
    @GetMapping("/ids")
    RestResponse<List<ShopDto>> queryByIds(@NotNull @RequestParam("shopIds") Set<Long> shopIds);

    /**
     * 查询店铺服务区域
     *
     * @return
     * @author wu.sheng
     * @date 2020/2/20 14:18
     */
    @GetMapping("/area/list")
    RestResponse<List<ShopAreaDto>> queryShopAreas(@ModelAttribute ShopAreaDto shopAreaDto);

    /**
     * 根据查询条件查询门店列表
     *
     * @param queryCondition 查询条件
     * @param pageSize       分页参数-页大小 不得小于1
     * @param page           分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表 {@linkplain com.dtyunxi.yundt.cube.center.shop.api.dto.request.ShopDto}
     */
    @GetMapping("/storeShopList")
    RestResponse<PageInfo<ShopDto>> queryStoreShopList(@ModelAttribute ShopQueryDto queryCondition,
                                                       @RequestParam(name = "pageSize", defaultValue = "10") int pageSize,
                                                       @RequestParam(name = "page", defaultValue = "1") int page);

    /**
     * 计算费用
     *
     * @param addressId 地址id
     * @param shopId    店铺id
     * @return
     */
    @GetMapping("/freight/count")
    RestResponse<FreightCountRespDto> countFreight(@RequestParam(name = "addressId") Long addressId, @RequestParam(name = "shopId") Long shopId);


    /**
     * 查询店铺关联的发货仓库信息：type: 1 线上业务(快递区域), 2 O2O业务(同城配送的电子围栏)
     *
     * @param shopId
     * @return
     */
    @GetMapping("/{shopId}/warehouse")
    RestResponse<List<ShopWarehouseDto>> queryShopWarehouseByShopId(@PathVariable("shopId") Long shopId, @RequestParam(value = "type", required = false) Integer type);

    /**
     * 根据条件搜索店铺列表
     *
     * @param queryCondition 请求参数
     * @return 店铺列表
     **/
    @GetMapping("/list")
    @ApiOperation(value = "根据条件搜索店铺列表")
    RestResponse<List<ShopDto>> queryShopList(@ModelAttribute ShopQueryDto queryCondition);

    /**
     * 根据条件搜索店铺列表
     *
     * @param queryCondition 请求参数
     * @return 店铺列表
     **/
    @PostMapping("/listNew")
    @ApiOperation(value = "根据条件搜索店铺列表")
    RestResponse<List<ShopDto>> queryShopListNew(@RequestBody ShopQueryDto queryCondition);

    /**
     * 根据店铺编码查询店铺和店铺仓库信息
     *
     * @param dto 店铺编码
     * @return 店铺和店铺仓库信息
     */
    @PostMapping("/queryByCodes")
    RestResponse<List<ShopDto>> queryShopByCode(@RequestBody ShopQueryReqDto dto);

    /**
     * 查询店铺支持的配送方信息
     *
     * @param shopId
     * @return
     */
    @GetMapping("/{shopId}/partners")
    @ApiOperation(value = "查询店铺支持的配送方列表")
    RestResponse<List<ShopTransportPartnerDto>> queryShopTransportPartner(@PathVariable("shopId") Long shopId);

    @GetMapping("/freightTemplate/count/{freightTemplateId}")
    RestResponse<Integer> countByFreightTemplateId(@PathVariable("freightTemplateId") Long freightTemplateId);

    /**
     * 根据查询条件分页获取店铺基本信息列表
     *
     * @param shopQueryDto   查询条件
     * @param pageSize       分页参数-页大小 不得小于1
     * @param page           分页参数-当前页 不得小于1
     * @return 带分页的店铺信息，结果为空时返回空列表
     */
    @GetMapping("/queryBaseShopPage")
    @ApiOperation(value = "根据查询条件分页获取店铺基本信息列表")
    RestResponse<PageInfo<ShopBaseDto>> queryBaseShopPage(@ModelAttribute ShopQueryDto shopQueryDto,
                                            @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                            @RequestParam(name = "page", defaultValue = "1") Integer page);

    /**
     * 根据条件搜索店铺基本信息列表
     *
     * @param shopQueryDto 请求参数
     * @return 店铺列表
     **/
    @GetMapping("/queryBaseShopList")
    @ApiOperation(value = "根据条件搜索店铺基本信息列表")
    RestResponse<List<ShopBaseDto>> queryBaseShopList(@ModelAttribute ShopQueryDto shopQueryDto);

    /**
     * 根据查询条件获取店铺数量
     *
     * @param shopQueryDto   查询条件
     * @return 带分页的店铺信息，结果为空时返回空列表
     */
    @GetMapping("/countShop")
    @ApiOperation(value = "根据查询条件获取店铺数量")
    RestResponse<Long> countShop(@ModelAttribute ShopQueryDto shopQueryDto);

    /**
     * 查询最早创建并且启用的店铺
     */
    @GetMapping("/queryEarliestShop")
    RestResponse<ShopDto> queryEarliestShop();

    /**
     * 根据查询条件获取店铺列表
     * @param queryCondition 查询条件
     */
    @GetMapping("/shopList")
    RestResponse<List<ShopDto>> shopList(@ModelAttribute ShopQueryDto queryCondition);

    /**
     * 根据ID列表查询店铺列表信息
     *
     * @param shopIds 店铺ID列表
     * @return 店铺详情列表
     */
    @PostMapping("/ids")
    RestResponse<List<ShopDto>> queryByIdsOnPost(@NotNull @RequestBody Set<Long> shopIds);
}
