package com.dtyunxi.yundt.module.customer.biz.impl.user;

import com.dtyunxi.icommerce.utils.RestResponseHelper;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerAreaRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.query.ICustomerAreaQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.CustomerAreaListExtReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.EmployeeCustomerBlacklistReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.EmployeeCustomerReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.EmployeeRegionReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.query.IEmployeeCustomerQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.query.IEmployeeRegionQueryApi;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeRoleReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.tcbj.EmployeeExtQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.EmployeeExtRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.RoleExpandRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserOrgizationRespDto;
import com.dtyunxi.yundt.cube.center.user.api.enums.YesOrNoEnum;
import com.dtyunxi.yundt.cube.center.user.api.query.IAccessQueryApi;
import com.dtyunxi.yundt.cube.center.user.api.query.IEmployeeExpandQueryApi;
import com.dtyunxi.yundt.cube.center.user.api.query.IEmployeeRoleQueryApi;
import com.dtyunxi.yundt.cube.center.user.api.query.IUserOrgExpandQueryApi;
import com.dtyunxi.yundt.module.customer.api.dto.request.user.EmployeeExpandReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.user.SellerEmployeeImportDto;
import com.dtyunxi.yundt.module.customer.api.enums.EmpGovernTypeEnum;
import com.dtyunxi.yundt.module.customer.api.user.IEmployeeExpand2Service;
import com.dtyunxi.yundt.module.customer.api.user.ISellerEmployeeExtService;
import com.dtyunxi.yundt.module.customer.biz.util.CommonUtil;
import com.google.common.collect.Lists;
import com.yx.tcbj.center.customer.api.dto.request.CustomerExtReqDto;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * @Description 商家人员处理
 * @date 2022-01-14 14:06
 * @since 1.0.0
 */
@Service
public class SellerEmployeeExtServiceImpl implements ISellerEmployeeExtService {

    private static final Logger logger = LoggerFactory.getLogger(SellerEmployeeExtServiceImpl.class);
    @Resource
    private IAccessQueryApi accessQueryApi;
    @Resource
    private IEmployeeRegionQueryApi employeeRegionQueryApi;
    @Resource
    private ICustomerAreaQueryApi customerAreaQueryApi;

    @Resource
    private IEmployeeCustomerQueryApi employeeCustomerQueryApi;
    @Resource
    private IEmployeeExpand2Service employeeExpand2Service;
    @Resource
    private IUserOrgExpandQueryApi userOrgExpandQueryApi;
    @Resource
    private IEmployeeExpandQueryApi employeeExpandQueryApi;

    @Resource
    private IEmployeeRoleQueryApi employeeRoleQueryApi;

    /**
     * 商家人员导入-数据传输的时候已经校验必填字段
     *
     * @param importDataList 导入数据列表
     * @return 处理后的导入数据
     */
    @Override
    public RestResponse<List<SellerEmployeeImportDto>> doSellerEmployeeImport(List<SellerEmployeeImportDto> importDataList) {
        if (CollectionUtils.isEmpty(importDataList)) {
            logger.warn("导入数据为空，无法继续处理");
            return new RestResponse<>(importDataList);
        }

        // 识别操作类型 - 根据字段标记判断是新增还是修改
        // 1. 如果第一条记录的type字段有值，使用type字段判断
        // 2. 否则，如果工号字段非空，则认为是修改模式
        // 3. 否则，默认为新增模式
        boolean isModifyMode = false;
        if (importDataList.get(0) != null) {
            if (StringUtils.isNotEmpty(importDataList.get(0).getType())) {
                isModifyMode = "修改".equals(importDataList.get(0).getType());
            } else {
                isModifyMode = StringUtils.isNotEmpty(importDataList.get(0).getEmployeeCode());
            }
        }

        logger.info("商家人员导入数据开始处理，总记录数: {}, 模式: {}",
                importDataList.size(),
                isModifyMode ? "修改模式" : "新增模式");

        // 记录处理前的重复情况
        Map<String, List<SellerEmployeeImportDto>> duplicateMap = importDataList.stream()
                .filter(vo -> StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber()))
                .collect(Collectors.groupingBy(vo -> vo.getUserName() + "_" + vo.getPhoneNumber()));

        duplicateMap.entrySet().stream()
                .filter(entry -> entry.getValue().size() > 1)
                .forEach(entry -> {
                    List<SellerEmployeeImportDto> duplicates = entry.getValue();
                    logger.info("处理前发现重复记录: 姓名={}, 手机号={}, 记录数={}",
                            duplicates.get(0).getUserName(),
                            duplicates.get(0).getPhoneNumber(),
                            duplicates.size());
                });

        // 1. 获取账户信息
        List<String> accountList = importDataList.stream()
                .map(SellerEmployeeImportDto::getAccount)
                .filter(StringUtils::isNotEmpty)
                .distinct()
                .collect(Collectors.toList());
        Map<String, List<UserOrgizationRespDto>> userOrgMap = employeeExpand2Service.getAccoutMap(accountList);

        // 2. 获取工号信息 - 查询现有员工（适用于修改模式）
        Map<String, EmployeeExtRespDto> existingEmployeeMap = new HashMap<>();
        if (isModifyMode) {
            List<String> employeeCodes = importDataList.stream()
                    .filter(data -> StringUtils.isNotEmpty(data.getEmployeeCode()))
                    .map(SellerEmployeeImportDto::getEmployeeCode)
                    .distinct()
                    .collect(Collectors.toList());

            if (CollectionUtils.isNotEmpty(employeeCodes)) {
                List<EmployeeExtRespDto> existingEmployees = RestResponseHelper.extractData(
                        employeeRoleQueryApi.queryEmployeeByNos(employeeCodes));
                if (CollectionUtils.isNotEmpty(existingEmployees)) {
                    existingEmployeeMap = existingEmployees.stream()
                            .collect(Collectors.toMap(EmployeeExtRespDto::getEmployeeNo, Function.identity(), (k1, k2) -> k1));
                }
            }
        }

        // 3. 基本字段校验 - 统一的校验方法
        importDataList = this.verifyData(importDataList, userOrgMap, existingEmployeeMap, isModifyMode);

        List<SellerEmployeeImportDto> validList = importDataList.stream()
                .filter(dto -> StringUtils.isEmpty(dto.getErrorReason()))
                .collect(Collectors.toList());
        logger.info("基本字段校验后有效记录数: {}", validList.size());

        if (CollectionUtils.isEmpty(validList)) {
            logger.warn("所有记录基本字段校验未通过，无法继续处理");
            return new RestResponse<>(importDataList);
        }

        // 4. 获取角色数据
        List<String> roleNameList = validList.stream()
                .map(SellerEmployeeImportDto::getRoleName)
                .filter(StringUtils::isNotEmpty)
                .distinct()
                .collect(Collectors.toList());
        Map<String, List<RoleExpandRespDto>> roleMap = employeeExpand2Service.getRoleMap(roleNameList, null);

        // 5. 获取区域数据
        List<String> areaNameList = validList.stream()
                .flatMap(employee -> Stream.of(employee.getSaleArea(), employee.getBigArea(), employee.getDistrict()))
                .filter(StringUtils::isNotBlank)
                .distinct()
                .collect(Collectors.toList());
        Map<String, List<CustomerAreaRespDto>> customerAreasMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(areaNameList)) {
            CustomerAreaListExtReqDto areaListReqDto = new CustomerAreaListExtReqDto();
            areaListReqDto.setNameList(areaNameList);
            RestResponse<List<CustomerAreaRespDto>> areaRest = employeeRegionQueryApi.queryCustomerAreaByNames(areaListReqDto);
            if (null != areaRest && CollectionUtils.isNotEmpty(areaRest.getData())) {
                List<CustomerAreaRespDto> customerAreaList = areaRest.getData();
                customerAreasMap = customerAreaList.stream()
                        .collect(Collectors.groupingBy(CustomerAreaRespDto::getName));
            }
        }

        // 6. 获取客户数据
        List<String> customerCodeList = validList.stream()
                .map(SellerEmployeeImportDto::getGovernCustomer)
                .filter(StringUtils::isNotEmpty)
                .distinct()
                .collect(Collectors.toList());
        List<String> blackCustomerCodelist = validList.stream()
                .map(SellerEmployeeImportDto::getBlacklist)
                .filter(StringUtils::isNotEmpty)
                .distinct()
                .collect(Collectors.toList());
        List<String> allCustomerCodes = new ArrayList<>(customerCodeList);
        allCustomerCodes.addAll(blackCustomerCodelist);

        // 根据客户组织id和商家id获取客户信息
        Map<String, CustomerRespDto> customerMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(allCustomerCodes)) {
            Long orgId = validList.get(0).getOrgId();
            CustomerExtReqDto customerExtReqDto = new CustomerExtReqDto();
            customerExtReqDto.setMerchantIdList(Lists.newArrayList(orgId));
            customerExtReqDto.setCodes(allCustomerCodes);
            List<CustomerRespDto> customerRespDtos = RestResponseHelper.extractData(
                    employeeCustomerQueryApi.queryCustomerByDto(customerExtReqDto));
            if (CollectionUtils.isNotEmpty(customerRespDtos)) {
                customerMap = customerRespDtos.stream()
                        .collect(Collectors.toMap(CustomerRespDto::getCode, Function.identity(), (k1, k2) -> k1));
            }
        }

        // 7. 进一步校验（处理区域、客户等具体业务数据）
        importDataList = this.verifyOrgData(importDataList, roleMap, customerAreasMap, customerMap, userOrgMap);
        validList = importDataList.stream()
                .filter(dto -> StringUtils.isEmpty(dto.getErrorReason()))
                .collect(Collectors.toList());
        logger.info("组织字段校验后有效记录数: {}", validList.size());

        if (CollectionUtils.isEmpty(validList)) {
            logger.warn("所有记录组织字段校验未通过，无法继续处理");
            return new RestResponse<>(importDataList);
        }

        // 8. 合并数据（根据姓名+手机号）
        List<SellerEmployeeImportDto> mergedList = this.mergeOrgData(validList);
        logger.info("合并后的记录数: {}", mergedList.size());

        // 9. 获取现有的员工信息 - 根据用户ID和组织ID（适用于新增模式下查重）
        Map<String, Long> employeeInfoMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(mergedList)) {
            List<Long> userIdList = mergedList.stream()
                    .map(SellerEmployeeImportDto::getUserId)
                    .distinct()
                    .collect(Collectors.toList());
            List<Long> orgIdList = mergedList.stream()
                    .map(SellerEmployeeImportDto::getOrgId)
                    .distinct()
                    .collect(Collectors.toList());
            employeeInfoMap = employeeExpand2Service.getEmployeeMap(userIdList, orgIdList);
        }

        // 10. 处理数据（根据模式执行新增或修改）
        for (SellerEmployeeImportDto data : mergedList) {
            try {
                // 构建人员对象
            EmployeeExpandReqDto employeeExtReqDto = new EmployeeExpandReqDto();
            employeeExtReqDto.setOrganizationId(data.getOrgId());
            employeeExtReqDto.setOrgId(data.getOrgId());
            employeeExtReqDto.setOrganizationStatus(YesOrNoEnum.NO.getStatus());

                // 设置必填字段
                employeeExtReqDto.setName(data.getUserName()); // 姓名
                employeeExtReqDto.setPosition(data.getPosition()); // 职位
                employeeExtReqDto.setPhoneNum(data.getPhoneNumber()); // 手机号码
                employeeExtReqDto.setUserId(data.getUserId()); // 账号

                // 设置人员类型
            employeeExtReqDto.setPersonType("内部".equals(data.getEmployeeType()) ? "Internal" : "External");

                // 设置其他字段
            employeeExtReqDto.setType(0);
            employeeExtReqDto.setStatus(1);
                employeeExtReqDto.setStartTime(data.getStartDate() != null ? data.getStartDate() : new Date());
            employeeExtReqDto.setCustomerInfoList(data.getCustomerInfoList());
            employeeExtReqDto.setCustomerBlackList(data.getCustomerBlackList());
            employeeExtReqDto.setRegionInfoList(data.getAreaList());

                // 设置工号
                if (StringUtils.isNotEmpty(data.getEmployeeCode())) {
                    employeeExtReqDto.setEmployeeNo(data.getEmployeeCode());
                }

                // 处理角色
                if (CollectionUtils.isNotEmpty(data.getRoleList())) {
                    employeeExtReqDto.setEmployeeRoleDtoList(data.getRoleList());
                }

                // 根据操作类型确定是新增还是修改
                if (isModifyMode) {
                    // 修改模式 - 根据工号查找对应员工
                    if (StringUtils.isNotEmpty(data.getEmployeeCode()) && existingEmployeeMap.containsKey(data.getEmployeeCode())) {
                        EmployeeExtRespDto existingEmployee = existingEmployeeMap.get(data.getEmployeeCode());
                        employeeExtReqDto.setId(existingEmployee.getId());

                        // 修改操作 - 可能有些字段为空，需要使用原值
                        if (StringUtils.isBlank(employeeExtReqDto.getName())) {
                            employeeExtReqDto.setName(existingEmployee.getName());
                        }
                        if (StringUtils.isBlank(employeeExtReqDto.getPhoneNum())) {
                            employeeExtReqDto.setPhoneNum(existingEmployee.getPhoneNum());
                        }
                        if (StringUtils.isBlank(employeeExtReqDto.getPosition())) {
                            employeeExtReqDto.setPosition(existingEmployee.getPosition());
                        }

                        // 执行修改操作
                        logger.info("导入商家人员修改: employeeCode={}, name={}, phone={}",
                                data.getEmployeeCode(), employeeExtReqDto.getName(), employeeExtReqDto.getPhoneNum());
                        employeeExpand2Service.modifyEmployee(data.getInstanceId(), employeeExtReqDto);
                    } else {
                        // 没找到对应的员工
                        logger.warn("找不到对应工号的员工: {}", data.getEmployeeCode());
                        setErrorsMsg(importDataList, data.getAccount(), "找不到对应工号的员工");
                    }
                } else {
                    // 新增模式 - 检查是否已存在
                    String key = data.getOrgId() + "_" + data.getUserId();
                    if (employeeInfoMap != null && !employeeInfoMap.isEmpty() && employeeInfoMap.containsKey(key)) {
                        // 已存在，执行修改
                        Long id = employeeInfoMap.get(key);
                        employeeExtReqDto.setId(id);
                        logger.info("新增模式下发现已存在的员工，执行更新: userId={}, name={}, phone={}",
                                data.getUserId(), data.getUserName(), data.getPhoneNumber());
                        employeeExpand2Service.modifyEmployee(data.getInstanceId(), employeeExtReqDto);
                    } else {
                        // 执行新增操作
                        logger.info("导入商家人员新增: userId={}, name={}, phone={}",
                                data.getUserId(), data.getUserName(), data.getPhoneNumber());
                        employeeExpand2Service.addEmployee(data.getInstanceId(), employeeExtReqDto);
                    }
                }
            } catch (Exception e) {
                logger.error("导入商家人员失败: {}", e.getMessage(), e);
                setErrorsMsg(importDataList, data.getAccount(), "执行失败！" + e.getMessage());
            }
        }

        logger.info("商家人员导入处理完成");
        return new RestResponse<>(importDataList);
    }

    /**
     * 统一的基本数据校验方法
     * 
     * @param importDataList 导入数据列表
     * @param userOrgMap 用户组织映射
     * @param existingEmployeeMap 现有员工映射（修改模式下使用）
     * @param isModifyMode 是否为修改模式
     * @return 校验后的数据
     */
    private List<SellerEmployeeImportDto> verifyData(List<SellerEmployeeImportDto> importDataList, 
                                                 Map<String, List<UserOrgizationRespDto>> userOrgMap,
                                                 Map<String, EmployeeExtRespDto> existingEmployeeMap,
                                                 boolean isModifyMode) {
        for (SellerEmployeeImportDto vo : importDataList) {
            // 如果已经有错误，跳过验证
            if (StringUtils.isNotBlank(vo.getErrorReason())) {
                continue;
            }

            // 用于收集所有验证错误
            List<String> errorList = new ArrayList<>();

            // 1. 验证操作类型 (type字段) - 如果提供了，必须是"新增"或"修改"
            if (StringUtils.isNotEmpty(vo.getType()) && !("新增".equals(vo.getType()) || "修改".equals(vo.getType()))) {
                errorList.add("操作类型只能为'新增'或'修改'");
            }

            // 2. 根据操作类型校验
            boolean recordModifyMode = isModifyMode;
            if (StringUtils.isNotEmpty(vo.getType())) {
                recordModifyMode = "修改".equals(vo.getType());
            }

            // 修改模式下工号必填
            if (recordModifyMode && StringUtils.isBlank(vo.getEmployeeCode())) {
                errorList.add("修改操作时，工号不能为空");
            }

            // 修改模式下验证工号是否存在
            if (recordModifyMode && StringUtils.isNotBlank(vo.getEmployeeCode())) {
                if (existingEmployeeMap == null || !existingEmployeeMap.containsKey(vo.getEmployeeCode())) {
                    errorList.add("工号不存在，无法修改");
                }
            }

            // 3. 必填字段校验
            if (StringUtils.isBlank(vo.getAccount())) {
                errorList.add("登录账号不能为空");
            }

            if (StringUtils.isBlank(vo.getUserName())) {
                errorList.add("姓名不能为空");
            }

            if (StringUtils.isBlank(vo.getPosition())) {
                errorList.add("职位不能为空");
            }

            if (StringUtils.isBlank(vo.getEmployeeType())) {
                errorList.add("人员类型不能为空");
            }

            if (StringUtils.isBlank(vo.getPhoneNumber())) {
                errorList.add("手机号不能为空");
            }

            if (StringUtils.isBlank(vo.getGovernType())) {
                errorList.add("管理类型不能为空");
            }

            if (StringUtils.isBlank(vo.getRoleName())) {
                errorList.add("角色不能为空");
            }

            // 验证账号是否存在
            if (StringUtils.isNotBlank(vo.getAccount()) && (userOrgMap.isEmpty() || !userOrgMap.containsKey(vo.getAccount()))) {
                errorList.add("登录账号错误");
            }

            // 4. 验证人员类型
            if (StringUtils.isNotBlank(vo.getEmployeeType()) && !("内部".equals(vo.getEmployeeType()) || "外部".equals(vo.getEmployeeType()))) {
                errorList.add("人员类型只能为'内部'或'外部'");
            }

            // 5. 验证手机号格式
            if (StringUtils.isNotEmpty(vo.getPhoneNumber()) && !CommonUtil.checkPhoneNum(vo.getPhoneNumber())) {
                errorList.add("手机号格式不正确");
            }

            // 6. 验证开始日期
            if (vo.getStartDate() == null) {
                errorList.add("开始日期不能为空");
            }

            // 如果账号存在，则继续进行账号相关验证
            if (StringUtils.isNotBlank(vo.getAccount()) && !userOrgMap.isEmpty() && userOrgMap.containsKey(vo.getAccount())) {
                // 7. 验证账号与组织的关联
                List<UserOrgizationRespDto> userOrgizationVos = userOrgMap.get(vo.getAccount());
                List<UserOrgizationRespDto> userOrgList = userOrgizationVos.stream()
                        .filter(e -> vo.getOrgId() != null && vo.getOrgId().compareTo(e.getOrgId()) == 0)
                        .collect(Collectors.toList());
    
                if (CollectionUtils.isEmpty(userOrgList)) {
                    errorList.add("登录账号所属组织错误");
                } else {
                    // 设置用户ID
                    vo.setUserId(userOrgList.get(0).getUserId());
                }
    
                // 8. 校验姓名和手机号唯一性 - 仅在新增模式下
                if (!recordModifyMode && vo.getOrgId() != null && StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber())) {
                    EmployeeExtQueryReqDto expandReqDto = new EmployeeExtQueryReqDto();
                    expandReqDto.setName(vo.getUserName());
                    expandReqDto.setPhoneNum(vo.getPhoneNumber());
            expandReqDto.setOrganizationId(vo.getOrgId());
                    
            List<EmployeeExtRespDto> employeeExtRespDtos = checkNameAndPhoneNum(expandReqDto);
            if (CollectionUtils.isNotEmpty(employeeExtRespDtos) && !Objects.isNull(employeeExtRespDtos.get(0).getId())) {
                        // 明确指出是姓名和手机号的组合重复，而不是暗示姓名或手机号重复
                        errorList.add("姓名为:" + expandReqDto.getName() + "且手机号为:" + expandReqDto.getPhoneNum() + "的员工已经存在");
                    }
                }
    
                // 9. 如果是修改模式，需要做一些特别校验
                if (recordModifyMode && StringUtils.isNotEmpty(vo.getEmployeeCode()) && existingEmployeeMap != null && existingEmployeeMap.containsKey(vo.getEmployeeCode())) {
                    // 在修改模式下，如果同时提供了姓名和手机号，需要校验与其他员工是否重复
                    if (StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber())) {
                        EmployeeExtQueryReqDto expandReqDto = new EmployeeExtQueryReqDto();
                        expandReqDto.setName(vo.getUserName());
                        expandReqDto.setPhoneNum(vo.getPhoneNumber());
                        expandReqDto.setOrganizationId(vo.getOrgId());
                        
                        List<EmployeeExtRespDto> existingByNamePhone = checkNameAndPhoneNum(expandReqDto);
                        if (CollectionUtils.isNotEmpty(existingByNamePhone)) {
                            // 查找当前工号对应的员工信息
                            EmployeeExtRespDto currentEmployee = existingEmployeeMap.get(vo.getEmployeeCode());
                            
                            // 检查是否有其他不同ID的员工具有相同的姓名和手机号
                            boolean otherEmployeeExists = existingByNamePhone.stream()
                                .anyMatch(emp -> !emp.getId().equals(currentEmployee.getId()));
                                
                            if (otherEmployeeExists) {
                                errorList.add("姓名为:" + vo.getUserName() + "且手机号为:" + vo.getPhoneNumber() + "的其他员工已经存在");
                            }
                        }
                    }
                }
            }

            // 如果收集到错误，设置错误原因
            if (!errorList.isEmpty()) {
                vo.setErrorReason(String.join("；", errorList));
            }
        }
        return importDataList;
    }

    public Map<String, List<UserOrgizationRespDto>> getAccoutMap(List<String> accountList) {
        if (CollectionUtils.isEmpty(accountList)) {
            return new HashMap<>();
        }
        Map<String, List<UserOrgizationRespDto>> userOrgMap = new HashMap<>(accountList.size() * 4 / 3 + 2);
        List<UserOrgizationRespDto> userOrgInfoList = RestResponseHelper.extractData(userOrgExpandQueryApi.queryUserOrgByUserName(accountList));
        if (CollectionUtils.isEmpty(userOrgInfoList)) {
            return userOrgMap;
        }
        userOrgMap = userOrgInfoList.stream().collect(Collectors.groupingBy(UserOrgizationRespDto::getUserName));
        return userOrgMap;
    }

    private List<EmployeeExtRespDto> checkNameAndPhoneNum(EmployeeExtQueryReqDto employeeDto) {
        return employeeExpandQueryApi.queryEmployeeByPhoneNumOrg(employeeDto).getData();
    }

    /**
     * 校验组织数据
     *
     * @param list        原数据
     * @param roleMap     角色映射
     * @param areasMap    区域映射
     * @param customerMap 客户映射
     * @param userOrgMap  用户组织映射
     * @return 校验后的数据
     */
    private List<SellerEmployeeImportDto> verifyOrgData(List<SellerEmployeeImportDto> list,
                                                        Map<String, List<RoleExpandRespDto>> roleMap,
                                                        Map<String, List<CustomerAreaRespDto>> areasMap,
                                                        Map<String, CustomerRespDto> customerMap,
                                                        Map<String, List<UserOrgizationRespDto>> userOrgMap) {

        for (SellerEmployeeImportDto dto : list) {
            // 如果已经有错误，跳过后续校验
            if (StringUtils.isNotEmpty(dto.getErrorReason())) {
                continue;
            }

            // 用于收集所有验证错误
            List<String> errorList = new ArrayList<>();
            Long orgId = dto.getOrgId();

            // 校验开始日期
            if (ObjectUtils.isEmpty(dto.getStartDate())) {
                errorList.add("开始日期不能为空");
            }

            // 校验账号与组织的关联
            if (userOrgMap.isEmpty() || !userOrgMap.containsKey(dto.getAccount())) {
                errorList.add("登录账号错误");
            }

            boolean canContinueUserCheck = false;
            if (!userOrgMap.isEmpty() && userOrgMap.containsKey(dto.getAccount())) {
                List<UserOrgizationRespDto> userOrgizationVos = userOrgMap.get(dto.getAccount());
                List<UserOrgizationRespDto> userOrgList = userOrgizationVos.stream()
                        .filter(e -> orgId != null && orgId.compareTo(e.getOrgId()) == 0)
                        .collect(Collectors.toList());
    
                if (CollectionUtils.isEmpty(userOrgList)) {
                    errorList.add("登录账号所属组织错误");
                } else {
                    // 设置用户ID
                    dto.setUserId(userOrgList.get(0).getUserId());
                    canContinueUserCheck = true;
                }
            }

            // 校验角色
            if (StringUtils.isNoneBlank(dto.getRoleName())) {
                if (roleMap.isEmpty() || !roleMap.containsKey(dto.getRoleName())) {
                    errorList.add("角色错误");
                } else {
                    List<RoleExpandRespDto> roleExpandEos = roleMap.get(dto.getRoleName());
                    List<RoleExpandRespDto> orgRoleList = roleExpandEos.stream()
                            .filter(e -> (Objects.nonNull(e.getBelongOrgId()) && 
                                    orgId != null && e.getBelongOrgId().compareTo(orgId) == 0) || 
                                    e.getRoleType() == 1)
                            .collect(Collectors.toList());
    
                    if (CollectionUtils.isEmpty(orgRoleList)) {
                        errorList.add("角色所属组织错误");
                    } else {
                EmployeeRoleReqDto employeeRoleReqDto = new EmployeeRoleReqDto();
                employeeRoleReqDto.setRoleId(orgRoleList.get(0).getId());
                employeeRoleReqDto.setRoleCode(orgRoleList.get(0).getCode());
                dto.setRole(employeeRoleReqDto);
                    }
                }
            } else {
                errorList.add("角色不能为空");
            }

            // 校验管理类型
            EmpGovernTypeEnum empGovernTypeEnum = EmpGovernTypeEnum.fromName(dto.getGovernType());
            if (ObjectUtils.isEmpty(empGovernTypeEnum)) {
                errorList.add("管理类型填写有误");
            } else {
                // 如果是外部人员只能设置为管理客户
                if ("外部".equals(dto.getEmployeeType()) && !EmpGovernTypeEnum.CUS.getName().equals(empGovernTypeEnum.getName())) {
                    errorList.add("外部人员只能选择管理客户类型");
                }
    
                // 检查同一个客户是否同时存在于管理客户和黑名单中
                if (StringUtils.isNotBlank(dto.getGovernCustomer()) && StringUtils.isNotBlank(dto.getBlacklist())
                        && dto.getGovernCustomer().equals(dto.getBlacklist())) {
                    errorList.add("同一个客户不能同时存在管理黑名单和管理客户");
                }
    
                // 根据管理类型验证相关字段
                if (canContinueUserCheck) {
                    switch (empGovernTypeEnum) {
                        case AREA: // 管理区域
                            validateAreaManagement(dto, areasMap, errorList);
                            break;
                        case CUS: // 管理客户
                            validateCustomerManagement(dto, customerMap, errorList);
                            break;
                        case BLA: // 管理黑名单
                            validateBlacklistManagement(dto, customerMap, errorList);
                            break;
                        default:
                            errorList.add("未知的管理类型");
                            break;
                    }
                }
            }

            // 校验角色 - 处理多角色情况
            if (StringUtils.isNotBlank(dto.getRoleName())) {
                processRoles(dto, roleMap, errorList);
            }

            // 如果收集到错误，设置错误原因
            if (!errorList.isEmpty()) {
                dto.setErrorReason(String.join("；", errorList));
            }
        }
        return list;
    }
    
    /**
     * 验证区域管理
     */
    private void validateAreaManagement(SellerEmployeeImportDto dto, 
                                        Map<String, List<CustomerAreaRespDto>> areasMap,
                                        List<String> errorList) {
        // 针对灵活的区域管理模式，允许只填写销售区域、大区或区域中的一个
        boolean hasSaleArea = StringUtils.isNotBlank(dto.getSaleArea());
        boolean hasBigArea = StringUtils.isNotBlank(dto.getBigArea());
        boolean hasDistrict = StringUtils.isNotBlank(dto.getDistrict());

        // 三个都没填，报错
        if (!hasSaleArea && !hasBigArea && !hasDistrict) {
            errorList.add("销售区域、大区、区域至少填写一项");
            return;
        }

        // 情况1：只填了销售区域 - 自动匹配所有下属大区和区域
        if (hasSaleArea && !hasBigArea && !hasDistrict) {
                List<CustomerAreaRespDto> saleAreaList = areasMap.get(dto.getSaleArea());
            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            saleAreaList = saleAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            // 获取销售区域的区域编码
            String saleAreaCode = saleAreaList.get(0).getCode();

            // 设置区域信息 - 使用销售区域编码
            EmployeeRegionReqDto employeeRegionReqDto = new EmployeeRegionReqDto();
            employeeRegionReqDto.setRegionCode(saleAreaCode);
            employeeRegionReqDto.setStartTime(dto.getStartDate());
            employeeRegionReqDto.setEndTime(dto.getEndDate());
            dto.setArea(employeeRegionReqDto);

            logger.info("只提供销售区域: {}, 自动绑定该销售区域及其下所有大区和区域", dto.getSaleArea());
        }
        // 情况2：只填了大区 - 自动查找对应的销售区域，匹配所有下属区域
        else if (!hasSaleArea && hasBigArea && !hasDistrict) {
                List<CustomerAreaRespDto> bigAreaList = areasMap.get(dto.getBigArea());
            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            bigAreaList = bigAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            // 大区已找到，获取大区编码
            String bigAreaCode = bigAreaList.get(0).getCode();

            // 设置区域信息 - 使用大区编码
            EmployeeRegionReqDto employeeRegionReqDto = new EmployeeRegionReqDto();
            employeeRegionReqDto.setRegionCode(bigAreaCode);
            employeeRegionReqDto.setStartTime(dto.getStartDate());
            employeeRegionReqDto.setEndTime(dto.getEndDate());
            dto.setArea(employeeRegionReqDto);

            // 为日志显示，自动填充销售区域信息
            String parentCode = bigAreaList.get(0).getParentCode();
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    if (parentCode.equals(area.getCode())) {
                        dto.setSaleArea(area.getName());
                        break;
                    }
                }
                if (StringUtils.isNotEmpty(dto.getSaleArea())) {
                    break;
                }
            }

            logger.info("只提供大区: {}, 自动匹配销售区域: {}, 并绑定该大区及其下所有区域", 
                    dto.getBigArea(), dto.getSaleArea());
        }
        // 情况3：只填了区域 - 仅包含该区域
        else if (!hasSaleArea && !hasBigArea && hasDistrict) {
                List<CustomerAreaRespDto> districtList = areasMap.get(dto.getDistrict());
            if (CollectionUtils.isEmpty(districtList)) {
                errorList.add("找不到区域");
                return;
            }

            districtList = districtList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(districtList)) {
                errorList.add("找不到区域");
                return;
            }

            // 区域已找到，获取区域编码
            String districtCode = districtList.get(0).getCode();

            // 设置区域信息 - 只使用区域编码
            EmployeeRegionReqDto employeeRegionReqDto = new EmployeeRegionReqDto();
            employeeRegionReqDto.setRegionCode(districtCode);
            employeeRegionReqDto.setStartTime(dto.getStartDate());
            employeeRegionReqDto.setEndTime(dto.getEndDate());
            dto.setArea(employeeRegionReqDto);

            // 为日志显示，尝试自动填充大区和销售区域信息
            String parentCode = districtList.get(0).getParentCode();
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    if (parentCode.equals(area.getCode())) {
                        dto.setBigArea(area.getName());

                        // 再找销售区域
                        String bigAreaParentCode = area.getParentCode();
                        for (Map.Entry<String, List<CustomerAreaRespDto>> entry2 : areasMap.entrySet()) {
                            for (CustomerAreaRespDto area2 : entry2.getValue()) {
                                if (bigAreaParentCode.equals(area2.getCode())) {
                                    dto.setSaleArea(area2.getName());
                                    break;
                                }
                            }
                            if (StringUtils.isNotEmpty(dto.getSaleArea())) {
                                break;
                            }
                        }
                        break;
                    }
                }
                if (StringUtils.isNotEmpty(dto.getBigArea())) {
                    break;
                }
            }

            logger.info("只提供区域: {}, 只绑定该区域", dto.getDistrict());
        }
        // 情况4：传统模式 - 三项都填了
        else if (hasSaleArea && hasBigArea && hasDistrict) {
            validateCompleteAreaHierarchy(dto, areasMap, errorList);
        }
        
        // 创建区域列表
        if (!errorList.contains("找不到销售区域") && !errorList.contains("找不到大区") && !errorList.contains("找不到区域") &&
            !errorList.contains("销售区域下不存在该大区") && !errorList.contains("大区下不存在该区域")) {
            List<EmployeeRegionReqDto> areaList = new ArrayList<>();
            if (dto.getArea() != null) {
                areaList.add(dto.getArea());
            }
            dto.setAreaList(areaList);
        }
    }
    
    /**
     * 验证完整的区域层级关系
     */
    private void validateCompleteAreaHierarchy(SellerEmployeeImportDto dto, 
                                             Map<String, List<CustomerAreaRespDto>> areasMap,
                                             List<String> errorList) {
        // 验证销售区域
        List<CustomerAreaRespDto> saleAreaList = areasMap.get(dto.getSaleArea());
        if (CollectionUtils.isEmpty(saleAreaList)) {
            errorList.add("找不到销售区域");
            return;
        }
        saleAreaList = saleAreaList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(saleAreaList)) {
            errorList.add("找不到销售区域");
            return;
        }

        // 验证大区
        List<CustomerAreaRespDto> bigAreaList = areasMap.get(dto.getBigArea());
        if (CollectionUtils.isEmpty(bigAreaList)) {
            errorList.add("找不到大区");
            return;
        }
        bigAreaList = bigAreaList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(bigAreaList)) {
            errorList.add("找不到大区");
            return;
        }
        if (!bigAreaList.get(0).getParentCode().equals(saleAreaList.get(0).getCode())) {
            errorList.add("销售区域下不存在该大区");
            return;
        }

        // 验证区域
        List<CustomerAreaRespDto> districtList = areasMap.get(dto.getDistrict());
        if (CollectionUtils.isEmpty(districtList)) {
            errorList.add("找不到区域");
            return;
        }
        districtList = districtList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), dto.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(districtList)) {
            errorList.add("找不到区域");
            return;
        }
        if (!districtList.get(0).getParentCode().equals(bigAreaList.get(0).getCode())) {
            errorList.add("大区下不存在该区域");
            return;
        }

        // 设置区域信息
                EmployeeRegionReqDto employeeRegionReqDto = new EmployeeRegionReqDto();
                employeeRegionReqDto.setRegionCode(districtList.get(0).getCode());
                employeeRegionReqDto.setStartTime(dto.getStartDate());
                employeeRegionReqDto.setEndTime(dto.getEndDate());
                dto.setArea(employeeRegionReqDto);
            }

    /**
     * 验证客户管理
     */
    private void validateCustomerManagement(SellerEmployeeImportDto dto, 
                                          Map<String, CustomerRespDto> customerMap,
                                          List<String> errorList) {
        if (StringUtils.isBlank(dto.getGovernCustomer())) {
            errorList.add("管理客户不能为空");
            return;
        }
        
                CustomerRespDto customerRespDto = customerMap.get(dto.getGovernCustomer());
        if (Objects.isNull(customerRespDto)) {
            errorList.add("管理客户（客户编号）错误");
            return;
        }
        
                EmployeeCustomerReqDto employeeCustomerReqDto = new EmployeeCustomerReqDto();
                employeeCustomerReqDto.setCustomerId(customerRespDto.getId());
                employeeCustomerReqDto.setStartTime(dto.getStartDate());
                dto.setCustomerDto(employeeCustomerReqDto);
        
        List<EmployeeCustomerReqDto> customerInfoList = new ArrayList<>();
        customerInfoList.add(employeeCustomerReqDto);
        dto.setCustomerInfoList(customerInfoList);
    }
    
    /**
     * 验证黑名单管理
     */
    private void validateBlacklistManagement(SellerEmployeeImportDto dto, 
                                           Map<String, CustomerRespDto> customerMap,
                                           List<String> errorList) {
        if ("外部".equals(dto.getEmployeeType())) {
            errorList.add("外部人员不能选择管理黑名单类型");
            return;
        }
        
        if (StringUtils.isBlank(dto.getBlacklist())) {
            errorList.add("管理黑名单不能为空");
            return;
        }
        
        CustomerRespDto customerBlackRespDto = customerMap.get(dto.getBlacklist());
        if (Objects.isNull(customerBlackRespDto)) {
            errorList.add("管理黑名单（客户编号）错误");
            return;
        }
        
                EmployeeCustomerBlacklistReqDto customerBlacklistReqDto = new EmployeeCustomerBlacklistReqDto();
        customerBlacklistReqDto.setCustomerId(customerBlackRespDto.getId());
                customerBlacklistReqDto.setStartTime(dto.getStartDate());
                dto.setCustomerBlackDto(customerBlacklistReqDto);
        
        List<EmployeeCustomerBlacklistReqDto> blackList = new ArrayList<>();
        blackList.add(customerBlacklistReqDto);
        dto.setCustomerBlackList(blackList);
    }
    
    /**
     * 处理多角色情况
     */
    private void processRoles(SellerEmployeeImportDto dto, 
                            Map<String, List<RoleExpandRespDto>> roleMap,
                            List<String> errorList) {
        String[] roleNames = dto.getRoleName().split(";");
        List<EmployeeRoleReqDto> roleList = new ArrayList<>();
        
        for (String roleName : roleNames) {
            roleName = roleName.trim();
            if (StringUtils.isNotEmpty(roleName)) {
                if (roleMap.containsKey(roleName)) {
                    List<RoleExpandRespDto> roleExpands = roleMap.get(roleName);
                    if (CollectionUtils.isNotEmpty(roleExpands)) {
                        EmployeeRoleReqDto roleReqDto = new EmployeeRoleReqDto();
                        roleReqDto.setRoleId(roleExpands.get(0).getId());
                        roleReqDto.setRoleCode(roleExpands.get(0).getCode());
                        roleList.add(roleReqDto);
                    }
                } else {
                    errorList.add("角色：" + roleName + " 不存在");
                }
            }
        }
        
        if (CollectionUtils.isNotEmpty(roleList)) {
            dto.setRoleList(roleList);
        }
    }

    /**
     * 合并数据，根据姓名和手机号
     *
     * @param list 待合并的列表
     * @return 合并后的列表
     */
    private List<SellerEmployeeImportDto> mergeOrgData(List<SellerEmployeeImportDto> list) {
        if (CollectionUtils.isEmpty(list)) {
            return list;
        }

        Map<String, List<SellerEmployeeImportDto>> namePhoneMap = list.stream()
                .filter(vo -> StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber()))
                .collect(Collectors.groupingBy(vo -> vo.getUserName() + "_" + vo.getPhoneNumber()));

        List<SellerEmployeeImportDto> mergedList = new ArrayList<>();

        // 处理重复的记录
        for (Map.Entry<String, List<SellerEmployeeImportDto>> entry : namePhoneMap.entrySet()) {
            List<SellerEmployeeImportDto> duplicates = entry.getValue();

            if (duplicates.size() == 1) {
                // 没有重复，直接添加
                mergedList.add(duplicates.get(0));
                continue;
            }

            logger.info("合并相同姓名和手机号的记录: {}, 记录数: {}", entry.getKey(), duplicates.size());

            // 合并记录
            SellerEmployeeImportDto mergedVo = new SellerEmployeeImportDto();

            // 保留第一条记录的基本信息
            SellerEmployeeImportDto firstVo = duplicates.get(0);
            mergedVo.setUserName(firstVo.getUserName());
            mergedVo.setPhoneNumber(firstVo.getPhoneNumber());

            // 使用第一个有效的工号
            String employeeCode = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getEmployeeCode()))
                    .map(SellerEmployeeImportDto::getEmployeeCode)
                    .findFirst()
                    .orElse(null);
            mergedVo.setEmployeeCode(employeeCode);

            // 使用第一个有效的账号
            String account = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getAccount()))
                    .map(SellerEmployeeImportDto::getAccount)
                    .findFirst()
                    .orElse(null);
            mergedVo.setAccount(account);

            // 使用第一个有效的职位
            String position = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getPosition()))
                    .map(SellerEmployeeImportDto::getPosition)
                    .findFirst()
                    .orElse(null);
            mergedVo.setPosition(position);

            // 使用第一个有效的用户ID
            Long userId = duplicates.stream()
                    .filter(vo -> vo.getUserId() != null)
                    .map(SellerEmployeeImportDto::getUserId)
                    .findFirst()
                    .orElse(null);
            mergedVo.setUserId(userId);

            // 使用第一个有效的组织ID
            Long orgId = duplicates.stream()
                    .filter(vo -> vo.getOrgId() != null)
                    .map(SellerEmployeeImportDto::getOrgId)
                    .findFirst()
                    .orElse(null);
            mergedVo.setOrgId(orgId);

            // 优先使用内部人员类型
            String employeeType = duplicates.stream()
                    .filter(vo -> "内部".equals(vo.getEmployeeType()))
                    .map(SellerEmployeeImportDto::getEmployeeType)
                    .findFirst()
                    .orElse(duplicates.get(0).getEmployeeType());
            mergedVo.setEmployeeType(employeeType);

            // 合并管理类型，优先级: 管理区域 > 管理黑名单 > 管理客户
            String governType = duplicates.stream()
                    .filter(vo -> EmpGovernTypeEnum.AREA.getName().equals(vo.getGovernType()))
                    .map(SellerEmployeeImportDto::getGovernType)
                    .findFirst()
                    .orElse(duplicates.stream()
                            .filter(vo -> EmpGovernTypeEnum.BLA.getName().equals(vo.getGovernType()))
                            .map(SellerEmployeeImportDto::getGovernType)
                            .findFirst()
                            .orElse(duplicates.stream()
                                    .filter(vo -> EmpGovernTypeEnum.CUS.getName().equals(vo.getGovernType()))
                                    .map(SellerEmployeeImportDto::getGovernType)
                                    .findFirst()
                                    .orElse(null)));
            mergedVo.setGovernType(governType);

            // 合并角色列表
            Set<String> roleSet = new HashSet<>();
            for (SellerEmployeeImportDto vo : duplicates) {
                if (StringUtils.isNotEmpty(vo.getRoleName())) {
                    roleSet.add(vo.getRoleName());
                }
            }
            // 合并所有角色，用分号连接
            if (CollectionUtils.isNotEmpty(roleSet)) {
                mergedVo.setRoleName(String.join(";", roleSet));
            }

            // 日期取最新的
            Date startDate = duplicates.stream()
                    .filter(vo -> vo.getStartDate() != null)
                    .map(SellerEmployeeImportDto::getStartDate)
                    .max(Date::compareTo)
                    .orElse(null);
            mergedVo.setStartDate(startDate);

            Date endDate = duplicates.stream()
                    .filter(vo -> vo.getEndDate() != null)
                    .map(SellerEmployeeImportDto::getEndDate)
                    .max(Date::compareTo)
                    .orElse(null);
            mergedVo.setEndDate(endDate);

            // 合并区域信息 - 使用第一个非空值
            String saleArea = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getSaleArea()))
                    .map(SellerEmployeeImportDto::getSaleArea)
                    .findFirst()
                    .orElse(null);
            mergedVo.setSaleArea(saleArea);

            String bigArea = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getBigArea()))
                    .map(SellerEmployeeImportDto::getBigArea)
                    .findFirst()
                    .orElse(null);
            mergedVo.setBigArea(bigArea);

            String district = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getDistrict()))
                    .map(SellerEmployeeImportDto::getDistrict)
                    .findFirst()
                    .orElse(null);
            mergedVo.setDistrict(district);

            // 收集客户信息
            String governCustomer = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getGovernCustomer()))
                    .map(SellerEmployeeImportDto::getGovernCustomer)
                    .findFirst()
                    .orElse(null);
            mergedVo.setGovernCustomer(governCustomer);

            // 收集黑名单信息
            String blacklist = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getBlacklist()))
                    .map(SellerEmployeeImportDto::getBlacklist)
                    .findFirst()
                    .orElse(null);
            mergedVo.setBlacklist(blacklist);

            // 继承第一条记录的区域对象、客户对象和角色列表
            mergedVo.setArea(firstVo.getArea());
            mergedVo.setCustomerDto(firstVo.getCustomerDto());
            mergedVo.setCustomerBlackDto(firstVo.getCustomerBlackDto());
            mergedVo.setRoleList(firstVo.getRoleList());

            mergedList.add(mergedVo);
        }

        return mergedList;
    }

    /**
     * @param list     原数据
     * @param account  账号
     * @param errorMsg 错误信息
     * @return
     */
    public List<SellerEmployeeImportDto> setErrorsMsg(List<SellerEmployeeImportDto> list, String account, String errorMsg) {
        list.forEach(vo -> {
            if (vo.getAccount() != null && vo.getAccount().equals(account)) {
                if (StringUtils.isBlank(vo.getErrorReason())) {
                    vo.setErrorReason(errorMsg);
                } else {
                    // 将新错误附加到现有错误后面，用分号分隔
                    vo.setErrorReason(vo.getErrorReason() + "；" + errorMsg);
                }
            }
        });
        return list;
    }

}
