package com.dtyunxi.yundt.module.customer.biz.impl.user;

import cn.afterturn.easypoi.excel.ExcelImportUtil;
import cn.afterturn.easypoi.excel.entity.ImportParams;
import cn.afterturn.easypoi.excel.entity.result.ExcelImportResult;
import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.dtyunxi.app.ServiceContext;
import com.dtyunxi.cube.commons.exceptions.BizException;
import com.dtyunxi.cube.utils.bean.CubeBeanUtils;
import com.dtyunxi.icommerce.utils.RestResponseHelper;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.tcbj.api.dto.request.OrgSellerCustomerQueryReqDto;
import com.dtyunxi.tcbj.api.dto.response.OrgSellerCustomerRespDto;
import com.dtyunxi.tcbj.api.query.IOrgSellerCustomerQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.IEmployeeCustomerApi;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.request.CustomerAreaListReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.request.ParentAreCodeReqDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerAreaRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.query.ICustomerAreaQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.dto.entity.NacosPropertiesOrgEntity;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.*;
import com.dtyunxi.yundt.cube.center.customer.api.dto.response.EmployeeCustomerRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.response.EmployeeNumCountRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.response.EmployeeRegionRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.query.ICustomerQueryV3Api;
import com.dtyunxi.yundt.cube.center.customer.api.query.IEmployeeCustomerQueryApi;
import com.dtyunxi.yundt.cube.center.customer.api.query.IEmployeeRegionQueryApi;
import com.dtyunxi.yundt.cube.center.shop.api.dto.response.SellerRespDto;
import com.dtyunxi.yundt.cube.center.user.api.IEmployeeExpandApi;
import com.dtyunxi.yundt.cube.center.user.api.dto.OrganizationDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.UserRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.entity.ImportBaseResponse;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeExtReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.EmployeeRoleReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.RoleExpandReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.UserQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.request.tcbj.EmployeeExtQueryReqDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.EmployeeExtRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.RoleExpandRespDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.response.UserOrgizationRespDto;
import com.dtyunxi.yundt.cube.center.user.api.enums.YesOrNoEnum;
import com.dtyunxi.yundt.cube.center.user.api.query.*;
import com.dtyunxi.yundt.cube.center.user.api.util.AssertUtil;
import com.dtyunxi.yundt.cube.center.user.api.util.BeanPropertyNullUtil;
import com.dtyunxi.yundt.cube.center.user.api.util.RestUtil;
import com.dtyunxi.yundt.module.context.api.IContext;
import com.dtyunxi.yundt.module.customer.api.dto.request.EmployeeImportExtReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.request.user.EmployeeExpandQueryReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.request.user.EmployeeExpandReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.user.EmployeeExpandRespDto;
import com.dtyunxi.yundt.module.customer.api.enums.EmpGovernTypeEnum;
import com.dtyunxi.yundt.module.customer.api.user.IEmployeeExpand2Service;
import com.dtyunxi.yundt.module.customer.api.vo.EmployeeExcelExtVo;
import com.dtyunxi.yundt.module.customer.api.vo.EmployeeExcelUpdateExtVo;
import com.dtyunxi.yundt.module.customer.biz.util.CommonUtil;
import com.dtyunxi.yundt.module.customer.biz.util.EasyPoiExportUtil;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import com.yx.tcbj.center.customer.api.dto.request.CustomerAreaExtReqDto;
import com.yx.tcbj.center.customer.api.dto.request.CustomerExtReqDto;
import com.yx.tcbj.center.shop.api.dto.request.SellerExtQueryReqDto;
import com.yx.tcbj.center.shop.api.query.ISellerExpandQueryApi;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @date 2022-02-14 14:49
 * @since 1.0.0
 */
@Service
public class EmployeeExpand2ServiceImpl implements IEmployeeExpand2Service {

    private static final Logger logger = LoggerFactory.getLogger(EmployeeExpand2ServiceImpl.class);

    @Resource
    private IEmployeeExpandApi employeeExpandApi;
    @Resource
    private IEmployeeExpandQueryApi employeeExpandQueryApi;

    @Resource
    private IEmployeeCustomerQueryApi employeeCustomerQueryApi;
    @Resource
    private IEmployeeCustomerApi employeeCustomerApi;
    @Resource
    private IEmployeeRegionQueryApi employeeRegionQueryApi;

    @Resource
    private ICustomerAreaQueryApi customerAreaQueryApi;
    @Resource
    private IOrgSellerCustomerQueryApi orgSellerCustomerQueryApi;
    @Resource
    private NacosPropertiesOrgEntity nacosPropertiesOrgEntity;
    @Resource
    private ICustomerQueryV3Api customerQueryV3Api;
    @Resource
    private ISellerExpandQueryApi sellerExpandQueryApi;
    @Resource
    private HttpServletRequest request;
    @Resource
    private IContext context;
    @Resource
    private IUserOrgExpandQueryApi userOrgExpandQueryApi;
    @Resource
    private IRoleQueryApi roleQueryApi;
    @Resource
    private IOrgInfoExtQueryApi orgInfoExtQueryApi;
    @Resource
    private IUserExtQueryApi userExtQueryApi;
    @Resource
    private IEmployeeRoleQueryApi employeeRoleQueryApi;



    /**
     * 获取客户信息
     *
     * @return
     */
    @Override
    public Map<String, CustomerRespDto> getOrgCustomerMap(List<String> customerCodeList) {
        if (CollectionUtils.isEmpty(customerCodeList)) {
            return new HashMap<>();
        }
        RestResponse<List<CustomerRespDto>> customerRest = employeeCustomerQueryApi.queryCustomerListByCodes(customerCodeList);
        if (customerRest == null || CollectionUtils.isEmpty(customerRest.getData())) {
            return new HashMap<>();
        }
        List<CustomerRespDto> customerList = customerRest.getData();
        Map<String, CustomerRespDto> customerMap = customerList.stream().collect(Collectors.toMap(CustomerRespDto::getCode, Function.identity(), (k1, k2) -> k1));
        return customerMap;
    }

    /**
     * 获取请求头的组织ID
     *
     * @return 组织ID
     */
    public String headerOrgId() {
        String organizationId = request.getHeader("yes-req-cus-b2b-organizationId");
        logger.info("获取请求头组织ID：{}", organizationId);
        return organizationId;
    }

    /**
     * 根据实例ID查询TenantId
     *
     * @return
     */
    public Long tenantId() {
        Long tenantId = context.tenantId();
        logger.info("tenantId >>> {}", tenantId);
        if (tenantId == null) {
            tenantId = 1L;
        }
        return tenantId;
    }


    /**
     * 新增人员信息
     *
     * @param instanceId  实例ID
     * @param employeeDto 请求dto
     * @return 人员主键ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public RestResponse<Long> addEmployee(Long instanceId, EmployeeExpandReqDto employeeDto) {
        logger.info("新增人员信息，入参：{}", JSONObject.toJSONString(employeeDto));

        // 收集所有验证错误
        List<String> errorList = new ArrayList<>();

        // 电话必填
        if (StringUtils.isBlank(employeeDto.getPhoneNum())) {
            errorList.add("手机号不能为空");
        }

        // 姓名必填
        if (StringUtils.isBlank(employeeDto.getName())) {
            errorList.add("姓名不能为空");
        }

        // 优先获取校验请求头[组织id]
        String orgId = this.headerOrgId();
        logger.info("请求头组织ID：{}", orgId);
        if (Objects.nonNull(employeeDto.getOrganizationId())) {
            // 其次获取请求参数
            orgId = employeeDto.getOrganizationId().toString();
            logger.info("请求参数组织ID：{}", orgId);
        }
        // 再次获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(orgId)) {
            orgId = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", orgId);
        }

        // 组织ID必填
        if (StringUtils.isBlank(orgId)) {
            errorList.add("请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        }

        // 实例ID必填
        if (instanceId == null) {
            errorList.add("实例ID[instanceId]不能为空");
        }

        // 如果有验证错误，抛出异常
        if (!errorList.isEmpty()) {
            throw new BizException(String.join("；", errorList));
        }

        Long tenantId = this.tenantId();
        employeeDto.setOrganizationId(StringUtils.isBlank(this.headerOrgId()) ? Long.parseLong(orgId) : Long.parseLong(this.headerOrgId()));

        //如果是组织人员信息
        if (YesOrNoEnum.isYes(employeeDto.getOrganizationStatus())) {
            if (employeeDto.getOrgId() == null) {
                throw new BizException("所选组织[orgId]不能为空");
            }
            if (StringUtils.isBlank(employeeDto.getOrganizationCode())) {
                throw new BizException("所选组织代码[organizationCode]不能为空");
            }
            employeeDto.setOrganizationId(employeeDto.getOrganizationId());
        }

        Integer type = Objects.isNull(employeeDto.getType()) ? 0 : employeeDto.getType();
        //如果是客户人员
        if (type == 1 && StringUtils.isNotEmpty(employeeDto.getCustomerCode())) {
            //根据客户编码查询当前组织ID
            Map<String, CustomerRespDto> orgCustomerMap = this.getOrgCustomerMap(Collections.singletonList(employeeDto.getCustomerCode()));
            if (orgCustomerMap.isEmpty() || !orgCustomerMap.containsKey(employeeDto.getCustomerCode())) {
                throw new BizException("客户编号[" + employeeDto.getCustomerCode() + "]对应客户信息不存在");
            }
            CustomerRespDto customerRespDto = orgCustomerMap.get(employeeDto.getCustomerCode());
            //人员所属组织ID为客户对应组织ID
            employeeDto.setOrganizationId(customerRespDto.getMerchantId());
        }

        // 检查姓名和电话号码唯一性
        if (StringUtils.isNotEmpty(employeeDto.getName()) && StringUtils.isNotEmpty(employeeDto.getPhoneNum())) {
            checkNameAndPhoneNum(employeeDto);
        }

        employeeDto.setInstanceId(instanceId);
        employeeDto.setTenantId(tenantId);
        employeeDto.setType(type);

        Long employeeId = RestResponseHelper.extractData(employeeExpandApi.addEmployee(instanceId, employeeDto));
        employeeDto.setId(employeeId);
        //绑定人员区域、人员客户、人员黑名单
        this.bindEmployeeCustomer(employeeDto);
        return new RestResponse<>(employeeId);
    }

    /**
     * 检查员工姓名和电话号码
     *
     * @param employeeDto 员工DTO
     */
    private void checkNameAndPhoneNum(EmployeeExpandReqDto employeeDto) {
        EmployeeExtQueryReqDto extQueryReqDto = new EmployeeExpandQueryReqDto();
        extQueryReqDto.setName(employeeDto.getName());
        extQueryReqDto.setPhoneNum(employeeDto.getPhoneNum());
        extQueryReqDto.setOrganizationId(employeeDto.getOrganizationId());
        List<EmployeeExtRespDto> employeeExtRespDtos = employeeExpandQueryApi.queryEmployeeByPhoneNumOrg(extQueryReqDto).getData();

        // 不再直接抛出异常，而是收集错误并返回，便于后续处理
        if (CollectionUtils.isNotEmpty(employeeExtRespDtos) && !Objects.isNull(employeeExtRespDtos.get(0).getId())) {
            String errorMsg = "姓名为:" + employeeDto.getName() + "且手机号为:" + employeeDto.getPhoneNum() + "的员工已经存在";
            throw new BizException(errorMsg);
        }
    }

    /**
     * 检查员工姓名和电话号码，返回错误消息而不是直接抛出异常
     *
     * @param employeeDto 员工DTO
     * @return 如果有错误，返回错误消息；否则返回null
     */
    private String checkNameAndPhoneNumWithMessage(EmployeeExpandReqDto employeeDto) {
        EmployeeExtQueryReqDto extQueryReqDto = new EmployeeExpandQueryReqDto();
        extQueryReqDto.setName(employeeDto.getName());
        extQueryReqDto.setPhoneNum(employeeDto.getPhoneNum());
        extQueryReqDto.setOrganizationId(employeeDto.getOrganizationId());
        List<EmployeeExtRespDto> employeeExtRespDtos = employeeExpandQueryApi.queryEmployeeByPhoneNumOrg(extQueryReqDto).getData();

        if (CollectionUtils.isNotEmpty(employeeExtRespDtos) && !Objects.isNull(employeeExtRespDtos.get(0).getId())) {
            return "姓名为:" + employeeDto.getName() + "且手机号为:" + employeeDto.getPhoneNum() + "的员工已经存在";
        }
        return null;
    }


    /**
     * 修改人员信息
     *
     * @param instanceId  实例ID
     * @param employeeDto 请求dto
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public RestResponse<Void> modifyEmployee(Long instanceId, EmployeeExpandReqDto employeeDto) {
        logger.info("修改人员信息，入参：{}", JSONObject.toJSONString(employeeDto));

        // 收集所有验证错误
        List<String> errorList = new ArrayList<>();

        if (employeeDto.getId() == null) {
            errorList.add("人员ID不能为空");
        }

        if (instanceId == null) {
            errorList.add("实例ID[instanceId]不能为空");
        }

        Long tenantId = this.tenantId();

        // 校验请求头[组织id]
        String orgId = this.headerOrgId();
        logger.info("请求头组织ID：{}", orgId);
        if (Objects.nonNull(employeeDto.getOrganizationId())) {
            // 其次获取请求参数
            orgId = employeeDto.getOrganizationId().toString();
            logger.info("请求参数组织ID：{}", orgId);
        }
        // 再次获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(orgId)) {
            orgId = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", orgId);
        }

        if (StringUtils.isBlank(orgId)) {
            errorList.add("请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        }

        //如果是组织人员信息
        if (YesOrNoEnum.isYes(employeeDto.getOrganizationStatus())) {
            if (employeeDto.getOrgId() == null) {
                errorList.add("所选组织[orgId]不能为空");
            }
            if (StringUtils.isBlank(employeeDto.getOrganizationCode())) {
                errorList.add("所选组织代码[organizationCode]不能为空");
            }
        }

        // 如果有验证错误，抛出异常
        if (!errorList.isEmpty()) {
            throw new BizException(String.join("；", errorList));
        }

        Integer type = Objects.isNull(employeeDto.getType()) ? 0 : employeeDto.getType();
        //如果是客户人员
        if (type == 1 && StringUtils.isNotEmpty(employeeDto.getCustomerCode())) {
            //根据客户编码查询当前组织ID
            Map<String, CustomerRespDto> orgCustomerMap = this.getOrgCustomerMap(Collections.singletonList(employeeDto.getCustomerCode()));
            if (orgCustomerMap.isEmpty() || !orgCustomerMap.containsKey(employeeDto.getCustomerCode())) {
                throw new BizException("客户编号[" + employeeDto.getCustomerCode() + "]对应客户信息不存在");
            }
            CustomerRespDto customerRespDto = orgCustomerMap.get(employeeDto.getCustomerCode());
            //人员所属组织ID为客户对应组织ID
            //employeeDto.setOrganizationId(customerRespDto.getOrgInfoId());
            orgId = customerRespDto.getMerchantId().toString();
            employeeDto.setCustomerCode(null);
        }

        employeeDto.setOrganizationId(Long.parseLong(orgId));
        employeeDto.setInstanceId(instanceId);
        employeeDto.setTenantId(tenantId);

        RestResponseHelper.extractData(employeeExpandApi.modifyEmployee(instanceId, employeeDto));
        //绑定人员区域、人员客户、人员黑名单
        this.bindEmployeeCustomer(employeeDto);
        return RestResponse.VOID;
    }

    /**
     * 分页查询人员信息
     *
     * @param reqDto 请求dto
     * @return 分页数据
     */
    @Override
    public RestResponse<PageInfo<EmployeeExpandRespDto>> queryByPage(EmployeeExpandQueryReqDto reqDto) {
        logger.info("请求参数组织ID：{}", reqDto.getOrganizationId());

        String organizationId = null;
        // 优先请求参数
        if (Objects.isNull(reqDto.getOrganizationId())) {
            // 其次请求头的参数
            organizationId = this.headerOrgId();
            logger.info("请求头组织ID：{}", organizationId);
        } else {
            organizationId = String.valueOf(reqDto.getOrganizationId());
        }
        // 再次获取nacos配置的组织ID（用于开发、测试）
//        if (StringUtils.isBlank(organizationId)){
//            organizationId = nacosPropertiesOrgEntity.orgId();
//            logger.info("nacos配置的组织ID：{}", organizationId);
//        }
        AssertUtil.assertNotBlank(organizationId, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        AssertUtil.assertNotNull(reqDto.getInstanceId(), "实例ID[instanceId]不能为空");
        Long orgId = StringUtils.isBlank(organizationId) ? null : Long.parseLong(organizationId);
        reqDto.setOrganizationId(orgId);
        Long tenantId = this.tenantId();

        PageInfo<EmployeeExpandRespDto> dtoPageInfo = new PageInfo<>();


        List<Long> employeeIds = new ArrayList<>();
        //客户人员查询，通过当前组织id查询组织下对应的客户组织信息
        if (Objects.nonNull(reqDto.getType()) && reqDto.getType() == 1) {
            EmployeeCustomerQueryReqDto queryReqDto = new EmployeeCustomerQueryReqDto();
//            queryReqDto.setUserId(ServiceContext.getContext().getRequestUserId());
            queryReqDto.setCustomerId(reqDto.getCustomerId());
//            queryReqDto.setMerchantId(reqDto.getOrganizationId());
            List<EmployeeCustomerRespDto> employeeCustomerRespDtos = RestResponseHelper.extractData(employeeCustomerQueryApi.queryEmployeeCustomerList(queryReqDto));
            employeeIds.addAll(employeeCustomerRespDtos.stream().map(EmployeeCustomerRespDto::getEmployeeId).distinct().collect(Collectors.toList()));
            reqDto.setCustomerIds(employeeCustomerRespDtos.stream().map(EmployeeCustomerRespDto::getCustomerId).collect(Collectors.toList()));
            List<CustomerRespDto> customerRespDtos = getCustomerRespDtos(reqDto);
            if (CollectionUtils.isEmpty(customerRespDtos) && CollectionUtils.isEmpty(employeeIds) ) {
                return new RestResponse<>(dtoPageInfo);
            }
            if (CollectionUtils.isNotEmpty(employeeIds)){
                reqDto.setEmpCustomerCode(null);
            }
            List<Long> orgIdList = customerRespDtos.stream().map(CustomerRespDto::getOrgInfoId).distinct().collect(Collectors.toList());
            reqDto.setOrganizationIdList(orgIdList);
            orgIdList.add(reqDto.getOrganizationId());
            reqDto.setOrganizationId(null);
        } else {
            List<CustomerRespDto> customerRespDtos = Optional.ofNullable(getCustomerRespDtos(reqDto)).orElse(new ArrayList<>());
            List<Long> orgIdList = customerRespDtos.stream().map(CustomerRespDto::getOrgInfoId).distinct().collect(Collectors.toList());
            orgIdList.add(reqDto.getOrganizationId());
            reqDto.setOrganizationIdList(orgIdList);
            reqDto.setOrganizationId(null);
        }
        // 客户名称、客户编号（大B客户人员列表查询参数）
        List<EmployeeCustomerRespDto> empCustDtos = null;
        if (StringUtils.isNotBlank(reqDto.getCustomerName()) || StringUtils.isNotBlank(reqDto.getCustomerCode())) {
            CustomerExtReqDto customerExtReqDto = new CustomerExtReqDto();
            customerExtReqDto.setCode(reqDto.getCustomerCode());
            customerExtReqDto.setName(reqDto.getCustomerName());
//            customerExtReqDto.setInstanceId(reqDto.getInstanceId());
            customerExtReqDto.setMerchantId(orgId);
            //customerExtReqDto.setTenantId(tenantId);
            // 审核状态（WAIT_AUDIT：待审核、AUDIT_REJECT：审核不通过、AUDIT_PASS：审核通过、INVALID：作废、DRAFT：草稿）
            // customerExtReqDto.setAuditStatus("AUDIT_PASS");
            logger.info("开始查询客户信息(根据客户名称、客户编号查询)，入参：{}", JSON.toJSONString(customerExtReqDto));
            empCustDtos = RestUtil.checkResponse(employeeCustomerQueryApi.queryEmployee(customerExtReqDto));
            logger.info("结束查询客户信息(根据客户名称、客户编号查询)，出参：{}", JSON.toJSONString(empCustDtos));
            if (CollectionUtils.isEmpty(empCustDtos)) {
                return new RestResponse<>(dtoPageInfo);
            } else {
                List<Long> _ids = empCustDtos.stream().map(EmployeeCustomerRespDto::getEmployeeId).collect(Collectors.toList());
                employeeIds.addAll(_ids);
            }
        }

        // 客户区域（大B客户管理查询）
        if (CollectionUtils.isNotEmpty(reqDto.getCustomerAreaList())) {
            List<String> areaCodes = reqDto.getCustomerAreaList().stream()
                    .filter(Objects::nonNull)
                    .filter(o -> Objects.nonNull(o.getAreaCode()))
                    .map(CustomerAreaExtReqDto::getAreaCode)
                    .distinct()
                    .collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(areaCodes)) {
                EmployeeRegionQueryReqDto regionQueryReqDto = new EmployeeRegionQueryReqDto();
                regionQueryReqDto.setInstanceId(reqDto.getInstanceId());
                regionQueryReqDto.setMerchantId(orgId);
                regionQueryReqDto.setRegionCodes(areaCodes);
                logger.info("开始客户区域（大B客户管理查询），入参：{}", JSON.toJSONString(regionQueryReqDto));
                List<EmployeeRegionRespDto> dtoList = RestUtil.checkResponse(employeeRegionQueryApi.queryEmployeeRegionByCodes(regionQueryReqDto));
                logger.info("结束客户区域（大B客户管理查询），出参：{}", JSON.toJSONString(dtoList));
                if (CollectionUtils.isEmpty(dtoList)) {
                    return new RestResponse<>(dtoPageInfo);
                } else {
                    List<Long> _ids = dtoList.stream().map(EmployeeRegionRespDto::getEmployeeId).collect(Collectors.toList());
                    employeeIds.addAll(_ids);
                }
            }
        }

        // 综合设置角色、客户名称、客户编号、客户区域查询参数（根据人员ID查询）
        if (CollectionUtils.isNotEmpty(employeeIds)) {
            reqDto.setIds(employeeIds);
        }

        PageInfo<EmployeeExtRespDto> employeePage = RestResponseHelper.extractData(employeeExpandQueryApi.queryByPage(reqDto));
        CubeBeanUtils.copyProperties(dtoPageInfo, employeePage, "list", "navigatepageNums");
        List<EmployeeExtRespDto> employeePageList = employeePage.getList();
        List<EmployeeExpandRespDto> dtoList = Lists.newArrayList();
        CubeBeanUtils.copyCollection(dtoList, employeePageList, EmployeeExpandRespDto.class);
        AtomicLong newId = new AtomicLong(0L);
        dtoList.forEach(r -> r.setNewId(newId.getAndIncrement()));
        if (Objects.nonNull(reqDto.getType()) && reqDto.getType() == 1) {
            // 填充客户名称
            this.fillCustomInfo(dtoList, orgId);
        } else {
            // 填充管理区域数和管理客户数
            this.fillCustomerAndRegionNum(dtoList, Long.valueOf(organizationId), reqDto.getInstanceId(), tenantId);
        }
        dtoPageInfo.setList(dtoList);
        return new RestResponse<>(dtoPageInfo);
    }

    @Nullable
    private List<CustomerRespDto> getCustomerRespDtos(EmployeeExpandQueryReqDto reqDto) {
        //客户人员查询
        CustomerSearchExtReqDto customerExtQuery = new CustomerSearchExtReqDto();
        customerExtQuery.setMerchantId(reqDto.getOrganizationId());
        customerExtQuery.setIdList(reqDto.getCustomerIds());
        //客户详情根据客户编码查询客户人员信息
        if (StringUtils.isNotEmpty(reqDto.getEmpCustomerCode())) {
            customerExtQuery.setCode(reqDto.getEmpCustomerCode());
        }
        //客户人员客户名称模糊查询
        customerExtQuery.setCustomerNameLike(reqDto.getCustomerNameLike());
        customerExtQuery.setCustomerCodeLike(reqDto.getCustomerCodeLike());
        logger.info("开始查询客户信息(根据客户名称模糊查询)，入参:{}", JSON.toJSONString(customerExtQuery));

        return RestResponseHelper.extractData(customerQueryV3Api.queryCustomerByExtDtoList(customerExtQuery));
    }

    @Override
    public RestResponse<EmployeeExpandRespDto> queryById(EmployeeExtReqDto reqDto) {
        AssertUtil.assertNotNull(reqDto.getId(), "人员主键ID不能为空");
        String organizationId = this.headerOrgId();
        logger.info("请求头组织ID：{}", organizationId);
        // 获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(organizationId)) {
            organizationId = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", organizationId);
        }
        AssertUtil.assertNotBlank(organizationId, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");

        List<EmployeeExpandRespDto> respDtos = new ArrayList<>(1);

        EmployeeExtRespDto employeeExtRespDto = RestResponseHelper.extractData(employeeExpandQueryApi.queryById(reqDto));
        if (Objects.nonNull(employeeExtRespDto) && Objects.nonNull(employeeExtRespDto.getId())) {
            EmployeeExpandRespDto expandRespDto = new EmployeeExpandRespDto();
            CubeBeanUtils.copyProperties(expandRespDto, employeeExtRespDto);
            respDtos.add(expandRespDto);
        }
        if (CollectionUtils.isNotEmpty(respDtos)) {
            // 填充客户名称
            this.fillCustomInfo(respDtos, Long.valueOf(organizationId));
            //填充组织信息
            this.fillOrganizationInfo(respDtos);
            return new RestResponse<>(respDtos.get(0));
        }
        return new RestResponse<>();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public RestResponse<Void> updateStatus(Long instanceId, EmployeeExtReqDto reqDto) {
        AssertUtil.assertNotEmpty(reqDto.getIds(), "人员主键ID[属性名：ids]不能为空");
        AssertUtil.assertNotNull(reqDto.getStatus(), "状态[属性名：status]不能为空");
        String organizationId = this.headerOrgId();
        logger.info("请求头组织ID：{}", organizationId);
        if (StringUtils.isBlank(organizationId)) {
            organizationId = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", organizationId);
        }
        AssertUtil.assertNotBlank(organizationId, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        reqDto.setOrganizationId(Long.valueOf(organizationId));
        RestResponseHelper.extractData(employeeExpandApi.upateStatus(instanceId, reqDto));
        return RestResponse.VOID;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public RestResponse<Void> setSuperAdmin(Long instanceId, EmployeeExtReqDto reqDto) {
        AssertUtil.assertNotNull(reqDto.getId(), "人员主键ID[属性名：id]不能为空");
        AssertUtil.assertNotNull(reqDto.getSuperAdmin(), "是否超级管理员[0-否，1-是]不能为空");
        AssertUtil.assertNotEmpty(reqDto.getEmployeeRoleDtoList(), "角色信息[属性名：employeeRoleDtoList]不能为空");
        /* 校验是否为超级管理员角色 */
//        RoleAccessDto roleAccessDto = roleService.queryById(reqDto.getEmployeeRoleDtoList().get(0).getRoleId());
//        AssertUtil.assertNotNull(roleAccessDto,"找不到对应的角色");
//        AssertUtil.isFalse(roleAccessDto.getCode().equals("admin_code"),"请求角色不是超级管理员角色");

        String organizationId = this.headerOrgId();
        logger.info("请求头组织ID：{}", organizationId);
        if (StringUtils.isBlank(organizationId)) {
            organizationId = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", organizationId);
        }
        AssertUtil.assertNotBlank(organizationId, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        reqDto.setOrganizationId(Long.valueOf(organizationId));
        RestResponseHelper.extractData(employeeExpandApi.setSuperAdmin(instanceId, reqDto));
        return RestResponse.VOID;
    }

    @Override
    public RestResponse<ImportBaseResponse> importExcelEmployee(EmployeeImportExtReqDto importReqDto) {
        String fileUrl = importReqDto.getFileUrl();
        AssertUtil.assertNotBlank(fileUrl, "导入的文件路径为空");
        Long instanceId = importReqDto.getInstanceId();
        String orgIdStr = this.headerOrgId();
        logger.info("请求头组织ID：{}", orgIdStr);
        // 再次获取nacos配置的组织ID（用于开发、测试）
        if (StringUtils.isBlank(orgIdStr)) {
            orgIdStr = nacosPropertiesOrgEntity.orgId();
            logger.info("nacos配置的组织ID：{}", orgIdStr);
        }
        AssertUtil.assertNotBlank(orgIdStr, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        Long orgId = Long.valueOf(orgIdStr);

        /* 设置默认结果 */
        ImportBaseResponse response = new ImportBaseResponse();

        /* 处理文件，读取数据 */
        ImportParams importParams = new ImportParams();
        importParams.setHeadRows(1);// 表格标题行数,默认0
        importParams.setTitleRows(0);//表头行数,默认1
        importParams.setKeyIndex(null);
        List<EmployeeExcelExtVo> importDataList = new ArrayList<>();

        // 识别导入模式 - 根据导入请求DTO中的模块参数
        boolean isModifyMode = !Objects.equals(importReqDto.getModule(), 0);
        try {
            if ((Objects.equals(importReqDto.getModule(), 0))) {
                ExcelImportResult<EmployeeExcelExtVo> result = ExcelImportUtil.importExcelMore(EasyPoiExportUtil.getInputStream(fileUrl), EmployeeExcelExtVo.class, importParams);
                if (ObjectUtils.isNotEmpty(result)) {
                    importDataList = result.getList();
                    //筛选对象属性全部为null数据
                    importDataList = BeanPropertyNullUtil.getAllFieldNullList(importDataList);
                }
            } else {
                ExcelImportResult<EmployeeExcelUpdateExtVo> result = ExcelImportUtil.importExcelMore(EasyPoiExportUtil.getInputStream(fileUrl), EmployeeExcelUpdateExtVo.class, importParams);
                if (ObjectUtils.isNotEmpty(result)) {
                    importDataList = result.getList().stream().map(r -> BeanUtil.copyProperties(r, EmployeeExcelExtVo.class)).collect(Collectors.toList());
                    //筛选对象属性全部为null数据
                    importDataList = BeanPropertyNullUtil.getAllFieldNullList(importDataList);
                }
            }

            if (CollectionUtils.isEmpty(importDataList)) {
                throw new BizException("无法读取 Excel 中的数据，请检查数据是否符合模板格式要求");
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BizException("无法读取 Excel 文件，请检查文件类型");
        }

        logger.info("导入人员数据开始处理，总记录数: {}, 模式: {}",
                importDataList.size(),
                isModifyMode ? "修改模式" : "新增模式");

        if (CollectionUtils.isEmpty(importDataList)) {
            response.setTotalNum(0);
            response.setSuccessNum(0);
            response.setErrorNum(0);
            return new RestResponse<>(response);
        }

        importDataList.forEach(r -> r.setOrgId(orgId));
        response.setTotalNum(importDataList.size());

        // 创建一个Set来存储成功导入的记录的唯一标识
        Set<Integer> successfulImportRows = new HashSet<>();
        List<EmployeeExcelExtVo> errorList = processImportEmployeeExcelExtVos(importDataList, orgId, isModifyMode, instanceId, successfulImportRows);

        // 重新计算成功和失败数量
        int totalCount = response.getTotalNum();
        int successCount = successfulImportRows.size();
        int failureCount = totalCount - successCount;



        if (CollectionUtils.isNotEmpty(errorList)) {
            response.setErrorNum(errorList.size());
            String excelName = "人员导入失败信息-明细Excel-" + com.dtyunxi.cube.utils.DateUtil.getDateFormat(new Date(), com.dtyunxi.cube.utils.DateUtil.YMDSTRING_DATA);
            String exportUrl;
            try {
                exportUrl = EasyPoiExportUtil.getExportUrl(errorList, EmployeeExcelExtVo.class, null, excelName, "xls");
                response.setErrorFileUrl(exportUrl);
            } catch (Exception e) {
                logger.error("人员导入失败信息文件生成失败: {}", e.getMessage(), e);
            }
        }

        response.setSuccessNum(successCount);
        response.setErrorNum(failureCount);

        logger.info("导入结果统计: 总条数={}, 成功数={}, 失败数={}",
                response.getTotalNum(), response.getSuccessNum(), response.getErrorNum());

        return new RestResponse<>(response);
    }

    @Override
    public List<EmployeeExcelExtVo> processImportEmployeeExcelExtVos(List<EmployeeExcelExtVo> importDataList, Long orgId, boolean isModifyMode, Long instanceId, Set<Integer> successfulImportRows) {
        // 为每条记录添加一个唯一标识，方便后续追踪处理结果
        int rowIndex = 0;
        for (EmployeeExcelExtVo vo : importDataList) {
            vo.setRowNum(rowIndex++);
        }

        // 检查重复的姓名和手机号
        Map<String, List<EmployeeExcelExtVo>> duplicateMap = importDataList.stream()
                .filter(vo -> StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber()))
                .collect(Collectors.groupingBy(vo -> vo.getUserName() + "_" + vo.getPhoneNumber()));

        // 记录存在重复的姓名和手机号
        duplicateMap.entrySet().stream()
                .filter(entry -> entry.getValue().size() > 1)
                .forEach(entry -> {
                    List<EmployeeExcelExtVo> duplicates = entry.getValue();
                    logger.info("发现重复的姓名和手机号, 将被合并处理: 姓名={}, 手机号={}, 记录数={}",
                            duplicates.get(0).getUserName(),
                            duplicates.get(0).getPhoneNumber(),
                            duplicates.size());
                });

        // 检查账号是否重复（新增模式下）
//        if (!isModifyMode) {
//            Map<String, List<EmployeeExcelExtVo>> accountDuplicateMap = importDataList.stream()
//                    .filter(vo -> StringUtils.isNotEmpty(vo.getAccount()))
//                    .collect(Collectors.groupingBy(EmployeeExcelExtVo::getAccount));
//
        // 标记重复账号的错误
//            accountDuplicateMap.entrySet().stream()
//                    .filter(entry -> entry.getValue().size() > 1)
//                    .forEach(entry -> {
//                        List<EmployeeExcelExtVo> duplicates = entry.getValue();
//                        String accountName = entry.getKey();
//                        logger.warn("发现重复的账号: {}, 记录数: {}", accountName, duplicates.size());
//
//                        // 保留第一条，其余标记为错误
//                        for (int i = 1; i < duplicates.size(); i++) {
//                            EmployeeExcelExtVo duplicate = duplicates.get(i);
//                            setErrorsMsg(Collections.singletonList(duplicate), duplicate.getAccount(),
//                                    "当前导入批次中已存在相同账号: " + accountName);
//                        }
//                    });
//        }

        // 获取所有工号，用于查询已存在的员工信息
        List<String> employeeCodes = importDataList.stream()
                .filter(data -> StringUtils.isNotEmpty(data.getEmployeeCode()))
                .map(EmployeeExcelExtVo::getEmployeeCode)
                .distinct()
                .collect(Collectors.toList());

        // 查询现有员工信息 - 根据工号
        Map<String, EmployeeExtRespDto> existingEmployeeMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(employeeCodes)) {
            List<EmployeeExtRespDto> existingEmployees = RestResponseHelper.extractData(employeeRoleQueryApi.queryEmployeeByNos(employeeCodes));
            if (CollectionUtils.isNotEmpty(existingEmployees)) {
                existingEmployeeMap = existingEmployees.stream()
                        .collect(Collectors.toMap(EmployeeExtRespDto::getEmployeeNo, Function.identity(), (k1, k2) -> k1));
            }
        }

        /* 获取角色列表数据 */
        // 获取所有角色名称，包括分号分隔的多角色
        List<String> allRoleNames = new ArrayList<>();
        importDataList.forEach(data -> {
            if (StringUtils.isNotEmpty(data.getRoleName())) {
                String[] roleNames = data.getRoleName().split(";");
                for (String roleName : roleNames) {
                    if (StringUtils.isNotEmpty(roleName.trim())) {
                        allRoleNames.add(roleName.trim());
                    }
                }
            }
        });
        List<String> roleNameList = allRoleNames.stream().distinct().collect(Collectors.toList());
        Map<String, List<RoleExpandRespDto>> roleMap = this.getRoleMap(roleNameList, null);

        //获取客户区域
        List<String> areaNameList = importDataList.stream()
                .flatMap(employee -> Stream.of(employee.getSaleArea(), employee.getBigArea(), employee.getDistrict()))
                .filter(StringUtils::isNotEmpty)
                .distinct().collect(Collectors.toList());
        Map<String, List<CustomerAreaRespDto>> customerAreasMap = this.getCustomerAreaMap(areaNameList, orgId);

        //获取账户信息
        List<String> accountList = importDataList.stream()
                .filter(data -> StringUtils.isNotEmpty(data.getAccount()))
                .map(EmployeeExcelExtVo::getAccount)
                .distinct()
                .collect(Collectors.toList());
        Map<String, List<UserOrgizationRespDto>> userOrgMap = this.getAccoutMap(accountList);

        /* 获取客户信息 */
        List<String> customerCodes = new ArrayList<>();
        importDataList.forEach(data -> {
            if (StringUtils.isNotBlank(data.getGovernCustomer())) {
                customerCodes.add(data.getGovernCustomer());
            }
            if (StringUtils.isNotBlank(data.getBlacklist())) {
                customerCodes.add(data.getBlacklist());
            }
        });

        // 查询客户信息
        Map<String, CustomerRespDto> customerMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(customerCodes)) {
            customerMap = this.getOrgCustomerMap(customerCodes);
        }

        /* 校验数据，使用统一的校验方法 */
        this.verifyData(importDataList, roleMap, customerAreasMap, customerMap, userOrgMap, existingEmployeeMap, isModifyMode);


        // 获取无错误的记录
        List<EmployeeExcelExtVo> validRecords = importDataList.stream()
                .filter(dto -> StringUtils.isEmpty(dto.getErrorMsg()))
                .collect(Collectors.toList());

        /* 合并数据 - 根据姓名和手机号合并 */
        logger.info("开始合并数据，总计{}条有效记录", validRecords.size());
        List<EmployeeExcelExtVo> employeeExcelVoList = this.mergeData(validRecords);
        logger.info("合并后剩余{}条记录", employeeExcelVoList.size());

        // 为合并后的记录保存原始行号的映射关系
        Map<EmployeeExcelExtVo, List<Integer>> mergedToOriginalRowsMap = new HashMap<>();
        for (EmployeeExcelExtVo mergedVo : employeeExcelVoList) {
            String namePhoneKey = mergedVo.getUserName() + "_" + mergedVo.getPhoneNumber();
            List<EmployeeExcelExtVo> originalVos = duplicateMap.get(namePhoneKey);
            if (originalVos != null) {
                mergedToOriginalRowsMap.put(mergedVo, originalVos.stream()
                        .map(EmployeeExcelExtVo::getRowNum)
                        .collect(Collectors.toList()));
            } else {
                // 如果没有重复，就只包含自己的行号
                mergedToOriginalRowsMap.put(mergedVo, Collections.singletonList(mergedVo.getRowNum()));
            }
        }

        /* 处理数据 */
        for (EmployeeExcelExtVo data : employeeExcelVoList) {
            try {

                // 如果是修改模式且有工号，查找对应的员工
                EmployeeExtRespDto existingEmployee = null;
                if (StringUtils.isNotEmpty(data.getEmployeeCode())) {
                    existingEmployee = existingEmployeeMap.get(data.getEmployeeCode());
                }

                // 在修改模式下，检查员工是否存在
                if (isModifyMode && existingEmployee == null) {
                    logger.warn("修改模式下找不到工号为[{}]的员工，无法处理", data.getEmployeeCode());

                    // 记录处理失败
                    List<Integer> failedRows = mergedToOriginalRowsMap.get(data);
                    if (failedRows != null) {
                        for (Integer rowNum : failedRows) {
                            for (EmployeeExcelExtVo original : importDataList) {
                                if (original.getRowNum() == rowNum) {
                                    if (StringUtils.isEmpty(original.getErrorMsg())) {
                                        original.setErrorMsg("修改模式下找不到对应工号，无法处理");
                                    }
                                    break;
                                }
                            }
                        }
                    }
                    continue;
                }

                // 在新增模式下，检查员工是否已存在
                if (!isModifyMode && existingEmployee != null) {
                    logger.warn("新增模式下工号为[{}]的员工已存在，无法处理", data.getEmployeeCode());

                    // 记录处理失败
                    List<Integer> failedRows = mergedToOriginalRowsMap.get(data);
                    if (failedRows != null) {
                        for (Integer rowNum : failedRows) {
                            for (EmployeeExcelExtVo original : importDataList) {
                                if (original.getRowNum() == rowNum) {
                                    if (StringUtils.isEmpty(original.getErrorMsg())) {
                                        original.setErrorMsg("新增模式下工号已存在，无法处理");
                                    }
                                    break;
                                }
                            }
                        }
                    }
                    continue;
                }

                // 处理员工数据 - 根据员工是否存在决定新增或修改
                boolean success = processEmployeeData(instanceId, data, existingEmployee);

                if (success) {
                    // 记录所有关联的原始行为成功
                    List<Integer> originalRows = mergedToOriginalRowsMap.get(data);
                    if (originalRows != null) {
                        successfulImportRows.addAll(originalRows);
                    }

                    logger.info("成功处理员工: {}, 行号: {}",
                            data.getUserName(),
                            mergedToOriginalRowsMap.get(data));
                } else {
                    // 记录处理失败
                    logger.error("处理员工失败: {}, 行号: {}",
                            data.getUserName(),
                            mergedToOriginalRowsMap.get(data));

                    // 设置错误消息
                    List<Integer> failedRows = mergedToOriginalRowsMap.get(data);
                    if (failedRows != null) {
                        for (Integer rowNum : failedRows) {
                            for (EmployeeExcelExtVo original : importDataList) {
                                if (original.getRowNum() == rowNum) {
                                    if (StringUtils.isEmpty(original.getErrorMsg())) {
                                        original.setErrorMsg("处理失败");
                                    }
                                    break;
                                }
                            }
                        }
                    }
                }
            } catch (Exception e) {
                logger.error("人员处理执行失败:{}", e.getMessage(), e);

                // 记录异常信息
                List<Integer> failedRows = mergedToOriginalRowsMap.get(data);
                if (failedRows != null) {
                    for (Integer rowNum : failedRows) {
                        for (EmployeeExcelExtVo original : importDataList) {
                            if (Objects.equals(original.getRowNum(), rowNum)) {
                                original.setErrorMsg(StringUtils.isEmpty(original.getErrorMsg()) ?
                                        "执行失败！" + e.getMessage() :
                                        original.getErrorMsg() + "；执行失败！" + e.getMessage());
                                break;
                            }
                        }
                    }
                }
            }
        }

        // 准备导出失败记录
        List<EmployeeExcelExtVo> errorList = importDataList.stream()
                .filter(vo -> !successfulImportRows.contains(vo.getRowNum()))
                .collect(Collectors.toList());
        return errorList;
    }

    /**
     * 统一的校验方法，支持新增和修改操作
     *
     * @param list                原数据
     * @param roleMap             角色映射
     * @param areasMap            区域映射
     * @param customerMap         客户映射
     * @param userOrgMap          用户组织映射
     * @param existingEmployeeMap 现有员工映射（修改模式下使用）
     * @param isModifyMode        是否为修改模式
     * @return 校验后的数据
     */
    private List<EmployeeExcelExtVo> verifyData(List<EmployeeExcelExtVo> list,
                                                Map<String, List<RoleExpandRespDto>> roleMap,
                                                Map<String, List<CustomerAreaRespDto>> areasMap,
                                                Map<String, CustomerRespDto> customerMap,
                                                Map<String, List<UserOrgizationRespDto>> userOrgMap,
                                                Map<String, EmployeeExtRespDto> existingEmployeeMap,
                                                boolean isModifyMode) {

        // 创建用于检查当前导入批次中名称+手机号重复的映射
        // key是 "姓名_手机号" 格式，value是对应的导入数据ID（取自Excel的行号或employeeCode）
        Map<String, String> importBatchNamePhoneMap = new HashMap<>();

        // 第一次扫描：收集所有要修改的名称+手机号组合，用于检查批次内重复
        if (isModifyMode) {
            for (EmployeeExcelExtVo vo : list) {
                // 只处理修改模式下的记录
                if (StringUtils.isNotBlank(vo.getEmployeeCode()) &&
                        StringUtils.isNotBlank(vo.getUserName()) &&
                        StringUtils.isNotBlank(vo.getPhoneNumber())) {

                    String namePhoneKey = vo.getUserName() + "_" + vo.getPhoneNumber();

                    // 如果已经存在相同的名称+手机号组合，记录冲突
                    if (importBatchNamePhoneMap.containsKey(namePhoneKey)) {
                        String existingCode = importBatchNamePhoneMap.get(namePhoneKey);
                        logger.warn("导入批次中发现重复的名称+手机号组合: {}, 已存在于工号: {}, 当前工号: {}",
                                namePhoneKey, existingCode, vo.getEmployeeCode());
                    } else {
                        importBatchNamePhoneMap.put(namePhoneKey, vo.getEmployeeCode());
                    }
                }
            }
        }
        Long orgId = list.get(0).getOrgId();
        for (EmployeeExcelExtVo vo : list) {
            // 用于收集所有验证错误
            List<String> errorList = new ArrayList<>();


            // 如果已经有错误
            if (StringUtils.isNotEmpty(vo.getErrorMsg())) {
                errorList.add(vo.getErrorMsg());
            }
//
            // 1. 根据操作类型校验
            if (isModifyMode) {
                // 修改模式下工号必填
                if (StringUtils.isBlank(vo.getEmployeeCode())) {
                    errorList.add("修改操作时，工号不能为空");
                }

                // 修改模式下验证工号是否存在
                if (StringUtils.isNotBlank(vo.getEmployeeCode())) {
                    if (existingEmployeeMap == null || !existingEmployeeMap.containsKey(vo.getEmployeeCode())) {
                        errorList.add("工号不存在，无法修改");
                    } else {
                        // 工号存在，进行名称+手机号唯一性校验（只有当这两个字段都有值时才校验）
                        if (StringUtils.isNotBlank(vo.getUserName()) && StringUtils.isNotBlank(vo.getPhoneNumber())) {
                            // 构建当前记录的名称+手机号键
                            String namePhoneKey = vo.getUserName() + "_" + vo.getPhoneNumber();

                            // 检查是否与当前导入批次中的其他记录冲突（非自身）
                            if (importBatchNamePhoneMap.containsKey(namePhoneKey) &&
                                    !importBatchNamePhoneMap.get(namePhoneKey).equals(vo.getEmployeeCode())) {
                                errorList.add("当前导入批次中已存在相同姓名和手机号的记录，工号为：" +
                                        importBatchNamePhoneMap.get(namePhoneKey));
                            }

                            // 检查是否与数据库中已有记录冲突（非自身）
                            EmployeeExtRespDto currentEmployee = existingEmployeeMap.get(vo.getEmployeeCode());

                            if (currentEmployee != null) {
                                // 如果当前员工的名称或手机号要修改（与现有记录不同）
                                boolean isNameChanged = !Objects.equals(currentEmployee.getName(), vo.getUserName());
                                boolean isPhoneChanged = !Objects.equals(currentEmployee.getPhoneNum(), vo.getPhoneNumber());

                                if (isNameChanged || isPhoneChanged) {
                                    // 查询数据库是否存在同名同手机号的其他记录
                                    EmployeeExpandReqDto employeeDto = new EmployeeExpandReqDto();
                                    employeeDto.setName(vo.getUserName());
                                    employeeDto.setPhoneNum(vo.getPhoneNumber());
                                    employeeDto.setOrganizationId(orgId);

                                    // 调用已有方法但不用try-catch，因为当前已在批量校验模式
                                    EmployeeExtQueryReqDto extQueryReqDto = new EmployeeExpandQueryReqDto();
                                    extQueryReqDto.setName(employeeDto.getName());
                                    extQueryReqDto.setPhoneNum(employeeDto.getPhoneNum());
                                    extQueryReqDto.setOrganizationId(employeeDto.getOrganizationId());
                                    List<EmployeeExtRespDto> employeeExtRespDtos = employeeExpandQueryApi.queryEmployeeByPhoneNumOrg(extQueryReqDto).getData();

                                    if (CollectionUtils.isNotEmpty(employeeExtRespDtos)) {
                                        for (EmployeeExtRespDto existingEmp : employeeExtRespDtos) {
                                            // 排除自身
                                            if (existingEmp != null && existingEmp.getId() != null &&
                                                    !Objects.equals(existingEmp.getId(), currentEmployee.getId())) {
                                                errorList.add("修改后的姓名为:" + vo.getUserName() +
                                                        "且手机号为:" + vo.getPhoneNumber() +
                                                        "的员工已经存在，员工编号为:" + existingEmp.getEmployeeNo());
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            // 2. 必填字段校验 - 对于两种模式有不同的要求
            // 账号在所有模式下都必填
            if (StringUtils.isBlank(vo.getAccount())) {
                errorList.add("登录账号不能为空");
            }

            // 只在新增模式下校验其他必填字段
            if (!isModifyMode) {
                if (StringUtils.isBlank(vo.getUserName())) {
                    errorList.add("姓名不能为空");
                }

                if (StringUtils.isBlank(vo.getPosition())) {
                    errorList.add("职位不能为空");
                }

                if (StringUtils.isBlank(vo.getEmployeeType())) {
                    errorList.add("人员类型不能为空");
                }

                if (StringUtils.isBlank(vo.getPhoneNumber())) {
                    errorList.add("手机号不能为空");
                }

                if (StringUtils.isBlank(vo.getGovernType())) {
                    errorList.add("管理类型不能为空");
                }

                if (StringUtils.isBlank(vo.getRoleName())) {
                    errorList.add("角色不能为空");
                }

                // 校验开始日期 - 只在新增模式下必填
                if (ObjectUtils.isEmpty(vo.getStartDate())) {
                    errorList.add("开始日期不能为空");
                }
            }

            // 3. 校验账号与组织的关联 - 对所有模式都需要
            if (userOrgMap.isEmpty() || !userOrgMap.containsKey(vo.getAccount())) {
                errorList.add("登录账号错误");
            }

            boolean canContinueUserCheck = false;
            if (!userOrgMap.isEmpty() && userOrgMap.containsKey(vo.getAccount())) {
                List<UserOrgizationRespDto> userOrgizationVos = userOrgMap.get(vo.getAccount());
                List<UserOrgizationRespDto> userOrgList = userOrgizationVos.stream()
                        .filter(e -> orgId != null && orgId.compareTo(e.getOrgId()) == 0)
                        .collect(Collectors.toList());

                if (CollectionUtils.isEmpty(userOrgList)) {
                    errorList.add("登录账号所属组织错误");
                } else {
                    // 设置用户ID
                    vo.setUserId(userOrgList.get(0).getUserId());
                    canContinueUserCheck = true;
                }
            }

            // 4. 以下校验只在填写了相关字段时进行，不再强制要求填写

            // 校验角色 - 如果提供了角色
            if (StringUtils.isNoneBlank(vo.getRoleName())) {
                if (roleMap.isEmpty() || !roleMap.containsKey(vo.getRoleName())) {
                    errorList.add("角色错误");
                } else {
                    List<RoleExpandRespDto> roleExpandEos = roleMap.get(vo.getRoleName());
                    List<RoleExpandRespDto> orgRoleList = roleExpandEos.stream()
                            .filter(e -> (Objects.nonNull(e.getBelongOrgId()) &&
                                    orgId != null && Objects.equals(e.getBelongOrgId(),orgId)) ||
                                    e.getRoleType() == 1)
                            .collect(Collectors.toList());

                    logger.info("角色筛选:{}",JSONObject.toJSONString(orgRoleList));
                    if (CollectionUtils.isEmpty(orgRoleList)) {
                        errorList.add("角色所属组织错误");
                    } else {
                        EmployeeRoleReqDto employeeRoleReqDto = new EmployeeRoleReqDto();
                        employeeRoleReqDto.setRoleId(orgRoleList.get(0).getId());
                        employeeRoleReqDto.setRoleCode(orgRoleList.get(0).getCode());
                        vo.setRole(employeeRoleReqDto);
                    }
                }
            }

            // 验证人员类型 - 如果提供了人员类型
            if (StringUtils.isNotBlank(vo.getEmployeeType()) && !("内部".equals(vo.getEmployeeType()) || "外部".equals(vo.getEmployeeType()))) {
                errorList.add("人员类型只能为'内部'或'外部'");
            }

            // 验证手机号格式 - 如果提供了手机号
            if (StringUtils.isNotEmpty(vo.getPhoneNumber()) && !CommonUtil.checkPhoneNum(vo.getPhoneNumber())) {
                errorList.add("手机号格式不正确");
            }

            // 校验管理类型 - 如果提供了管理类型
            if (StringUtils.isNotEmpty(vo.getGovernType())) {
                EmpGovernTypeEnum empGovernTypeEnum = EmpGovernTypeEnum.fromName(vo.getGovernType());
                if (ObjectUtils.isEmpty(empGovernTypeEnum)) {
                    errorList.add("管理类型填写有误");
                } else {
                    // 如果是外部人员只能设置为管理客户
                    if ("外部".equals(vo.getEmployeeType()) && !EmpGovernTypeEnum.CUS.getName().equals(empGovernTypeEnum.getName())) {
                        errorList.add("外部人员只能选择管理客户类型");
                    }

                    // 检查同一个客户是否同时存在于管理客户和黑名单中
                    if (StringUtils.isNotBlank(vo.getGovernCustomer()) && StringUtils.isNotBlank(vo.getBlacklist())
                            && vo.getGovernCustomer().equals(vo.getBlacklist())) {
                        errorList.add("同一个客户不能同时存在管理黑名单和管理客户");
                    }

                    // 根据管理类型验证相关字段
                    if (canContinueUserCheck) {
                        switch (empGovernTypeEnum) {
                            case AREA: // 管理区域
                                validateAreaManagement(vo, areasMap, errorList);
                                break;
                            case CUS: // 管理客户
                                validateCustomerManagement(vo, customerMap, errorList);
                                break;
                            case BLA: // 管理黑名单
                                validateBlacklistManagement(vo, customerMap, errorList);
                                break;
                            default:
                                errorList.add("未知的管理类型");
                                break;
                        }
                    }
                }
            }

            // 9. 只在新增模式下检查姓名和电话号码唯一性
            if (!isModifyMode && StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber())) {
                // 首先检查当前导入批次内是否有重复
                String namePhoneKey = vo.getUserName() + "_" + vo.getPhoneNumber();

                if (importBatchNamePhoneMap.containsKey(namePhoneKey)) {
                    // 如果已经记录过这个组合但不是当前记录（说明当前批次内有重复）
                    String firstOccurrence = importBatchNamePhoneMap.get(namePhoneKey);
                    if (!firstOccurrence.equals(vo.getEmployeeCode())) {
                        errorList.add("当前导入批次中已存在相同姓名和手机号的记录");
                    }
                } else {
                    // 记录当前的姓名+手机号组合
                    importBatchNamePhoneMap.put(namePhoneKey,
                            StringUtils.isNotBlank(vo.getEmployeeCode()) ? vo.getEmployeeCode() : "新增记录");
                }

                // 然后检查数据库中是否已存在
                EmployeeExpandReqDto employeeDto = new EmployeeExpandReqDto();
                employeeDto.setName(vo.getUserName());
                employeeDto.setPhoneNum(vo.getPhoneNumber());
                employeeDto.setOrganizationId(orgId);

                String errorMsg = checkNameAndPhoneNumWithMessage(employeeDto);
                if (errorMsg != null) {
                    errorList.add(errorMsg);
                }
            }

            // 10. 校验角色 - 处理多角色情况
            if (StringUtils.isNotBlank(vo.getRoleName())) {
                processRoles(vo, roleMap, errorList);
            }

            // 如果收集到错误，设置错误原因
            if (!errorList.isEmpty()) {
                vo.setErrorMsg(String.join("；", errorList));
            }
        }

        return list;
    }

    /**
     * 填充客户姓名和编号
     *
     * @param dtoList
     */
    private void fillCustomInfo(List<EmployeeExpandRespDto> dtoList, Long merchantId) {
        List<Long> orgIdList = dtoList.stream().map(EmployeeExtRespDto::getOrganizationId).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(orgIdList)) {
            //客户人员查询
            CustomerSearchExtReqDto customerExtQuery = new CustomerSearchExtReqDto();
            customerExtQuery.setMerchantId(merchantId);
            customerExtQuery.setOrgInfoIds(orgIdList);
            List<CustomerRespDto> customerRespDtos = RestResponseHelper.extractData(customerQueryV3Api.queryCustomerByExtDtoList(customerExtQuery));
            if (CollectionUtils.isEmpty(customerRespDtos)) {
                return;
            }
            Map<String, CustomerRespDto> customerMap = customerRespDtos.stream().collect(Collectors.toMap(dto -> {
                return merchantId + "_" + dto.getOrgInfoId();
            }, Function.identity()));
            for (EmployeeExtRespDto dto : dtoList) {
                String key = merchantId + "_" + dto.getOrganizationId();
                if (!customerMap.isEmpty() && customerMap.containsKey(key)) {
                    CustomerRespDto customerRespDto = customerMap.get(key);
                    dto.setCustomerName(customerRespDto.getName());
                    dto.setCustomerCode(customerRespDto.getCode());
                }
            }
        }
    }

    /**
     * 填充客户编号和客户名称
     *
     * @param dtoList 返回的数据list
     */
    private void fillCustomerCodeAndName(List<EmployeeExpandRespDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        for (EmployeeExtRespDto respDto : dtoList) {
            if (Objects.isNull(respDto.getId())) {
                continue;
            }
            CustomerExtReqDto customerExtReqDto = new CustomerExtReqDto();
            customerExtReqDto.setInstanceId(respDto.getInstanceId());
            //customerExtReqDto.setTenantId(respDto.getTenantId());
            customerExtReqDto.setOrgInfoId(respDto.getOrganizationId());
            customerExtReqDto.setEmployeeIds(Arrays.asList(respDto.getId()));
            customerExtReqDto.setType(respDto.getType());
            // 审核状态（WAIT_AUDIT：待审核、AUDIT_REJECT：审核不通过、AUDIT_PASS：审核通过、INVALID：作废、DRAFT：草稿）
            // customerExtReqDto.setAuditStatus("AUDIT_PASS");
            List<EmployeeCustomerRespDto> customerRespDtos = RestUtil.checkResponse(employeeCustomerQueryApi.queryEmployee(customerExtReqDto));
            if (CollectionUtils.isNotEmpty(customerRespDtos)) {
                EmployeeCustomerRespDto customerRespDto = customerRespDtos.get(0);
                respDto.setCustomerName(customerRespDto.getCustomerName());
                respDto.setCustomerCode(customerRespDto.getCustomerCode());
            }
        }
    }

    /**
     * 根据客户ID查询管理区域数和管理客户数
     *
     * @param dtoList    返回dto数组
     * @param orgId      组织ID
     * @param instanceId 实例ID
     * @param tenantId   应用ID
     */
    private void fillCustomerAndRegionNum(List<EmployeeExpandRespDto> dtoList, Long orgId, Long instanceId, Long tenantId) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        List<Long> employeeIds = dtoList.stream().map(EmployeeExpandRespDto::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(employeeIds)) {
            return;
        }
        EmployeeNumCountReqDto reqDto = new EmployeeNumCountReqDto();
        reqDto.setEmployeeIds(employeeIds);
        reqDto.setOrgId(orgId);
        //reqDto.setInstanceId(instanceId);
        //reqDto.setTenantId(tenantId);
        RestResponse<List<EmployeeNumCountRespDto>> response = employeeCustomerQueryApi.queryCountByEmployeeId(reqDto);
        if (Objects.nonNull(response) && "0".equals(response.getResultCode()) && CollectionUtils.isNotEmpty(response.getData())) {
            List<EmployeeNumCountRespDto> list = response.getData();
            for (EmployeeExpandRespDto dto : dtoList) {
                for (EmployeeNumCountRespDto countNumDto : list) {
                    if (Objects.nonNull(dto) && Objects.nonNull(dto.getId())
                            && Objects.nonNull(countNumDto) && Objects.nonNull(countNumDto.getEmployeeId())
                            && dto.getId().longValue() == countNumDto.getEmployeeId().longValue()) {
                        dto.setCustomerNum(countNumDto.getCustomerNum());
                        dto.setRegionNum(countNumDto.getRegionNum());
                    }
                }
            }
        }
    }

    /**
     * 封装组织信息
     *
     * @param dtoList
     */
    private void fillOrganizationInfo(List<EmployeeExpandRespDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return;
        }
        //组织id
        List<Long> organizationIdList = dtoList.stream().map(EmployeeExpandRespDto::getOrganizationId).collect(Collectors.toList());

        OrgSellerCustomerQueryReqDto queryReqDto = new OrgSellerCustomerQueryReqDto();
        queryReqDto.setPageNum(1);
        queryReqDto.setPageSize(organizationIdList.size());
        queryReqDto.setMerchantIdList(organizationIdList);
        RestResponse<PageInfo<OrgSellerCustomerRespDto>> orgSellerCustomerRest = orgSellerCustomerQueryApi.orgSellerCustomerPage(queryReqDto);
        Map<Long, OrgSellerCustomerRespDto> orgSellerMap = new HashMap<>();
        if (orgSellerCustomerRest != null && CollectionUtils.isNotEmpty(orgSellerCustomerRest.getData().getList())) {
            List<OrgSellerCustomerRespDto> orgSellerList = orgSellerCustomerRest.getData().getList();
            orgSellerMap = orgSellerList.stream().collect(Collectors.toMap(OrgSellerCustomerRespDto::getOrgId, Function.identity(), (k1, k2) -> k1));
        }

        for (EmployeeExpandRespDto respDto : dtoList) {
            if (!orgSellerMap.isEmpty() && orgSellerMap.containsKey(respDto.getOrganizationId())) {
                OrgSellerCustomerRespDto orgSellerCustomerRespDto = orgSellerMap.get(respDto.getOrganizationId());
                respDto.setOrganizationCode(orgSellerCustomerRespDto.getOrgCode());
                respDto.setOrganizationName(orgSellerCustomerRespDto.getOrgName());
                respDto.setSellerId(orgSellerCustomerRespDto.getSellerId());
                respDto.setSellerName(orgSellerCustomerRespDto.getSellerName());
                respDto.setSuperiorSellerName(orgSellerCustomerRespDto.getSuperiorSellerName());
                respDto.setDealerCode(orgSellerCustomerRespDto.getDealerCode());
            }
        }
    }

    /**
     * 绑定人员区域、人员客户、人员黑名单
     *
     * @param employeeDto
     */
//    @Transactional(rollbackFor = Exception.class)
    public void bindEmployeeCustomer(EmployeeExpandReqDto employeeDto) {
        logger.info("进入绑定人员客户");
//        String organizationId = super.getHeaderOrgId();
        //AssertUtil.assertNotBlank(organizationId, "请求头缺失组织ID[yes-req-cus-b2b-organizationId]");
        EmployeeCustomerAddReqDto bindCustomerDto = new EmployeeCustomerAddReqDto();
        bindCustomerDto.setEmployeeId(employeeDto.getId());
        bindCustomerDto.setUserId(employeeDto.getUserId());
        if (Objects.isNull(employeeDto.getUserId()) && StringUtils.isNotEmpty(employeeDto.getUserName())) {
            //根据用户名称获取用户id
            UserQueryReqDto userQueryReqDto = new UserQueryReqDto();
            userQueryReqDto.setUserNames(Collections.singletonList(employeeDto.getUserName()));
            List<UserRespDto> userRespDtos = RestResponseHelper.extractData(userExtQueryApi.queryUserByUserNames(userQueryReqDto));
            bindCustomerDto.setUserId(userRespDtos.get(0).getId());
        }

        bindCustomerDto.setMerchantId(Optional.ofNullable(getHeaderOrgId()).orElse(employeeDto.getOrganizationId()));
        if (YesOrNoEnum.isYes(employeeDto.getOrganizationStatus())) {
            bindCustomerDto.setMerchantId(Optional.ofNullable(getHeaderOrgId()).orElse(employeeDto.getOrgId()));
        }
        bindCustomerDto.setRegionInfoList(employeeDto.getRegionInfoList());
        bindCustomerDto.setCustomerInfoList(employeeDto.getCustomerInfoList());
        if (CollectionUtils.isEmpty(bindCustomerDto.getCustomerInfoList()) && StringUtils.isNotEmpty(employeeDto.getCustomerCode())) {
            EmployeeCustomerReqDto employeeCustomerReqDto = new EmployeeCustomerReqDto();
            employeeCustomerReqDto.setEmployeeId(employeeDto.getId());
            employeeCustomerReqDto.setMerchantId(getHeaderOrgId());
            employeeCustomerReqDto.setUserId(employeeDto.getUserId());
            employeeCustomerReqDto.setStartTime(new Date());
            CustomerSearchExtReqDto searchReqDto = new CustomerSearchExtReqDto();
            searchReqDto.setCode(employeeDto.getCustomerCode());
            searchReqDto.setPageNum(1);
            searchReqDto.setPageSize(1);
            List<CustomerRespDto> customerRespDtos = RestUtil.checkResponse(customerQueryV3Api.queryCustomerByExtDtoList(searchReqDto));
            if (CollectionUtils.isNotEmpty(customerRespDtos)) {
                employeeCustomerReqDto.setCustomerId(customerRespDtos.get(0).getId());
                bindCustomerDto.setCustomerInfoList(Lists.newArrayList(employeeCustomerReqDto));
            }
        }
        bindCustomerDto.setCustomerBlackList(employeeDto.getCustomerBlackList());
        employeeCustomerApi.saveEmployeeBindInfo(bindCustomerDto);
        logger.info("进入绑定人员客户结束");
    }

    public Long getHeaderOrgId() {
        String organizationId = ServiceContext.getContext().getAttachment("yes.req.cus.b2b.organizationId");
        logger.info("获取请求头组织ID：{}", organizationId);
        if (StringUtils.isEmpty(organizationId)) {
            return null;
        }
        return Long.parseLong(organizationId);
    }

    @Override
    public RestResponse<PageInfo<EmployeeExpandRespDto>> queryPageByOrg(Long organizationId, Integer pageNum, Integer pageSize) {
        EmployeeExpandQueryReqDto queryReqDto = new EmployeeExpandQueryReqDto();
        queryReqDto.setPageNum(pageNum);
        queryReqDto.setPageSize(pageSize);
        queryReqDto.setOrganizationId(organizationId);
        return this.queryByPage(queryReqDto);
    }


    /**
     * 查询用户的绑定人员有效的组织id
     *
     * @param userId
     * @return
     */
    @Override
    public RestResponse<List<Long>> getEmployeeOrgIdByUserId(Long userId) {
        return employeeExpandQueryApi.getEmployeeOrgIdByUserId(userId);
    }

    @Override
    public RestResponse<List<EmployeeExtRespDto>> queryByRole(Long roleId, Long instanceId) {
        String organizationId = this.headerOrgId();
        logger.info("请求头组织ID：{}", organizationId);
        ServiceContext.getContext().set("yes.req.cus.b2b.organizationId", organizationId);
        MDC.put("yes.req.cus.b2b.organizationId", organizationId);
        return employeeExpandQueryApi.queryByRole(roleId, instanceId, organizationId);
    }

    /**
     * 查询用户的绑定人员有效的组织
     *
     * @param userId
     * @return
     */
    @Override
    public RestResponse<List<OrganizationDto>> getOrgInfoByUserId(Long userId) {
        List<OrganizationDto> organizationDtos = RestResponseHelper.extractData(employeeExpandQueryApi.getOrgInfoByUserId(userId));
        if (CollectionUtils.isEmpty(organizationDtos)) {
            return new RestResponse<>();
        }
        List<Long> orgIdList = organizationDtos.stream().map(OrganizationDto::getId).collect(Collectors.toList());
        SellerExtQueryReqDto sellerExtQueryReqDto = new SellerExtQueryReqDto();
        sellerExtQueryReqDto.setOrgIdList(orgIdList);
        //查询商家信息
        List<SellerRespDto> sellerRespDtos = RestResponseHelper.extractData(sellerExpandQueryApi.queryBatchSellerList(sellerExtQueryReqDto));
        if (CollectionUtils.isEmpty(sellerRespDtos)) {
            return new RestResponse<>(organizationDtos);
        }
        Map<Long, SellerRespDto> sellerOrgMap = sellerRespDtos.stream().collect(Collectors.toMap(SellerRespDto::getOrganizationId, Function.identity(), (k1, k2) -> k1));
        for (OrganizationDto organizationDto : organizationDtos) {
            if (sellerOrgMap.isEmpty() || !sellerOrgMap.containsKey(organizationDto.getId())) {
                continue;
            }
            SellerRespDto sellerRespDto = sellerOrgMap.get(organizationDto.getId());
            organizationDto.setName(sellerRespDto.getName());
        }

        return new RestResponse<>(organizationDtos);
    }

    /**
     * 根据角色名称获取角色信息
     *
     * @param roleNameList
     * @param orgId
     * @return
     */
    @Override
    public Map<String, List<RoleExpandRespDto>> getRoleMap(List<String> roleNameList, Long orgId) {
        if (CollectionUtils.isEmpty(roleNameList)) {
            return new HashMap();
        }
        Map<String, List<RoleExpandRespDto>> roleMap = new HashMap<>(roleNameList.size() * 4 / 3 + 2);
        RoleExpandReqDto roleExpandReqDto = new RoleExpandReqDto();
        roleExpandReqDto.setOrgId(orgId);
        roleExpandReqDto.setRoleNameList(roleNameList);
        List<RoleExpandRespDto> roleExpandRespDtos = RestResponseHelper.extractData(roleQueryApi.getRoleByNameList(roleExpandReqDto));
        if (CollectionUtils.isEmpty(roleExpandRespDtos)) {
            return roleMap;
        }
        roleMap = roleExpandRespDtos.stream().collect(Collectors.groupingBy(RoleExpandRespDto::getName));
        return roleMap;
    }

    /**
     * 根据用户账户获取用户信息
     *
     * @param accountList
     * @return
     */
    @Override
    public Map<String, List<UserOrgizationRespDto>> getAccoutMap(List<String> accountList) {
        if (CollectionUtils.isEmpty(accountList)) {
            return new HashMap<>();
        }
        Map<String, List<UserOrgizationRespDto>> userOrgMap = new HashMap<>(accountList.size() * 4 / 3 + 2);
        List<UserOrgizationRespDto> userOrgInfoList = RestResponseHelper.extractData(userOrgExpandQueryApi.queryUserOrgByUserName(accountList));
        if (CollectionUtils.isEmpty(userOrgInfoList)) {
            return userOrgMap;
        }
        userOrgMap = userOrgInfoList.stream().collect(Collectors.groupingBy(UserOrgizationRespDto::getUserName));
        return userOrgMap;
    }

    /**
     * 根据客户区域编码获取区域信息
     *
     * @param governAreaList
     * @param orgId
     * @return
     */
    @Override
    public Map<String, List<CustomerAreaRespDto>> getCustomerAreaMap(List<String> governAreaList, Long orgId) {
        Map<String, List<CustomerAreaRespDto>> customerAreasMap = new HashMap<>(governAreaList.size() * 4 / 3 + 2);
        if (CollectionUtils.isEmpty(governAreaList)) {
            return customerAreasMap;
        }

        // 1. 首先根据名称获取基础区域信息
        CustomerAreaListExtReqDto areaListReqDto = new CustomerAreaListExtReqDto();
        areaListReqDto.setNameList(governAreaList);
        areaListReqDto.setOrgInfoId(orgId);
        RestResponse<List<CustomerAreaRespDto>> areaRest = employeeRegionQueryApi.queryCustomerAreaByNames(areaListReqDto);

        if (null == areaRest || CollectionUtils.isEmpty(areaRest.getData())) {
            return customerAreasMap;
        }

        List<CustomerAreaRespDto> customerAreaList = areaRest.getData();

        // 2. 收集所有区域的编码，用于查询子区域和父区域
        List<String> areaCodes = customerAreaList.stream()
                .map(CustomerAreaRespDto::getCode)
                .distinct()
                .collect(Collectors.toList());

        // 3. 查询所有子区域
//        CustomerAreaListReqDto subTreeReqDto = new CustomerAreaListReqDto();
//        subTreeReqDto.setCodes(areaCodes);
//        subTreeReqDto.setOrgInfoId(orgId);
        Set<String> allAreaCodes = new HashSet<>();
        try {
            RestResponse<List<String>> subCodesResp = employeeRegionQueryApi.querySubCustomerAreaByCodes(areaCodes);
            if (subCodesResp != null && CollectionUtils.isNotEmpty(subCodesResp.getData())) {
                List<String> allSubCodes = subCodesResp.getData();
//                subTreeReqDto.setCodes(subCodesResp.getData());
                RestResponse<List<String>> subCodesResp1 = employeeRegionQueryApi.querySubCustomerAreaByCodes(subCodesResp.getData());
                // 子区域编码与原始编码合并去重
                allAreaCodes.addAll(areaCodes);
                allAreaCodes.addAll(allSubCodes);
                allAreaCodes.addAll(subCodesResp1.getData());
                // 4. 查询所有父区域
                ParentAreCodeReqDto parentReqDto = new ParentAreCodeReqDto();
                parentReqDto.setCodes(new ArrayList<>(allAreaCodes));
                RestResponse<List<String>> parentCodesResp = customerAreaQueryApi.queryAllParentCode(parentReqDto);

                if (parentCodesResp != null && CollectionUtils.isNotEmpty(parentCodesResp.getData())) {
                    allAreaCodes.addAll(parentCodesResp.getData());

                    // 5. 使用完整的区域编码集合查询所有区域详细信息
                    RestResponse<List<CustomerAreaRespDto>> completeAreasResp =
                            employeeRegionQueryApi.queryCustomerAreaByCodes(new ArrayList<>(allAreaCodes));

                    if (completeAreasResp != null && CollectionUtils.isNotEmpty(completeAreasResp.getData())) {
                        // 使用完整的区域信息替换原始的区域信息
                        customerAreaList = completeAreasResp.getData();
                    }
                }
            }else {
                // 4. 查询所有父区域
                ParentAreCodeReqDto parentReqDto = new ParentAreCodeReqDto();
                parentReqDto.setCodes(new ArrayList<>(areaCodes));
                RestResponse<List<String>> parentCodesResp = customerAreaQueryApi.queryAllParentCode(parentReqDto);

                if (parentCodesResp != null && CollectionUtils.isNotEmpty(parentCodesResp.getData())) {
                    allAreaCodes.addAll(parentCodesResp.getData());
                    parentReqDto.setCodes(new ArrayList<>(parentCodesResp.getData()));
                    RestResponse<List<String>> parentCodesResp2 = customerAreaQueryApi.queryAllParentCode(parentReqDto);
                    allAreaCodes.addAll(parentCodesResp2.getData());
                    // 5. 使用完整的区域编码集合查询所有区域详细信息
                    RestResponse<List<CustomerAreaRespDto>> completeAreasResp =
                            employeeRegionQueryApi.queryCustomerAreaByCodes(new ArrayList<>(allAreaCodes));

                    if (completeAreasResp != null && CollectionUtils.isNotEmpty(completeAreasResp.getData())) {
                        // 使用完整的区域信息替换原始的区域信息
                        customerAreaList = completeAreasResp.getData();
                    }
                }
            }
        } catch (Exception e) {
            logger.error("查询区域层级关系失败: {}", e.getMessage(), e);
        }
        logger.info("区域获取:{}",JSONObject.toJSONString(customerAreasMap));
        // 按名称分组
        customerAreasMap = customerAreaList.stream()
                .collect(Collectors.groupingBy(CustomerAreaRespDto::getName));

        // 按编码添加一份，便于后续处理
        Map<String, List<CustomerAreaRespDto>> areaCodeMap = customerAreaList.stream()
                .collect(Collectors.groupingBy(CustomerAreaRespDto::getCode));

        // 将编码映射合并到名称映射中
        for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areaCodeMap.entrySet()) {
            List<CustomerAreaRespDto> areas = entry.getValue();
            if (!areas.isEmpty()) {
                String areaName = areas.get(0).getName();
                if (!customerAreasMap.containsKey(areaName)) {
                    customerAreasMap.put(areaName, areas);
                }
            }
        }

        logger.info("已获取区域信息总数: {}, 包含销售区域、大区和区域", customerAreaList.size());
        return customerAreasMap;
    }

    /**
     * 根据组织id和用户id获取员工ID信息
     *
     * @param userIdList
     * @param orgIdList
     * @return
     */
    @Override
    public Map<String, Long> getEmployeeMap(List<Long> userIdList, List<Long> orgIdList) {
        Map<String, Long> employeeInfoMap = new HashMap<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return employeeInfoMap;
        }
        EmployeeExtQueryReqDto dto = new EmployeeExtQueryReqDto();
        dto.setOrganizationIdList(orgIdList);
        dto.setUserIds(userIdList);
        List<EmployeeExtRespDto> employeeList = RestResponseHelper.extractData(employeeExpandQueryApi.getEmployeeByUserOrgList(dto));
        if (CollectionUtils.isNotEmpty(employeeList)) {
            employeeInfoMap = employeeList.stream().collect(Collectors.toMap(ee -> ee.getOrganizationId() + "_" + ee.getUserId() + "_" + ee.getPhoneNum(), EmployeeExtRespDto::getId, (o, o1) -> o));
        }
        return employeeInfoMap;
    }


    /**
     * 根据组织名称获取组织信息
     *
     * @param nameList
     * @return
     */
    @Override
    public Map<String, OrganizationDto> getOrgByNameMap(List<String> nameList) {
        if (CollectionUtils.isEmpty(nameList)) {
            return new HashMap<>();
        }
        List<OrganizationDto> dtoList = RestResponseHelper.extractData(orgInfoExtQueryApi.getOrgByNameList(nameList));
        if (CollectionUtils.isEmpty(dtoList)) {
            return new HashMap<>();
        }
        Map<String, OrganizationDto> organizationEoMap = dtoList.stream().collect(Collectors.toMap(OrganizationDto::getName, Function.identity(), (k1, k2) -> k1));
        return organizationEoMap;
    }

    /**
     * @param list     原数据
     * @param account  账号
     * @param errorMsg 错误信息
     * @return
     */
    public List<EmployeeExcelExtVo> setErrorsMsg(List<EmployeeExcelExtVo> list, String account, String errorMsg) {
        list.forEach(vo -> {
            if (StringUtils.isBlank(vo.getErrorMsg()) && vo.getAccount().equals(account)) {
                vo.setErrorMsg(errorMsg);
            } else if (vo.getAccount().equals(account)) {
                // 将新错误附加到现有错误后面，用分号分隔
                vo.setErrorMsg(vo.getErrorMsg() + "；" + errorMsg);
            }
        });
        return list;
    }

    /**
     * 检查员工是否已存在 - 通过工号或姓名+手机号
     *
     * @param employeeCode 工号
     * @param userName     姓名
     * @param phoneNumber  手机号
     * @param orgId        组织ID
     * @return 现有员工信息，如不存在则返回null
     */
    private EmployeeExtRespDto checkExistingEmployee(String employeeCode, String userName, String phoneNumber, Long orgId) {
        // 优先通过工号检查
        if (StringUtils.isNotEmpty(employeeCode)) {
            List<String> codes = new ArrayList<>();
            codes.add(employeeCode);
            List<EmployeeExtRespDto> existingEmployees = RestResponseHelper.extractData(employeeRoleQueryApi.queryEmployeeByNos(codes));
            if (CollectionUtils.isNotEmpty(existingEmployees)) {
                return existingEmployees.get(0);
            }
        }

        // 如果工号没有匹配，尝试通过姓名和手机号检查
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(phoneNumber)) {
            EmployeeExpandReqDto expandReqDto = new EmployeeExpandReqDto();
            expandReqDto.setName(userName);
            expandReqDto.setPhoneNum(phoneNumber);
            expandReqDto.setOrganizationId(orgId);
            checkNameAndPhoneNum(expandReqDto);
        }

        return null;
    }

    /**
     * 处理员工数据 - 根据操作类型（新增/修改）
     *
     * @param instanceId       实例ID
     * @param data             员工数据
     * @param existingEmployee 现有员工信息（如果存在）
     * @return 处理是否成功
     */
    private boolean processEmployeeData(Long instanceId, EmployeeExcelExtVo data, EmployeeExtRespDto existingEmployee) {
        try {
            //构建人员对象
            EmployeeExpandReqDto employeeExtReqDto = new EmployeeExpandReqDto();
            employeeExtReqDto.setOrganizationId(data.getOrgId());
            employeeExtReqDto.setOrgId(data.getOrgId());
            employeeExtReqDto.setName(data.getUserName());//姓名
            employeeExtReqDto.setPosition(data.getPosition());//职位
            // 设置人员类型，内部为0，外部为1
            employeeExtReqDto.setPersonType("内部".equals(data.getEmployeeType()) ? "Internal" : "External");
            employeeExtReqDto.setPhoneNum(data.getPhoneNumber());//手机号码
            employeeExtReqDto.setEmployeeRoleDtoList(data.getRoleList());//角色
            employeeExtReqDto.setUserId(data.getUserId());//账号
            employeeExtReqDto.setStartTime(new Date());
            employeeExtReqDto.setCustomerInfoList(data.getCustomerInfoList());
            employeeExtReqDto.setCustomerBlackList(data.getCustomerBlackList());
            employeeExtReqDto.setRegionInfoList(data.getAreaList());
            employeeExtReqDto.setOrganizationStatus(YesOrNoEnum.NO.getStatus());
            employeeExtReqDto.setStatus(1);

            // 如果有工号，设置工号
            if (StringUtils.isNotEmpty(data.getEmployeeCode())) {
                employeeExtReqDto.setEmployeeNo(data.getEmployeeCode());
            }

            // 检查是否为修改模式
            boolean isModifyMode = StringUtils.isNotBlank(data.getEmployeeCode()) && existingEmployee != null;

            if (isModifyMode) {
                // 修改操作 - 员工必须已存在
                employeeExtReqDto.setId(existingEmployee.getId());

                // 可能有些字段为空，需要使用原值
                if (StringUtils.isBlank(employeeExtReqDto.getName())) {
                    employeeExtReqDto.setName(existingEmployee.getName());
                }
                if (StringUtils.isBlank(employeeExtReqDto.getPhoneNum())) {
                    employeeExtReqDto.setPhoneNum(existingEmployee.getPhoneNum());
                }
                if (StringUtils.isBlank(employeeExtReqDto.getPosition())) {
                    employeeExtReqDto.setPosition(existingEmployee.getPosition());
                }

                logger.info("修改人员入参:{}", JSON.toJSONString(employeeExtReqDto));
                this.modifyEmployee(instanceId, employeeExtReqDto);
            } else {
                // 新增操作 - 员工必须不存在
                if (existingEmployee != null) {
                    logger.warn("工号为[{}]的员工已存在，但当前为新增模式，跳过处理", data.getEmployeeCode());
                    return false;
                }

                logger.info("新增人员入参:{}", JSON.toJSONString(employeeExtReqDto));
                this.addEmployee(instanceId, employeeExtReqDto);
            }
            logger.info("人员处理成功: {}", data.getAccount());
            return true;
        } catch (Exception e) {
            logger.error("人员处理执行失败:{}", e.getMessage(), e);
            return false;
        }
    }

    /**
     * 校验组织数据
     *
     * @param areasMap 区域映射
     * @return 校验后的数据
     */
    private void validateAreaManagement(EmployeeExcelExtVo vo,
                                        Map<String, List<CustomerAreaRespDto>> areasMap,
                                        List<String> errorList) {
        // 针对灵活的区域管理模式，允许只填写销售区域、大区或区域中的一个
        boolean hasSaleArea = StringUtils.isNotBlank(vo.getSaleArea());
        boolean hasBigArea = StringUtils.isNotBlank(vo.getBigArea());
        boolean hasDistrict = StringUtils.isNotBlank(vo.getDistrict());

        // 三个都没填，报错
        if (!hasSaleArea && !hasBigArea && !hasDistrict) {
            errorList.add("销售区域、大区、区域至少填写一项");
            return;
        }

        // 创建区域列表用于存储所有需要绑定的区域
        List<EmployeeRegionReqDto> areaList = new ArrayList<>();

        // 情况1：只填了销售区域 - 需要绑定销售区域下所有大区和区域
        if (hasSaleArea && !hasBigArea && !hasDistrict) {
            List<CustomerAreaRespDto> saleAreaList = areasMap.get(vo.getSaleArea());
            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            saleAreaList = saleAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            // 获取销售区域的区域编码
            String saleAreaCode = saleAreaList.get(0).getCode();

//            // 1. 添加销售区域本身
//            EmployeeRegionReqDto saleAreaDto = new EmployeeRegionReqDto();
//            saleAreaDto.setRegionCode(saleAreaCode);
//            saleAreaDto.setStartTime(vo.getStartDate());
//            saleAreaDto.setEndTime(vo.getEndDate());
//            areaList.add(saleAreaDto);

            logger.info("绑定销售区域: {}", vo.getSaleArea());

            // 2. 查找并添加该销售区域下的所有大区
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    // 如果父编码与销售区域编码匹配，则为大区
                    if (saleAreaCode.equals(area.getParentCode()) &&
                            Objects.equals(area.getOrgInfoId(), vo.getOrgId())) {

//                        EmployeeRegionReqDto bigAreaDto = new EmployeeRegionReqDto();
//                        bigAreaDto.setRegionCode(area.getCode());
//                        bigAreaDto.setStartTime(vo.getStartDate());
//                        bigAreaDto.setEndTime(vo.getEndDate());
//                        areaList.add(bigAreaDto);

                        logger.info("绑定大区: {}", area.getName());

                        // 3. 查找并添加该大区下的所有区域
                        String bigAreaCode = area.getCode();
                        for (Map.Entry<String, List<CustomerAreaRespDto>> subEntry : areasMap.entrySet()) {
                            for (CustomerAreaRespDto subArea : subEntry.getValue()) {
                                // 如果父编码与大区编码匹配，则为区域
                                if (bigAreaCode.equals(subArea.getParentCode()) &&
                                        Objects.equals(subArea.getOrgInfoId(), vo.getOrgId())) {

                                    EmployeeRegionReqDto districtDto = new EmployeeRegionReqDto();
                                    districtDto.setRegionCode(subArea.getCode());
                                    districtDto.setStartTime(vo.getStartDate());
                                    districtDto.setEndTime(vo.getEndDate());
                                    areaList.add(districtDto);

                                    logger.info("绑定区域: {}", subArea.getName());
                                }
                            }
                        }
                    }
                }
            }
        }
        // 情况2：填了销售区域和大区 - 需要绑定指定大区下所有区域
        else if (hasSaleArea && hasBigArea && !hasDistrict) {
            // 先验证销售区域
            List<CustomerAreaRespDto> saleAreaList = areasMap.get(vo.getSaleArea());
            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            saleAreaList = saleAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(saleAreaList)) {
                errorList.add("找不到销售区域");
                return;
            }

            // 再验证大区
            List<CustomerAreaRespDto> bigAreaList = areasMap.get(vo.getBigArea());
            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            bigAreaList = bigAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            // 检查大区是否属于销售区域
            String saleAreaCode = saleAreaList.get(0).getCode();
            CustomerAreaRespDto bigArea = bigAreaList.get(0);

            if (!bigArea.getParentCode().equals(saleAreaCode)) {
                errorList.add("销售区域下不存在该大区");
                return;
            }

            // 1. 添加销售区域
//            EmployeeRegionReqDto saleAreaDto = new EmployeeRegionReqDto();
//            saleAreaDto.setRegionCode(saleAreaCode);
//            saleAreaDto.setStartTime(vo.getStartDate());
//            saleAreaDto.setEndTime(vo.getEndDate());
//            areaList.add(saleAreaDto);

            logger.info("绑定销售区域: {}", vo.getSaleArea());

            // 2. 添加大区
            String bigAreaCode = bigArea.getCode();
//            EmployeeRegionReqDto bigAreaDto = new EmployeeRegionReqDto();
//            bigAreaDto.setRegionCode(bigAreaCode);
//            bigAreaDto.setStartTime(vo.getStartDate());
//            bigAreaDto.setEndTime(vo.getEndDate());
//            areaList.add(bigAreaDto);

            logger.info("绑定大区: {}", vo.getBigArea());

            // 3. 查找并添加该大区下的所有区域
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    // 如果父编码与大区编码匹配，则为区域
                    if (bigAreaCode.equals(area.getParentCode()) &&
                            Objects.equals(area.getOrgInfoId(), vo.getOrgId())) {

                        EmployeeRegionReqDto districtDto = new EmployeeRegionReqDto();
                        districtDto.setRegionCode(area.getCode());
                        districtDto.setStartTime(vo.getStartDate());
                        districtDto.setEndTime(vo.getEndDate());
                        areaList.add(districtDto);

                        logger.info("绑定区域: {}", area.getName());
                    }
                }
            }
        }
        // 情况3：只填了大区 - 自动查找对应的销售区域，匹配所有下属区域
        else if (!hasSaleArea && hasBigArea && !hasDistrict) {
            List<CustomerAreaRespDto> bigAreaList = areasMap.get(vo.getBigArea());
            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            bigAreaList = bigAreaList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(bigAreaList)) {
                errorList.add("找不到大区");
                return;
            }

            // 大区已找到，获取大区编码
            String bigAreaCode = bigAreaList.get(0).getCode();
            String parentCode = bigAreaList.get(0).getParentCode();

            // 1. 查找并添加对应的销售区域
            CustomerAreaRespDto saleArea = null;
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    if (parentCode.equals(area.getCode())) {
                        saleArea = area;
                        vo.setSaleArea(area.getName());

//                        EmployeeRegionReqDto saleAreaDto = new EmployeeRegionReqDto();
//                        saleAreaDto.setRegionCode(area.getCode());
//                        saleAreaDto.setStartTime(vo.getStartDate());
//                        saleAreaDto.setEndTime(vo.getEndDate());
//                        areaList.add(saleAreaDto);

                        logger.info("绑定销售区域: {}", area.getName());
                        break;
                    }
                }
                if (saleArea != null) {
                    break;
                }
            }

//            // 2. 添加大区本身
//            EmployeeRegionReqDto bigAreaDto = new EmployeeRegionReqDto();
//            bigAreaDto.setRegionCode(bigAreaCode);
//            bigAreaDto.setStartTime(vo.getStartDate());
//            bigAreaDto.setEndTime(vo.getEndDate());
//            areaList.add(bigAreaDto);
//
//            logger.info("绑定大区: {}", vo.getBigArea());

            // 3. 查找并添加该大区下的所有区域
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    // 如果父编码与大区编码匹配，则为区域
                    if (bigAreaCode.equals(area.getParentCode()) &&
                            Objects.equals(area.getOrgInfoId(), vo.getOrgId())) {

                        EmployeeRegionReqDto districtDto = new EmployeeRegionReqDto();
                        districtDto.setRegionCode(area.getCode());
                        districtDto.setStartTime(vo.getStartDate());
                        districtDto.setEndTime(vo.getEndDate());
                        areaList.add(districtDto);

                        logger.info("绑定区域: {}", area.getName());
                    }
                }
            }
        }
        // 情况4：只填了区域 - 仅包含该区域及其对应的大区和销售区域
        else if (!hasSaleArea && !hasBigArea && hasDistrict) {
            List<CustomerAreaRespDto> districtList = areasMap.get(vo.getDistrict());
            if (CollectionUtils.isEmpty(districtList)) {
                errorList.add("找不到区域");
                return;
            }

            districtList = districtList.stream()
                    .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isEmpty(districtList)) {
                errorList.add("找不到区域");
                return;
            }

            // 区域已找到，获取区域编码和父编码（大区编码）
            String districtCode = districtList.get(0).getCode();
            String bigAreaCode = districtList.get(0).getParentCode();

            // 1. 查找并添加对应的大区和销售区域
            CustomerAreaRespDto bigArea = null;
            for (Map.Entry<String, List<CustomerAreaRespDto>> entry : areasMap.entrySet()) {
                for (CustomerAreaRespDto area : entry.getValue()) {
                    if (bigAreaCode.equals(area.getCode())) {
                        bigArea = area;
                        vo.setBigArea(area.getName());

//                        // 添加大区
//                        EmployeeRegionReqDto bigAreaDto = new EmployeeRegionReqDto();
//                        bigAreaDto.setRegionCode(area.getCode());
//                        bigAreaDto.setStartTime(vo.getStartDate());
//                        bigAreaDto.setEndTime(vo.getEndDate());
//                        areaList.add(bigAreaDto);

                        logger.info("绑定大区: {}", area.getName());

                        // 查找销售区域
                        String saleAreaCode = area.getParentCode();
                        for (Map.Entry<String, List<CustomerAreaRespDto>> entry2 : areasMap.entrySet()) {
                            for (CustomerAreaRespDto area2 : entry2.getValue()) {
                                if (saleAreaCode.equals(area2.getCode())) {
                                    vo.setSaleArea(area2.getName());

                                    // 添加销售区域
//                                    EmployeeRegionReqDto saleAreaDto = new EmployeeRegionReqDto();
//                                    saleAreaDto.setRegionCode(area2.getCode());
//                                    saleAreaDto.setStartTime(vo.getStartDate());
//                                    saleAreaDto.setEndTime(vo.getEndDate());
//                                    areaList.add(saleAreaDto);

                                    logger.info("绑定销售区域: {}", area2.getName());
                                    break;
                                }
                            }
                            if (StringUtils.isNotEmpty(vo.getSaleArea())) {
                                break;
                            }
                        }

                        break;
                    }
                }
                if (bigArea != null) {
                    break;
                }
            }

            // 2. 添加区域本身
            EmployeeRegionReqDto districtDto = new EmployeeRegionReqDto();
            districtDto.setRegionCode(districtCode);
            districtDto.setStartTime(vo.getStartDate());
            districtDto.setEndTime(vo.getEndDate());
            areaList.add(districtDto);

            logger.info("绑定区域: {}", vo.getDistrict());
        }
        // 情况5：传统模式 - 销售区域、大区、区域都填了
        else if (hasSaleArea && hasBigArea && hasDistrict) {
            validateCompleteAreaHierarchy(vo, areasMap, errorList);

            // 如果验证通过，添加所有区域
            if (!errorList.contains("找不到销售区域") &&
                    !errorList.contains("找不到大区") &&
                    !errorList.contains("找不到区域") &&
                    !errorList.contains("销售区域下不存在该大区") &&
                    !errorList.contains("大区下不存在该区域")) {

                // 从销售区域查找
                List<CustomerAreaRespDto> saleAreaList = areasMap.get(vo.getSaleArea());
                saleAreaList = saleAreaList.stream()
                        .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                        .collect(Collectors.toList());
                String saleAreaCode = saleAreaList.get(0).getCode();

                // 添加销售区域
//                EmployeeRegionReqDto saleAreaDto = new EmployeeRegionReqDto();
//                saleAreaDto.setRegionCode(saleAreaCode);
//                saleAreaDto.setStartTime(vo.getStartDate());
//                saleAreaDto.setEndTime(vo.getEndDate());
//                areaList.add(saleAreaDto);

                logger.info("绑定销售区域: {}", vo.getSaleArea());

                // 从大区查找
                List<CustomerAreaRespDto> bigAreaList = areasMap.get(vo.getBigArea());
//                bigAreaList = bigAreaList.stream()
//                        .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
//                        .collect(Collectors.toList());
//                String bigAreaCode = bigAreaList.get(0).getCode();

                // 添加大区
//                EmployeeRegionReqDto bigAreaDto = new EmployeeRegionReqDto();
//                bigAreaDto.setRegionCode(bigAreaCode);
//                bigAreaDto.setStartTime(vo.getStartDate());
//                bigAreaDto.setEndTime(vo.getEndDate());
//                areaList.add(bigAreaDto);

                logger.info("绑定大区: {}", vo.getBigArea());

                // 从区域查找
                List<CustomerAreaRespDto> districtList = areasMap.get(vo.getDistrict());
                districtList = districtList.stream()
                        .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                        .collect(Collectors.toList());
                String districtCode = districtList.get(0).getCode();

                // 添加区域
                EmployeeRegionReqDto districtDto = new EmployeeRegionReqDto();
                districtDto.setRegionCode(districtCode);
                districtDto.setStartTime(vo.getStartDate());
                districtDto.setEndTime(vo.getEndDate());
                areaList.add(districtDto);

                logger.info("绑定区域: {}", vo.getDistrict());
            }
        }

        // 如果没有错误，设置区域列表
        if (!errorList.contains("找不到销售区域") &&
                !errorList.contains("找不到大区") &&
                !errorList.contains("找不到区域") &&
                !errorList.contains("销售区域下不存在该大区") &&
                !errorList.contains("大区下不存在该区域")) {

            // 去除可能的重复区域
            List<String> addedCodes = new ArrayList<>();
            List<EmployeeRegionReqDto> uniqueList = new ArrayList<>();

            for (EmployeeRegionReqDto area : areaList) {
                if (!addedCodes.contains(area.getRegionCode())) {
                    addedCodes.add(area.getRegionCode());
                    uniqueList.add(area);
                }
            }

            vo.setAreaList(uniqueList);
            logger.info("最终绑定区域数量: {}", uniqueList.size());
        }
    }

    /**
     * 验证完整的区域层级关系
     */
    private void validateCompleteAreaHierarchy(EmployeeExcelExtVo vo,
                                               Map<String, List<CustomerAreaRespDto>> areasMap,
                                               List<String> errorList) {
        // 验证销售区域
        List<CustomerAreaRespDto> saleAreaList = areasMap.get(vo.getSaleArea());
        if (CollectionUtils.isEmpty(saleAreaList)) {
            errorList.add("找不到销售区域");
            return;
        }
        saleAreaList = saleAreaList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(saleAreaList)) {
            errorList.add("找不到销售区域");
            return;
        }

        // 验证大区
        List<CustomerAreaRespDto> bigAreaList = areasMap.get(vo.getBigArea());
        if (CollectionUtils.isEmpty(bigAreaList)) {
            errorList.add("找不到大区");
            return;
        }
        bigAreaList = bigAreaList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(bigAreaList)) {
            errorList.add("找不到大区");
            return;
        }
        if (!bigAreaList.get(0).getParentCode().equals(saleAreaList.get(0).getCode())) {
            errorList.add("销售区域下不存在该大区");
            return;
        }

        // 验证区域
        List<CustomerAreaRespDto> districtList = areasMap.get(vo.getDistrict());
        if (CollectionUtils.isEmpty(districtList)) {
            errorList.add("找不到区域");
            return;
        }
        districtList = districtList.stream()
                .filter(e -> Objects.equals(e.getOrgInfoId(), vo.getOrgId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(districtList)) {
            errorList.add("找不到区域");
            return;
        }
        if (!districtList.get(0).getParentCode().equals(bigAreaList.get(0).getCode())) {
            errorList.add("大区下不存在该区域");
            return;
        }

        // 设置区域信息
        EmployeeRegionReqDto employeeRegionReqDto = new EmployeeRegionReqDto();
        employeeRegionReqDto.setRegionCode(districtList.get(0).getCode());
        employeeRegionReqDto.setStartTime(vo.getStartDate());
        employeeRegionReqDto.setEndTime(vo.getEndDate());
        vo.setArea(employeeRegionReqDto);
    }

    /**
     * 验证客户管理
     */
    private void validateCustomerManagement(EmployeeExcelExtVo vo,
                                            Map<String, CustomerRespDto> customerMap,
                                            List<String> errorList) {
        if (StringUtils.isBlank(vo.getGovernCustomer())) {
            errorList.add("管理客户不能为空");
            return;
        }

        CustomerRespDto customerRespDto = customerMap.get(vo.getGovernCustomer());
        if (Objects.isNull(customerRespDto)) {
            errorList.add("管理客户（客户编号）错误");
            return;
        }

        EmployeeCustomerReqDto employeeCustomerReqDto = new EmployeeCustomerReqDto();
        employeeCustomerReqDto.setCustomerId(customerRespDto.getId());
        employeeCustomerReqDto.setStartTime(vo.getStartDate());
        vo.setCustomerDto(employeeCustomerReqDto);

        List<EmployeeCustomerReqDto> customerInfoList = new ArrayList<>();
        customerInfoList.add(employeeCustomerReqDto);
        vo.setCustomerInfoList(customerInfoList);
    }

    /**
     * 验证黑名单管理
     */
    private void validateBlacklistManagement(EmployeeExcelExtVo vo,
                                             Map<String, CustomerRespDto> customerMap,
                                             List<String> errorList) {
        if ("外部".equals(vo.getEmployeeType())) {
            errorList.add("外部人员不能选择管理黑名单类型");
            return;
        }

        if (StringUtils.isBlank(vo.getBlacklist())) {
            errorList.add("管理黑名单不能为空");
            return;
        }

        CustomerRespDto customerBlackRespDto = customerMap.get(vo.getBlacklist());
        if (Objects.isNull(customerBlackRespDto)) {
            errorList.add("管理黑名单（客户编号）错误");
            return;
        }

        EmployeeCustomerBlacklistReqDto customerBlacklistReqDto = new EmployeeCustomerBlacklistReqDto();
        customerBlacklistReqDto.setCustomerId(customerBlackRespDto.getId());
        customerBlacklistReqDto.setStartTime(vo.getStartDate());
        vo.setCustomerBlackDto(customerBlacklistReqDto);

        List<EmployeeCustomerBlacklistReqDto> blackList = new ArrayList<>();
        blackList.add(customerBlacklistReqDto);
        vo.setCustomerBlackList(blackList);
    }

    /**
     * 处理多角色情况
     */
    private void processRoles(EmployeeExcelExtVo vo,
                              Map<String, List<RoleExpandRespDto>> roleMap,
                              List<String> errorList) {
        String[] roleNames = vo.getRoleName().split(";");
        List<EmployeeRoleReqDto> roleList = new ArrayList<>();

        for (String roleName : roleNames) {
            roleName = roleName.trim();
            if (StringUtils.isNotEmpty(roleName)) {
                if (roleMap.containsKey(roleName)) {
                    List<RoleExpandRespDto> roleExpands = roleMap.get(roleName);
                    if (CollectionUtils.isNotEmpty(roleExpands)) {
                        EmployeeRoleReqDto roleReqDto = new EmployeeRoleReqDto();
                        roleReqDto.setRoleId(roleExpands.get(0).getId());
                        roleReqDto.setRoleCode(roleExpands.get(0).getCode());
                        roleList.add(roleReqDto);
                    }
                } else {
                    errorList.add("角色：" + roleName + " 不存在");
                }
            }
        }

        if (CollectionUtils.isNotEmpty(roleList)) {
            vo.setRoleList(roleList);
        }
    }

    /**
     * 合并数据，根据姓名和手机号
     *
     * @param list 待合并的列表
     * @return 合并后的列表
     */
    public List<EmployeeExcelExtVo> mergeData(List<EmployeeExcelExtVo> list) {
        if (CollectionUtils.isEmpty(list)) {
            return list;
        }

        Map<String, List<EmployeeExcelExtVo>> namePhoneMap = list.stream()
                .filter(vo -> StringUtils.isNotEmpty(vo.getUserName()) && StringUtils.isNotEmpty(vo.getPhoneNumber()))
                .collect(Collectors.groupingBy(vo -> vo.getUserName() + "_" + vo.getPhoneNumber()));

        List<EmployeeExcelExtVo> mergedList = new ArrayList<>();

        // 处理重复的记录
        for (Map.Entry<String, List<EmployeeExcelExtVo>> entry : namePhoneMap.entrySet()) {
            List<EmployeeExcelExtVo> duplicates = entry.getValue();

            if (duplicates.size() == 1) {
                // 没有重复，直接添加
                mergedList.add(duplicates.get(0));
                continue;
            }

            logger.info("合并相同姓名和手机号的记录: {}, 记录数: {}", entry.getKey(), duplicates.size());

            // 合并记录
            EmployeeExcelExtVo mergedVo = new EmployeeExcelExtVo();

            // 保留第一条记录的基本信息
            EmployeeExcelExtVo firstVo = duplicates.get(0);
            mergedVo.setUserName(firstVo.getUserName());
            mergedVo.setPhoneNumber(firstVo.getPhoneNumber());

            // 使用第一个有效的工号
            String employeeCode = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getEmployeeCode()))
                    .map(EmployeeExcelExtVo::getEmployeeCode)
                    .findFirst()
                    .orElse(null);
            mergedVo.setEmployeeCode(employeeCode);

            // 使用第一个有效的账号
            String account = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getAccount()))
                    .map(EmployeeExcelExtVo::getAccount)
                    .findFirst()
                    .orElse(null);
            mergedVo.setAccount(account);

            // 使用第一个有效的职位
            String position = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getPosition()))
                    .map(EmployeeExcelExtVo::getPosition)
                    .findFirst()
                    .orElse(null);
            mergedVo.setPosition(position);

            // 使用第一个有效的用户ID
            Long userId = duplicates.stream()
                    .filter(vo -> vo.getUserId() != null)
                    .map(EmployeeExcelExtVo::getUserId)
                    .findFirst()
                    .orElse(null);
            mergedVo.setUserId(userId);

            // 使用第一个有效的组织ID
            Long orgId = duplicates.stream()
                    .filter(vo -> vo.getOrgId() != null)
                    .map(EmployeeExcelExtVo::getOrgId)
                    .findFirst()
                    .orElse(null);
            mergedVo.setOrgId(orgId);

            // 优先使用内部人员类型
            String employeeType = duplicates.stream()
                    .filter(vo -> "内部".equals(vo.getEmployeeType()))
                    .map(EmployeeExcelExtVo::getEmployeeType)
                    .findFirst()
                    .orElse(duplicates.get(0).getEmployeeType());
            mergedVo.setEmployeeType(employeeType);

            // 合并管理类型，优先级: 管理区域 > 管理黑名单 > 管理客户
            String governType = duplicates.stream()
                    .filter(vo -> EmpGovernTypeEnum.AREA.getName().equals(vo.getGovernType()))
                    .map(EmployeeExcelExtVo::getGovernType)
                    .findFirst()
                    .orElse(duplicates.stream()
                            .filter(vo -> EmpGovernTypeEnum.BLA.getName().equals(vo.getGovernType()))
                            .map(EmployeeExcelExtVo::getGovernType)
                            .findFirst()
                            .orElse(duplicates.stream()
                                    .filter(vo -> EmpGovernTypeEnum.CUS.getName().equals(vo.getGovernType()))
                                    .map(EmployeeExcelExtVo::getGovernType)
                                    .findFirst()
                                    .orElse(null)));
            mergedVo.setGovernType(governType);

            // 合并角色列表
            Set<EmployeeRoleReqDto> roleSet = new HashSet<>();
            for (EmployeeExcelExtVo vo : duplicates) {
                if (CollectionUtils.isNotEmpty(vo.getRoleList())) {
                    roleSet.addAll(vo.getRoleList());
                }
            }
            mergedVo.setRoleList(new ArrayList<>(roleSet));

            // 日期取最新的
            Date startDate = duplicates.stream()
                    .filter(vo -> vo.getStartDate() != null)
                    .map(EmployeeExcelExtVo::getStartDate)
                    .max(Date::compareTo)
                    .orElse(null);
            mergedVo.setStartDate(startDate);

            Date endDate = duplicates.stream()
                    .filter(vo -> vo.getEndDate() != null)
                    .map(EmployeeExcelExtVo::getEndDate)
                    .max(Date::compareTo)
                    .orElse(null);
            mergedVo.setEndDate(endDate);

            // 合并区域信息
            List<EmployeeRegionReqDto> areaList = new ArrayList<>();
            for (EmployeeExcelExtVo vo : duplicates) {
                if (vo.getArea() != null) {
                    areaList.add(vo.getArea());
                }
            }
            mergedVo.setAreaList(areaList);

            // 合并客户信息
            List<EmployeeCustomerReqDto> customerList = new ArrayList<>();
            for (EmployeeExcelExtVo vo : duplicates) {
                if (vo.getCustomerDto() != null) {
                    customerList.add(vo.getCustomerDto());
                }
            }
            mergedVo.setCustomerInfoList(customerList);

            // 合并黑名单信息
            List<EmployeeCustomerBlacklistReqDto> blackList = new ArrayList<>();
            for (EmployeeExcelExtVo vo : duplicates) {
                if (vo.getCustomerBlackDto() != null) {
                    blackList.add(vo.getCustomerBlackDto());
                }
            }
            mergedVo.setCustomerBlackList(blackList);

            // 合并销售区域、大区、区域信息 - 使用第一个非空值
            String saleArea = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getSaleArea()))
                    .map(EmployeeExcelExtVo::getSaleArea)
                    .findFirst()
                    .orElse(null);
            mergedVo.setSaleArea(saleArea);

            String bigArea = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getBigArea()))
                    .map(EmployeeExcelExtVo::getBigArea)
                    .findFirst()
                    .orElse(null);
            mergedVo.setBigArea(bigArea);

            String district = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getDistrict()))
                    .map(EmployeeExcelExtVo::getDistrict)
                    .findFirst()
                    .orElse(null);
            mergedVo.setDistrict(district);

            // 收集管理客户和黑名单数据
            String governCustomer = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getGovernCustomer()))
                    .map(EmployeeExcelExtVo::getGovernCustomer)
                    .findFirst()
                    .orElse(null);
            mergedVo.setGovernCustomer(governCustomer);

            String blacklist = duplicates.stream()
                    .filter(vo -> StringUtils.isNotEmpty(vo.getBlacklist()))
                    .map(EmployeeExcelExtVo::getBlacklist)
                    .findFirst()
                    .orElse(null);
            mergedVo.setBlacklist(blacklist);

            mergedList.add(mergedVo);
        }

        return mergedList;
    }

}
