package com.dtyunxi.yundt.module.customer.rest;

import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.tcbj.center.openapi.api.dto.request.company.CompanyVerifyReqDto;
import com.dtyunxi.tcbj.center.openapi.api.dto.response.company.CompanyVerifyRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.CustomerAddResultDto;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.response.CustomerExtDetailRespDto;
import com.dtyunxi.yundt.cube.center.customer.api.dto.request.CustomerReqExtDto;
import com.dtyunxi.yundt.module.customer.api.ICustomerExtService;
import com.dtyunxi.yundt.module.customer.api.dto.MerchantDetailRespDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.MerchantListRespDto;
import com.github.pagehelper.PageInfo;
import com.yx.tcbj.center.customer.api.dto.request.CustomerBatchUpReqDto;
import com.yx.tcbj.center.customer.api.dto.response.CustomerExtRespDto;
import com.yx.tcbj.center.customer.api.dto.response.CustomerExtV2DetailRespDto;
import com.yx.tcbj.center.customer.api.dto.response.store.StoreRespDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.List;

/**
 * 客户信息
 * @author xuhuaqiang
 * @version 1.0.0
 * @description
 * @date 2020/9/29 18:06
 */
@Api(tags = {"客商组件：客户信息服务(v2)"})
@RestController
@RequestMapping("/v2/customer")
public class CustomerExtRest {

    private static final Logger logger = LoggerFactory.getLogger(CustomerExtRest.class);

    @Resource
    private ICustomerExtService customerExtService;

    /**
     * 查询客户信息详情
     *
     * @param id
     * @return 统一返回响应实体，data为客户信息响应DTO
     */
    @ApiOperation(value = "查询客户信息详情", notes = "查询客户信息详情")
    @ApiImplicitParam(name = "id", value = "ID", dataType = "Long", paramType = "path", required = true)
    @GetMapping(value = "/{id}")
    RestResponse<CustomerExtV2DetailRespDto> queryById(@PathVariable("id") Long id){
        return new RestResponse<>(customerExtService.queryById(id));
    }

    /**
     * 所属商家信息分页查询
     *
     * @param filter   过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @GetMapping(value = "/merchant", produces = "application/json")
    @ApiOperation(value = "所属商家信息分页查询", notes = "所属商家信息分页查询")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "过滤条件\t\n " +
                    "filter:封装对象JSON {\n" +
                    "    \"merchantCode\": 商家编号,\n" +
                    "    \"merchantName\": 商家名称,\n" +
                    "    \"companyName\":公司名称,\n" +
                    "    \"settleStatus\": 商家状态（3: 启用 4: 禁用）,\n" +
                    "    \"instanceId\": 0,\n" +
                    "    \"tenantId\": 0\n" +
                    "}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    RestResponse<PageInfo<MerchantListRespDto>> queryMerchantByPage(@RequestParam(name = "filter", required = false) String filter,
                                                                    @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>(customerExtService.queryMerchantByPage(filter, pageNum, pageSize));
    }


    @GetMapping(value = "/page", produces = "application/json")
    @ApiOperation(value = "客户列表分页查询", notes = "客户列表分页查询")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "过滤条件\t\n " +
                    "filter:封装对象JSON {\n" +
                    "    \"keyword\": 关键字:名称/编号\n" +
                    "    \"userName\": 登录账号,\n" +
                    "    \"regionIds\":所属区域集合,\n" +
                    "    \"regionCodes\":所属区域编码集合,\n" +
                    "    \"customerTypeIds\":客户类型集合,\n" +
                    "    \"statusId\":客户状态,\n" +
                    "    \"userIds\":管理员ID集合(登录账号集合),\n" +
                    "    \"idList\":id列表,\n" +
                    "    \"userFlag\":账号标识（true:查到符合条件的账号，false:未查到符合条件的账号）,\n" +
                    "    \"srcType\":来源类型（1:经销商小程序端，2:PC管理端，3:业务员小程序端）,\n" +
                    "    \"salesmanId\": 业务员ID,\n" +
                    "    \"customerGroupIds\": 客户分组id集合,\n" +
                    "    \"levelId\": 客户级别id,\n" +
                    "    \"companyName\": 客户名称,\n" +
                    "    \"listFlag\": 查询标识（true:列表查询，false:审核列表）,默认false查询审核列表,\n" +
                    "    \"type\": 客户类型（1：品牌商，2：大B，3：零售商）,\n" +
                    "    \"auditStatus\": 审核状态（WAIT_AUDIT：待审核、AUDIT_REJECT：审核不通过、AUDIT_PASS：审核通过、INVALID：作废、DRAFT：草稿）,\n" +
                    "    \"merchantId\": 所属商家,\n" +
                    "    \"settleStatus\": 入驻状态（1：未入驻、2：申请中、3：已入驻(启用)、4：禁用）,\n" +
                    "    \"settleStatusList\": 入驻状态集合,\n" +
                    "    \"channel\": 渠道类型（1：品牌方运营端，2：大B卖家端）,\n" +
                    "    \"instanceId\": 0,\n" +
                    "    \"tenantId\": 0\n" +
                    "}", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    RestResponse<PageInfo<CustomerExtRespDto>> queryByPage(@RequestParam(name = "filter", required = false) String filter,
                                                           @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>(customerExtService.queryByPage(filter, pageNum, pageSize));
    }

    /**
     * 根据当前登录人查询商家信息(F2B2b模式下的大B卖家端中商家详情专用)
     *
     * @return PageInfo
     */
    @GetMapping(value = "/merchant/detail", produces = "application/json")
    @ApiOperation(value = "根据当前登录人查询商家信息(F2B2b模式下的大B卖家端中商家详情专用)", notes = "根据当前登录人查询商家信息(F2B2b模式下的大B卖家端中商家详情专用)")
    RestResponse<MerchantDetailRespDto> queryMerchantByCurrentUser(){
        return new RestResponse<>(customerExtService.queryMerchantByCurrentUser());
    }

    @GetMapping(value = {"/companyInfo/creditCode"}, produces = {"application/json"})
    @ApiOperation(value = "根据统一社会信用代码查询公司信息", notes = "根据统一社会信用代码查询公司信息")
    @ApiImplicitParams({@ApiImplicitParam(name = "creditCode", value = "creditCode", dataType = "String", paramType = "query")})
    RestResponse<StoreRespDto> queryCompanyByCreditCode(@RequestParam("creditCode") String creditCode,@RequestParam("customerCode") String customerCode) {
        return new RestResponse<>(customerExtService.queryCompanyByCreditCode(creditCode,customerCode));
    }

    @PostMapping(value = {"/companyInfo/verifyCompany"}, produces = {"application/json"})
    @ApiOperation(value = "企业信息三要素校验", notes = "企业信息三要素校验")
    RestResponse<CompanyVerifyRespDto> verifyCompany(@RequestBody CompanyVerifyReqDto verifyCompanyReqDto) {
        return new RestResponse<>(customerExtService.verifyCompany(verifyCompanyReqDto));
    }

    /**
     * 用于增加第三种状态品牌方要对客户进行冻结，不允许经销商对客户进行修改。
     * @date 2022/8/1 10:24
     * @author lshaowu
     * @param id 客户表主键id
     * @param statusId 状态ID值
     * @return
     */
    @GetMapping(
            value = {"/updateCustomerStatus"},
            produces = {"application/json"}
    )
    @ApiOperation(
            value = "修改客户信息-客户状态",
            notes = "修改客户信息-客户状态"
    )
    RestResponse<Void> updateCustomerStatus(@RequestParam("id")Long id,@RequestParam("statusId") Long statusId){
        return new RestResponse<>(this.customerExtService.updateCustomerStatus(id,statusId));
    }

    /**
    * 客户更新提取到module层来，因为涉及到其他层面的更新
    * @date 2022/8/2 15:58
    * @author lshaowu
    * @param
    * @return
    */
    @PutMapping(
            value = {""},
            produces = {"application/json"}
    )
    @ApiOperation(
            value = "修改客户信息",
            notes = "修改客户信息"
    )
    RestResponse<Void> update(@Valid @RequestBody CustomerReqExtDto reqDto){
        this.customerExtService.update(reqDto);
        return RestResponse.VOID;
    }

    @PostMapping(
            value = {""},
            produces = {"application/json"}
    )
    @ApiOperation(
            value = "新增客户信息",
            notes = "新增客户信息"
    )
    RestResponse<CustomerAddResultDto> add(@RequestBody @Valid CustomerReqExtDto reqDto){
        return this.customerExtService.add(reqDto);
    }

    /**
     * 用于对批量租户的更新是否整箱起订而不更新其他，也支持以customerid链表维度进行更新，后续可扩展至其他。目前仅支持状态、整箱起订、客户级别、发货方式
     * @date 2022/12/6 10:24
     * @author lshaowu
     * @param reqDto 更新请求，两者其一不为空
     * @return
     */
    @PostMapping(
            value = {"/batchUpCusIfWholePurchase"},
            produces = {"application/json"}
    )
    @ApiOperation(
            value = "用于对批量租户的更新是否整箱起订而不更新其他",
            notes = "用于对批量租户的更新是否整箱起订而不更新其他"
    )
    RestResponse<List<String>> batchUpCusIfWholePurchase(@RequestBody CustomerBatchUpReqDto reqDto){
        return this.customerExtService.batchUpCusIfWholePurchase(reqDto);
    }
}
