package com.dtyunxi.yundt.module.customer.rest;

import com.alibaba.fastjson.JSON;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.customer.api.customer.dto.request.CustomerSearchReqDto;
import com.dtyunxi.yundt.cube.center.user.api.IUserApi;
import com.dtyunxi.yundt.cube.center.user.api.dto.AddressDto;
import com.dtyunxi.yundt.module.customer.api.ICustomerService;
import com.dtyunxi.yundt.module.customer.api.dto.request.CustomerDetailReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.request.UserAccountReqDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.CustomerDetailAddRespDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.CustomerDetailRespDto;
import com.dtyunxi.yundt.module.customer.api.dto.response.CustomerListRespDto;
import com.github.pagehelper.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * 客户信息
 * @author xuhuaqiang
 * @version 1.0.0
 * @description
 * @date 2020/9/29 18:06
 */
@Api(tags = {"客商组件：客户信息服务"})
@RestController
@RequestMapping("/v1/customer")
public class CustomerRest {

    private static final Logger logger = LoggerFactory.getLogger(CustomerRest.class);

    @Autowired
    private ICustomerService customerService;

    @Autowired
    private IUserApi userApi;

    @ApiOperation(value = "新增客户信息", notes = "新增客户信息")
//    @PostMapping("")
    @Deprecated
    RestResponse<CustomerDetailAddRespDto> add(@Valid @RequestBody CustomerDetailReqDto reqDto) {
        logger.info("创建客户信息入参：{}", JSON.toJSONString(reqDto));
        return new RestResponse<>(customerService.add(reqDto));
    }

    @ApiOperation(value = "保存全部客户相关信息", notes = "保存全部客户相关信息")
//    @PostMapping("/save-all")
    @Deprecated
    RestResponse<CustomerDetailAddRespDto> saveAll(@Valid @RequestBody CustomerDetailReqDto reqDto) {
        logger.info("保存全部客户相关信息入参：{}", JSON.toJSONString(reqDto));
        return new RestResponse<>(customerService.saveAll(reqDto));
    }

    @ApiOperation(value = "修改客户信息", notes = "修改客户信息")
//    @PutMapping("")
    @Deprecated
    RestResponse<Void> update(@Valid @RequestBody CustomerDetailReqDto reqDto) {
        logger.info("修改客户信息入参：{}", JSON.toJSONString(reqDto));
        customerService.update(reqDto);
        return new RestResponse<>();
    }

    /**
     * 创建客户账号信息
     */
    @PostMapping(value = {"/addAccount"})
    @ApiOperation(value = "创建客户账号信息", notes = "创建客户账号信息")
    RestResponse<Long> addCustomerAccount(@Validated @RequestBody UserAccountReqDto userAccountReqDto) {
        customerService.addCustomerAccount(userAccountReqDto);
        return new RestResponse<>();
    }

    /**
     * 编辑客户账号信息
     */
    @PostMapping(value = {"/updateAccount"})
    @ApiOperation(value = "编辑客户账号信息", notes = "编辑客户账号信息")
    RestResponse<Long> updateCustomerAccount(@Validated @RequestBody UserAccountReqDto userAccountReqDto) {
        customerService.updateCustomerAccount(userAccountReqDto);
        return new RestResponse<>();
    }

    /**
     * 客户信息分页查询
     *
     * @param filter   过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @GetMapping(value = "", produces = "application/json")
    @ApiOperation(value = "客户信息分页查询", notes = "客户信息分页查询")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "filter", value = "过滤条件, tenantId, instanceId通过filter传递", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    RestResponse<PageInfo<CustomerListRespDto>> queryByPage(@RequestParam(name = "filter", required = false) String filter,
                                                            @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>(customerService.queryByPage(filter, pageNum, pageSize));
    }

    /**
     * 客户信息分页查询
     *
     * @param searchReqDto   过滤条件
     * @param pageNum  起始页
     * @param pageSize 每一页的记录数
     * @return PageInfo
     */
    @PostMapping(value = "/queryByPage", produces = "application/json")
    @ApiOperation(value = "客户信息分页查询", notes = "客户信息分页查询")
    RestResponse<PageInfo<CustomerListRespDto>> queryByPage(@RequestBody CustomerSearchReqDto searchReqDto,
                                                            @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>(customerService.queryByPage(JSON.toJSONString(searchReqDto), pageNum, pageSize));
    }

    /**
     * 查询客户信息详情
     *
     * @param id
     * @return 统一返回响应实体，data为客户信息响应DTO
     */
    @ApiOperation(value = "查询客户信息详情", notes = "查询客户信息详情")
    @ApiImplicitParam(name = "id", value = "ID", dataType = "Long", paramType = "path", required = true)
    @GetMapping(value = "/{id}", produces = "application/json")
    RestResponse<CustomerDetailRespDto> queryById(@PathVariable("id") Long id){
        return new RestResponse<>(customerService.queryById(id));
    }
    /**
     * 迁移客户到指定区域下
     *
     * @param sourceRegionCode  起始区域
     * @param targetRegionCode 目标区域
     * @return PageInfo
     */
    @PutMapping(value = "/move-customer", produces = "application/json")
    @ApiOperation(value = " 迁移客户到指定区域下", notes = " 迁移客户到指定区域下")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "sourceRegionCode", value = "起始区域", dataType = "String", paramType = "query", required = true),
            @ApiImplicitParam(name = "targetRegionCode", value = "目标区域", dataType = "String", paramType = "query", required = true)})
    RestResponse<Void> moveCustomerToRegion(@RequestParam("sourceRegionCode") String sourceRegionCode,@RequestParam("targetRegionCode") String targetRegionCode){
        customerService.moveCustomerToRegion(sourceRegionCode, targetRegionCode);
        return RestResponse.VOID;
    }

    /**
     * 导出客户
     *
     * @param customerSearchReqDto
     * @return
     */
    @PostMapping(value = "/export", produces = "application/json")
    @ApiOperation(value = "导出客户", notes = "导出客户")
    RestResponse<String> exportCustomer(@RequestBody CustomerSearchReqDto customerSearchReqDto){
        return new RestResponse<>(customerService.exportCustomer(customerSearchReqDto));
    }

    @GetMapping(value = "/exportExt", produces = "application/json")
    @ApiOperation(value = "导出客户Ext", notes = "导出客户Ext")
    RestResponse<String> exportCustomer(@RequestParam(name = "filter", required = false) String filter,
                                        @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>("0",customerService.exportCustomerExt(filter));
    }

    /**
     * 校验客户导出上传oss是否成功
     *
     * @param fileName
     * @return
     */
    @ApiOperation(value = "校验客户导出上传oss是否成功", notes = "校验客户导出上传oss是否成功")
    @GetMapping("/check-export")
    RestResponse<Boolean> checkExportCustomer(@RequestParam(value = "fileName") String fileName){
        return new RestResponse<>(customerService.checkExportCustomer(fileName));
    }

    /**
     * 客户的地址信息分页查询
     *
     * @param keyword   关键字搜索(收货人/联系方式)
     * @param customerId  客户ID
     * @return List
     */
    @GetMapping(value = "/address-page", produces = "application/json")
    @ApiOperation(value = "客户的地址信息分页查询", notes = "客户的地址信息分页查询")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "keyword", value = "关键字搜索(收货人/联系方式)", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "addressType", value = "地址类型:1收货地址，2收票地址", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "customerId", value = "客户ID", dataType = "Long", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageNum", value = "起始页", dataType = "int", paramType = "query", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每一页的记录数", dataType = "int", paramType = "query", required = true)})
    RestResponse<PageInfo<AddressDto>> queryAddressByPage(@RequestParam(name = "keyword", required = false) String keyword,@RequestParam(name = "addressType", required = false) String addressType, @RequestParam("customerId") Long customerId,
                                                          @RequestParam("pageNum") Integer pageNum, @RequestParam("pageSize") Integer pageSize){
        return new RestResponse<>(customerService.queryAddressByPage(keyword,addressType, customerId,pageNum,pageSize));
    }

    /**
     * 客户的默认地址信息查询
     *
     * @param addressType   地址类型
     * @param customerId  客户ID
     * @return List
     */
    @GetMapping(value = "/default-address", produces = "application/json")
    @ApiOperation(value = "客户的默认地址信息查询", notes = "客户的默认地址信息查询")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "addressType", value = "地址类型:1FF收货地址，2收票地址", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "customerId", value = "客户ID", dataType = "Long", paramType = "query", required = true)})
    RestResponse<AddressDto> queryDefaultAddress(@RequestParam(name = "addressType", required = false) String addressType, @RequestParam("customerId") Long customerId){
        return new RestResponse<>(customerService.queryDefaultAddress(addressType, customerId));
    }


}
