package com.dtyunxi.yundt.module.customer.rest;

import cn.hutool.core.util.StrUtil;
import com.dtyunxi.icommerce.utils.RestResponseHelper;
import com.dtyunxi.rest.RestResponse;
import com.dtyunxi.yundt.cube.center.user.api.ICustomerUserApi;
import com.dtyunxi.yundt.cube.center.user.api.dto.AddressDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.BillInfoDto;
import com.dtyunxi.yundt.cube.center.user.api.dto.ContactsInfoDto;
import com.dtyunxi.yundt.cube.center.user.api.query.ICustomerUserQueryApi;
import com.dtyunxi.yundt.module.context.api.IContext;
import com.dtyunxi.yundt.module.customer.api.dto.request.CustomerDetailReqDto;
import com.dtyunxi.yundt.module.customer.api.enums.DefaultSettingEnum;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * @author xuhuaqiang
 * 客商组件，客户管理对应地址、发票信息、联系人信息管理
 *
 * @time 2020/10/9 11:07
 */
@Api(tags = {"客商组件：客户管理对应地址、发票信息、联系人信息管理"})
@RestController
@RequestMapping("/v1/customer-user")
public class CustomerUserRest {

    @Resource
    private ICustomerUserApi customerUserApi;

    @Resource
    private ICustomerUserQueryApi customerUserQueryApi;

    @Resource
    private IContext context;

    /**
     * 创建客户地址
     *
     * @param addressDto    地址
     * @return Long
     */
    @PostMapping(value = {"/address"})
    @ApiOperation(value = "创建客户地址", notes = "创建客户地址")
    RestResponse<Long> addCustomerAddress(@RequestBody AddressDto addressDto){
        addressDto.setInstanceId(context.instanceId());
        addressDto.setTenantId(context.tenantId());
        // 添加时可以添加多种类型
        String addressType = addressDto.getAddressType();
        if (StrUtil.isNotBlank(addressType)){
            String[] addressTypes = addressType.split(StrUtil.COMMA);
            for (String type : addressTypes){
                addressDto.setAddressType(type);
                customerUserApi.addCustomerAddress(addressDto).getData();
            }
        }
        return new RestResponse<>(0L);
    }

    /**
     * 更新客户地址
     *
     * @param addressDto    地址
     * @return void
     */
    @PutMapping(value = {"/address"})
    @ApiOperation(value = "更新客户地址", notes = "更新客户地址")
    RestResponse<Void> updateCustomerAddress(@RequestBody AddressDto addressDto){
        addressDto.setInstanceId(context.instanceId());
        addressDto.setTenantId(context.tenantId());
        customerUserApi.updateCustomerAddress(addressDto);
        return RestResponse.VOID;
    }

    /**
     * 创建发票信息
     *
     * @param billInfoDto    发票信息
     * @return Long
     */
    @PostMapping(value = {"/bill-info"})
    @ApiOperation(value = "创建发票信息", notes = "创建发票信息")
    RestResponse<Long> addCustomerBillInfo(@RequestBody BillInfoDto billInfoDto){
        billInfoDto.setInstanceId(context.instanceId());
        billInfoDto.setTenantId(context.tenantId());
        // 如果状态是默认，则将该组织下的其他发票状态值设置为非默认状态
        if (DefaultSettingEnum.YES.getCode().equals(billInfoDto.getInvoiceStatus())) {
            //查询财务信息
            RestResponse<List<BillInfoDto>> billInfoRestResp =
                    customerUserQueryApi.queryBillInfoListByOrgInfoId(billInfoDto.getOrgInfoId());
            List<BillInfoDto> billInfoDtos = RestResponseHelper.extractData(billInfoRestResp);
            for (BillInfoDto infoDto : billInfoDtos) {
                if (DefaultSettingEnum.YES.getCode().equals(infoDto.getInvoiceStatus())){
                    infoDto.setInvoiceStatus(0);
                    customerUserApi.updateCustomerBillInfo(infoDto);
                }
            }
        }
        Long data = customerUserApi.addCustomerBillInfo(billInfoDto).getData();
        return new RestResponse<>(data);
    }

    /**
     * 批量创建发票信息
     *
     * @param billReqDto    发票信息
     * @return Long
     */
    @PostMapping(value = {"/batch/bill-info"})
    @ApiOperation(value = "批量创建发票信息", notes = "批量创建发票信息")
    RestResponse<Void> addCustomerBillInfoBatch(@RequestBody CustomerDetailReqDto billReqDto){
        if (CollectionUtils.isNotEmpty(billReqDto.getBillInfoList())){
            for (BillInfoDto billInfoDto : billReqDto.getBillInfoList()) {
                billInfoDto.setInstanceId(context.instanceId());
                billInfoDto.setTenantId(context.tenantId());
                RestResponse<Long> addBillInfoRestResp = customerUserApi.addCustomerBillInfo(billInfoDto);
                RestResponseHelper.extractData(addBillInfoRestResp);
            }
        }
        return RestResponse.VOID;
    }

    /**
     * 更新发票地址
     *
     * @param billInfoDto    发票信息
     * @return void
     */
    @PutMapping(value = {"/bill-info"})
    @ApiOperation(value = "更新发票信息", notes = "更新发票信息")
    RestResponse<Void> updateCustomerBillInfo(@RequestBody BillInfoDto billInfoDto){
        billInfoDto.setInstanceId(context.instanceId());
        billInfoDto.setTenantId(context.tenantId());
        // 如果状态是默认，则将该组织下的其他发票状态值设置为非默认状态
        if (DefaultSettingEnum.YES.getCode().equals(billInfoDto.getInvoiceStatus())) {
            //查询财务信息
            RestResponse<List<BillInfoDto>> billInfoRestResp =
                    customerUserQueryApi.queryBillInfoListByOrgInfoId(billInfoDto.getOrgInfoId());
            List<BillInfoDto> billInfoDtos = RestResponseHelper.extractData(billInfoRestResp);
            for (BillInfoDto infoDto : billInfoDtos) {
                if (infoDto.getId().equals(billInfoDto.getId())){
                    customerUserApi.updateCustomerBillInfo(billInfoDto);
                } else {
                    if (DefaultSettingEnum.YES.getCode().equals(infoDto.getInvoiceStatus())){
                        infoDto.setInvoiceStatus(0);
                        customerUserApi.updateCustomerBillInfo(infoDto);
                    }
                }
            }
        } else {
            customerUserApi.updateCustomerBillInfo(billInfoDto);
        }
        return RestResponse.VOID;
    }

    /**
     * 创建客户联系人
     *
     * @param contactsInfoDto    联系人
     * @return Long
     */
    @PostMapping(value = {"/contacts"})
    @ApiOperation(value = "创建客户联系人", notes = "创建客户联系人")
    RestResponse<Long> addCustomerContactsInfo(@RequestBody ContactsInfoDto contactsInfoDto){
        contactsInfoDto.setInstanceId(context.instanceId());
        contactsInfoDto.setTenantId(context.tenantId());
        // 如果状态是默认，则将该组织下的其他联系人状态值设置为非默认状态
        if (DefaultSettingEnum.YES.getCode().equals(contactsInfoDto.getStatus())) {
            //查询联系人信息
            RestResponse<List<ContactsInfoDto>> contactsInfoRestResp =
                    customerUserQueryApi.queryContactsInfoListByOrgInfoId(contactsInfoDto.getOrgInfoId());
            List<ContactsInfoDto> contactsInfoDtos = RestResponseHelper.extractData(contactsInfoRestResp);
            for (ContactsInfoDto infoDto : contactsInfoDtos) {
                if (DefaultSettingEnum.YES.getCode().equals(infoDto.getStatus())){
                    infoDto.setStatus(0);
                    customerUserApi.updateCustomerContactsInfo(infoDto);
                }
            }
        }
        Long data = customerUserApi.addCustomerContactsInfo(contactsInfoDto).getData();
        return new RestResponse<>(data);
    }

    /**
     * 批量创建客户联系人
     *
     * @param contactsReqDto    联系人
     * @return Long
     */
    @PostMapping(value = {"/batch/contacts"})
    @ApiOperation(value = "批量创建客户联系人", notes = "批量创建客户联系人")
    RestResponse<Void> addCustomerContactsInfoBatch(@RequestBody CustomerDetailReqDto contactsReqDto){
        if (CollectionUtils.isNotEmpty(contactsReqDto.getContactsInfoList())){
            for (ContactsInfoDto contactsInfoDto : contactsReqDto.getContactsInfoList()) {
                contactsInfoDto.setInstanceId(context.instanceId());
                contactsInfoDto.setTenantId(context.tenantId());
                RestResponse<Long> addContactsInfoRestResp = customerUserApi.addCustomerContactsInfo(contactsInfoDto);
                RestResponseHelper.extractData(addContactsInfoRestResp);
            }
        }
        return RestResponse.VOID;
    }

    /**
     * 更新客户联系人
     *
     * @param contactsInfoDto    联系人
     * @return Long
     */
    @PutMapping(value = {"/contacts"})
    @ApiOperation(value = "更新客户联系人", notes = "更新客户联系人")
    RestResponse<Void> updateCustomerContactsInfo(@RequestBody ContactsInfoDto contactsInfoDto){
        contactsInfoDto.setInstanceId(context.instanceId());
        contactsInfoDto.setTenantId(context.tenantId());
        // 如果状态是默认，则将该组织下的其他联系人状态值设置为非默认状态
        if (DefaultSettingEnum.YES.getCode().equals(contactsInfoDto.getStatus())) {
            //查询联系人信息
            RestResponse<List<ContactsInfoDto>> contactsInfoRestResp =
                    customerUserQueryApi.queryContactsInfoListByOrgInfoId(contactsInfoDto.getOrgInfoId());
            List<ContactsInfoDto> contactsInfoDtos = RestResponseHelper.extractData(contactsInfoRestResp);
            for (ContactsInfoDto infoDto : contactsInfoDtos) {
                if (infoDto.getId().equals(contactsInfoDto.getId())){
                    customerUserApi.updateCustomerContactsInfo(contactsInfoDto);
                } else {
                    if (DefaultSettingEnum.YES.getCode().equals(infoDto.getStatus())){
                        infoDto.setStatus(0);
                        customerUserApi.updateCustomerContactsInfo(infoDto);
                    }
                }
            }
        } else {
            customerUserApi.updateCustomerContactsInfo(contactsInfoDto);
        }
        return RestResponse.VOID;
    }

    /**
     * 删除发票信息
     *
     * @param id    ID
     * @return void
     */
    @DeleteMapping({"/bill-info/{id}"})
    @ApiOperation(
            value = "删除发票信息",
            notes = "删除发票信息"
    )
    RestResponse<Void> deleteCustomerBillInfo(@PathVariable("id") Long id){
        customerUserApi.deleteCustomerBillInfo(id);
        return RestResponse.VOID;
    }

    /**
     * 删除客户联系人
     *
     * @param id    ID
     * @return void
     */
    @DeleteMapping({"/contacts/{id}"})
    @ApiOperation(
            value = "删除客户联系人",
            notes = "删除客户联系人"
    )
    RestResponse<Void> deleteCustomerContactsInfo(@PathVariable("id") Long id){
        customerUserApi.deleteCustomerContactsInfo(id);
        return RestResponse.VOID;
    }

    /**
     * 删除客户地址
     *
     * @param id    ID
     * @return void
     */
    @DeleteMapping({"/address/{id}"})
    @ApiOperation(
            value = "删除客户地址",
            notes = "删除客户地址"
    )
    RestResponse<Void> deleteCustomerAddress(@PathVariable("id") Long id){
        customerUserApi.deleteCustomerAddress(id);
        return RestResponse.VOID;
    }

    @GetMapping(value = {"/address/by-org-id/{orgInfoId}"})
    @ApiOperation(value = "根据组织信息id,查询客户地址列表",notes = "根据组织信息id,查询客户地址列表")
    @ApiImplicitParam(name = "orgInfoId",value = "组织信息id",dataType = "Long", paramType = "path", required = true)
    RestResponse<List<AddressDto>> queryAddressListByOrgInfoId(@PathVariable("orgInfoId") Long orgInfoId){
        List<AddressDto> addressDtoList = customerUserQueryApi.queryAddressListByOrgInfoId(orgInfoId).getData();
        return new RestResponse<>(addressDtoList);
    }

}
